package foundry.veil.api.client.render.texture;

import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.logging.LogUtils;
import foundry.veil.mixin.pipeline.accessor.PipelineNativeImageAccessor;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import javax.annotation.Nullable;
import net.minecraft.class_1011;
import net.minecraft.class_1060;
import net.minecraft.class_1084;
import net.minecraft.class_156;
import net.minecraft.class_2350;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import static org.lwjgl.opengl.GL12C.*;
import static org.lwjgl.opengl.GL13C.GL_TEXTURE_CUBE_MAP;
import static org.lwjgl.opengl.GL14C.GL_TEXTURE_LOD_BIAS;

/**
 * A cubemap texture type that loads from 2D textures.
 *
 * @since 3.1.0
 */
public class SimpleCubemapTexture extends CubemapTexture implements VeilPreloadedTexture {

    private static final Logger LOGGER = LogUtils.getLogger();

    protected final class_2960 location;
    private CompletableFuture<TextureImage> imageFuture;

    public SimpleCubemapTexture(class_2960 location) {
        this.location = location;
        this.imageFuture = null;
    }

    @Override
    public CompletableFuture<?> preload(class_3300 resourceManager, Executor backgroundExecutor) {
        if (this.imageFuture == null || this.imageFuture.isDone()) {
            this.imageFuture = CompletableFuture.supplyAsync(() -> TextureImage.load(resourceManager, this.location), backgroundExecutor);
        }
        return this.imageFuture;
    }

    @Override
    public void method_4625(@NotNull class_3300 resourceManager) throws IOException {
        TextureImage textureImages = this.getTextureImage(resourceManager);
        try (class_1011 image = textureImages.getImage()) {
            int width = image.method_4307();
            int height = image.method_4323();
            if (height != width * 3 / 4) {
                throw new IOException("Expected cubemap image to be " + width + "x" + (width * 3 / 4) + ". Was " + width + "x" + height);
            }

            class_1084 texturemetadatasection = textureImages.getTextureMetadata();
            boolean blur;
            boolean clamp;
            if (texturemetadatasection != null) {
                blur = texturemetadatasection.method_4696();
                clamp = texturemetadatasection.method_4697();
            } else {
                blur = false;
                clamp = false;
            }

            this.method_4527(blur, clamp);
            if (!RenderSystem.isOnRenderThreadOrInit()) {
                RenderSystem.recordRenderCall(() -> this.loadImages(image));
            } else {
                this.loadImages(image);
            }
        }
    }

    private void loadImages(class_1011 image) {
        try (image) {
            this.method_23207();
            GlStateManager._texParameter(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_LEVEL, 0);
            GlStateManager._texParameter(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_LOD, 0);
            GlStateManager._texParameter(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAX_LOD, 0);
            GlStateManager._texParameter(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_LOD_BIAS, 0.0F);

            int tileSize = image.method_4307() / 4;
            PipelineNativeImageAccessor accessor = (PipelineNativeImageAccessor) (Object) image;
            accessor.invokeCheckAllocated();
            class_1011.class_1012 format = image.method_4318();
            format.method_4340();
            long pixels = accessor.getPixels();

            GlStateManager._pixelStore(GL_UNPACK_ROW_LENGTH, image.method_4307());

            // Top
            GlStateManager._pixelStore(GL_UNPACK_SKIP_PIXELS, tileSize);
            GlStateManager._pixelStore(GL_UNPACK_SKIP_ROWS, 0);
            glTexImage2D(getGlFace(class_2350.field_11036), 0, GL_RGBA8, tileSize, tileSize, 0, format.method_4333(), GL_UNSIGNED_BYTE, pixels);

            // Left
            GlStateManager._pixelStore(GL_UNPACK_SKIP_PIXELS, 0);
            GlStateManager._pixelStore(GL_UNPACK_SKIP_ROWS, tileSize);
            glTexImage2D(getGlFace(class_2350.field_11039), 0, GL_RGBA8, tileSize, tileSize, 0, format.method_4333(), GL_UNSIGNED_BYTE, pixels);

            // Front
            GlStateManager._pixelStore(GL_UNPACK_SKIP_PIXELS, tileSize);
            GlStateManager._pixelStore(GL_UNPACK_SKIP_ROWS, tileSize);
            glTexImage2D(getGlFace(class_2350.field_11035), 0, GL_RGBA8, tileSize, tileSize, 0, format.method_4333(), GL_UNSIGNED_BYTE, pixels);

            // Right
            GlStateManager._pixelStore(GL_UNPACK_SKIP_PIXELS, tileSize * 2);
            GlStateManager._pixelStore(GL_UNPACK_SKIP_ROWS, tileSize);
            glTexImage2D(getGlFace(class_2350.field_11034), 0, GL_RGBA8, tileSize, tileSize, 0, format.method_4333(), GL_UNSIGNED_BYTE, pixels);

            // Back
            GlStateManager._pixelStore(GL_UNPACK_SKIP_PIXELS, tileSize * 3);
            GlStateManager._pixelStore(GL_UNPACK_SKIP_ROWS, tileSize);
            glTexImage2D(getGlFace(class_2350.field_11043), 0, GL_RGBA8, tileSize, tileSize, 0, format.method_4333(), GL_UNSIGNED_BYTE, pixels);

            // Down
            GlStateManager._pixelStore(GL_UNPACK_SKIP_PIXELS, tileSize);
            GlStateManager._pixelStore(GL_UNPACK_SKIP_ROWS, tileSize * 2);
            glTexImage2D(getGlFace(class_2350.field_11033), 0, GL_RGBA8, tileSize, tileSize, 0, format.method_4333(), GL_UNSIGNED_BYTE, pixels);
        }
    }

    @Override
    public void method_18169(@NotNull class_1060 textureManager, @NotNull class_3300 resourceManager, @NotNull class_2960 location, @NotNull Executor gameExecutor) {
        this.preload(resourceManager, class_156.method_18349()).thenRunAsync(() -> textureManager.method_4616(location, this), gameExecutor);
    }

    protected TextureImage getTextureImage(class_3300 resourceManager) {
        if (this.imageFuture != null) {
            TextureImage image = this.imageFuture.join();
            this.imageFuture = null;
            return image;
        } else {
            return TextureImage.load(resourceManager, this.location);
        }
    }

    protected static class TextureImage implements Closeable {
        @Nullable
        private final class_1084 metadata;
        @Nullable
        private final class_1011 image;
        @Nullable
        private final IOException exception;

        public TextureImage(IOException exception) {
            this.exception = exception;
            this.metadata = null;
            this.image = null;
        }

        public TextureImage(@Nullable class_1084 metadata, class_1011 image) {
            this.exception = null;
            this.metadata = metadata;
            this.image = image;
        }

        public static TextureImage load(class_3300 resourceManager, class_2960 location) {
            try {
                class_3298 resource = resourceManager.getResourceOrThrow(location);

                class_1011 nativeimage;
                try (InputStream inputstream = resource.method_14482()) {
                    nativeimage = class_1011.method_4309(inputstream);
                }

                class_1084 texturemetadatasection = null;

                try {
                    texturemetadatasection = resource.method_14481().method_43041(class_1084.field_5344).orElse(null);
                } catch (RuntimeException runtimeexception) {
                    LOGGER.warn("Failed reading metadata of: {}", location, runtimeexception);
                }

                return new TextureImage(texturemetadatasection, nativeimage);
            } catch (IOException ioexception) {
                return new TextureImage(ioexception);
            }
        }

        @Nullable
        public class_1084 getTextureMetadata() {
            return this.metadata;
        }

        public class_1011 getImage() throws IOException {
            if (this.exception != null) {
                throw this.exception;
            } else {
                return this.image;
            }
        }

        @Override
        public void close() {
            if (this.image != null) {
                this.image.close();
            }
        }

        public void throwIfError() throws IOException {
            if (this.exception != null) {
                throw this.exception;
            }
        }
    }
}
