package foundry.veil.api.resource;

import foundry.veil.VeilClient;
import foundry.veil.api.resource.type.McMetaResource;
import net.minecraft.class_2960;
import net.minecraft.class_3264;
import net.minecraft.class_3300;
import net.minecraft.class_7368;
import org.jetbrains.annotations.Nullable;

/**
 * Client-side resource manager implementation that supports server-sided resources on the client side.
 * <br>
 * This is mainly used by the Veil resource API to view and edit resources in mods.
 *
 * @author Ocelot, RyanH
 * @see VeilResource
 * @since 1.0.0
 */
public interface VeilResourceManager {

    /**
     * @return The resource manager instance for the client
     * @since 1.3.0
     */
    static VeilResourceManager get() {
        return VeilClient.resourceManager();
    }

    /**
     * Retrieves the correct resource manager for the specified resource.
     *
     * @param resourceInfo The info of the resource to get
     * @return The sided resource manager for that resource
     */
    default class_3300 resources(VeilResourceInfo resourceInfo) {
        return resourceInfo.packType() == class_3264.field_14190 ? this.serverResources() : this.clientResources();
    }

    /**
     * @return The regular client-sided resources in the <code>assets</code> folder
     */
    class_3300 clientResources();

    /**
     * @return The server-sided resources in the <code>data</code> folder
     */
    class_3300 serverResources();

    /**
     * Searches for a veil resource by namespace and path.
     *
     * @param namespace The namespace to get the resource from
     * @param path      The path of the resource
     * @return The resource found or <code>null</code>
     */
    @Nullable
    VeilResource<?> getVeilResource(String namespace, String path);

    /**
     * Searches for a veil resource by location.
     *
     * @param location The location to get the resource from
     * @return The resource found or <code>null</code>
     */
    default @Nullable VeilResource<?> getVeilResource(class_2960 location) {
        return this.getVeilResource(location.method_12836(), location.method_12832());
    }

    /**
     * Searches for veil resource metadata by location.
     *
     * @param namespace The namespace to get the resource from
     * @param path      The path of the resource
     * @return The metadata for the resource found or <code>null</code>
     */
    default @Nullable class_7368 getResourceMetadata(String namespace, String path) {
        VeilResource<?> resource = this.getVeilResource(namespace, path);
        return resource instanceof McMetaResource mcMetaResource ? mcMetaResource.metadata() : null;
    }

    /**
     * Searches for veil resource metadata by location.
     *
     * @param location The location to get the resource from
     * @return The metadata for the resource found or <code>null</code>
     */
    default @Nullable class_7368 getResourceMetadata(class_2960 location) {
        return this.getResourceMetadata(location.method_12836(), location.method_12832());
    }
}
