package foundry.veil.impl.client.render.shader.compiler;

import foundry.veil.api.client.render.shader.compiler.CompiledShader;
import foundry.veil.api.client.render.shader.compiler.ShaderException;
import foundry.veil.api.client.render.shader.compiler.VeilShaderSource;
import it.unimi.dsi.fastutil.ints.Int2ObjectArrayMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.Objects;
import net.minecraft.class_2960;

/**
 * Attempts to cache the exact same shader sources to reduce the number of compiled shaders.
 *
 * @author Ocelot
 */
@ApiStatus.Internal
public class CachedShaderCompiler extends DirectShaderCompiler {

    private final Int2ObjectMap<CompiledShader> shaders;

    public CachedShaderCompiler(@Nullable ShaderProvider provider) {
        super(provider);
        this.shaders = new Int2ObjectArrayMap<>();
    }

    @Override
    public CompiledShader compile(int type, class_2960 path) throws IOException, ShaderException {
        int hash = Objects.hash(type, path);
        if (this.shaders.containsKey(hash)) {
            return this.shaders.get(hash);
        }
        CompiledShader shader = super.compile(type, path);
        this.shaders.put(hash, shader);
        return shader;
    }

    @Override
    public CompiledShader compile(int type, VeilShaderSource source) throws ShaderException {
        int hash = Objects.hash(type, source.sourceId());
        if (this.shaders.containsKey(hash)) {
            return this.shaders.get(hash);
        }
        CompiledShader shader = super.compile(type, source);
        this.shaders.put(hash, shader);
        return shader;
    }

    @Override
    public void free() {
        super.free();
        this.shaders.clear();
    }
}
