/*
 * Copyright (C) 2023 Cobblemon Contributors
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

package org.dystoria.tweaks.rendering

import com.cobblemon.mod.common.client.render.item.CobblemonBuiltinItemRenderer
import com.cobblemon.mod.common.client.render.item.PokemonItemRenderer
import com.cobblemon.mod.common.client.render.models.blockbench.FloatingState
import com.cobblemon.mod.common.client.render.models.blockbench.repository.RenderContext
import com.cobblemon.mod.common.client.render.models.blockbench.repository.VaryingModelRepository
import com.cobblemon.mod.common.entity.PoseType
import com.cobblemon.mod.common.item.PokemonItem
import com.cobblemon.mod.common.util.math.fromEulerXYZDegrees
import net.minecraft.client.render.*
import net.minecraft.client.render.model.json.ModelTransformationMode
import net.minecraft.client.util.math.MatrixStack
import net.minecraft.component.DataComponentTypes
import net.minecraft.item.ItemStack
import org.joml.Quaternionf
import org.joml.Vector3f
import kotlin.math.absoluteValue
import kotlin.math.roundToInt

class DystorianPokemonItemRenderer : CobblemonBuiltinItemRenderer {
    val context = RenderContext().also {
        it.put(RenderContext.RENDER_STATE, RenderContext.RenderState.PROFILE)
        it.put(RenderContext.DO_QUIRKS, false)
    }

    override fun render(stack: ItemStack, mode: ModelTransformationMode, matrices: MatrixStack, vertexConsumers: VertexConsumerProvider, light: Int, overlay: Int) {
        val pokemonItem = stack.item as? PokemonItem ?: return
        val (species, aspects) = pokemonItem.getSpeciesAndAspects(stack) ?: return
        val state = FloatingState()
        state.currentAspects = aspects
        matrices.push()
        val model = VaryingModelRepository.getPoser(species.resourceIdentifier, state)
        model.context = context
        context.put(RenderContext.RENDER_STATE, RenderContext.RenderState.PROFILE)
        context.put(RenderContext.SPECIES, species.resourceIdentifier)
        context.put(RenderContext.ASPECTS, aspects)
        context.put(RenderContext.POSABLE_STATE, state)
        state.currentModel = model

        val transformations = positions[mode]!!

        if (mode == ModelTransformationMode.GUI) {
            DiffuseLighting.disableGuiDepthLighting()
        }

        var itemFrame3D = mode == ModelTransformationMode.FIXED
        if (itemFrame3D) {
            val normal = matrices.peek().normalMatrix
            if (normal.m20().absoluteValue.roundToInt() == 1 || normal.m22().absoluteValue.roundToInt() == 1) {
                itemFrame3D = false
            }
        }

        if (mode == ModelTransformationMode.HEAD) {
            val scale = model.profileScale * species.baseScale * 1.5F
            matrices.translate(0.5, 0.9, 0.5)
            matrices.scale(scale, scale, scale)

            state.setPoseToFirstSuitable(PoseType.PORTRAIT)
            model.applyAnimations(null, state, 0F, 0F, 0F, 0F, 0F)

            val rotation = Quaternionf().fromEulerXYZDegrees(Vector3f(0F, 0F, 180F))
            matrices.multiply(rotation)
            rotation.conjugate()
        }
        else if (itemFrame3D) {
            val scale = species.baseScale * 1.75F
            matrices.translate(0.5, 0.5, 0.5)
            matrices.scale(scale, scale, scale)

            state.setPoseToFirstSuitable(PoseType.PORTRAIT)
            model.applyAnimations(null, state, 0F, 0F, 0F, 0F, 0F)

            val rotation = Quaternionf().fromEulerXYZDegrees(Vector3f(90F, 0F, 0F))
            matrices.multiply(rotation)
            rotation.conjugate()

        }
        else {
            matrices.scale(transformations.scale.x, transformations.scale.y, transformations.scale.z)
            matrices.translate(transformations.translation.x, transformations.translation.y, transformations.translation.z)
            state.setPoseToFirstSuitable(PoseType.PORTRAIT)
            model.applyAnimations(null, state, 0F, 0F, 0F, 0F, 0F)

            matrices.scale(model.profileScale, model.profileScale, 0.15F)

            val rotation = Quaternionf().fromEulerXYZDegrees(Vector3f(transformations.rotation.x, transformations.rotation.y, transformations.rotation.z))
            matrices.multiply(rotation)
            rotation.conjugate()
        }

        val renderLayer = RenderLayer.getEntityCutout(VaryingModelRepository.getTexture(species.resourceIdentifier, state))
        val isEnchanted = stack.get(DataComponentTypes.ENCHANTMENTS)?.isEmpty == false
        val vertexConsumer: VertexConsumer =
            if (isEnchanted) {
                VertexConsumers.union(
                    vertexConsumers.getBuffer(RenderLayer.getDirectEntityGlint()),
                    vertexConsumers.getBuffer(renderLayer),
                )
            } else {
                vertexConsumers.getBuffer(renderLayer)
            }

        matrices.push()
        val packedLight = if (mode == ModelTransformationMode.GUI) {
            LightmapTextureManager.pack(13, 13)
        }
        else {
            light
        }

        // x = red, y = green, z = blue, w = alpha
        val tint = pokemonItem.tint(stack)
        model.withLayerContext(vertexConsumers, state, VaryingModelRepository.getLayers(species.resourceIdentifier, state)) {
            val tintRed = (tint.x * 255).toInt()
            val tintGreen = (tint.y * 255).toInt()
            val tintBlue = (tint.z * 255).toInt()
            val tintAlpha = (tint.w * 255).toInt()
            val color = (tintAlpha shl 24) or (tintRed shl 16) or (tintGreen shl 8) or tintBlue
            model.render(context, matrices, vertexConsumer, packedLight, OverlayTexture.DEFAULT_UV, color)
        }

        model.setDefault()
        matrices.pop()
        matrices.pop()

        if (mode == ModelTransformationMode.GUI) {
            DiffuseLighting.enableGuiDepthLighting()
        }
    }

    companion object {
        val positions: MutableMap<ModelTransformationMode, PokemonItemRenderer.Transformations> = mutableMapOf()

        init {
            positions[ModelTransformationMode.GUI] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(1.0, -0.2, -0.5),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -3.5F),
                PokemonItemRenderer().Transformation(0F, 35F, 0F)
            )
            positions[ModelTransformationMode.FIXED] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(1.0, -0.5, -5.0),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.1F),
                PokemonItemRenderer().Transformation(0F, 35F - 180F, 0F)
            )
            positions[ModelTransformationMode.FIRST_PERSON_RIGHT_HAND] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(2.75, 0.0, 1.0),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, 35F, 0F)
            )
            positions[ModelTransformationMode.FIRST_PERSON_LEFT_HAND] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(-0.75, 0.0, 1.0),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, -35F, 0F)
            )
            positions[ModelTransformationMode.THIRD_PERSON_RIGHT_HAND] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(1.0, -1.0, -1.25),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, 35F, 0F)
            )
            positions[ModelTransformationMode.THIRD_PERSON_LEFT_HAND] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(1.0, -1.0, -1.25),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, -35F, 0F)
            )
            positions[ModelTransformationMode.GROUND] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(1.0, -0.5, -1.0),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, 35F, 0F)
            )
            positions[ModelTransformationMode.HEAD] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(1.0, -3.5, 3.0),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, 215F, 0F)
            )
            positions[ModelTransformationMode.NONE] = PokemonItemRenderer().Transformations(
                PokemonItemRenderer().Transformation(0.0, 0.0, 0.0),
                PokemonItemRenderer().Transformation(0.5F, -0.5F, -0.5F),
                PokemonItemRenderer().Transformation(0F, 0F, 0F)
            )
        }
    }
}