/*
 * Decompiled with CFR 0.152.
 */
package com.restoreit.config;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.logging.Logger;
import org.bukkit.configuration.file.FileConfiguration;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;

public final class ConfigManager {
    private final JavaPlugin plugin;
    private final Logger logger;
    private final File configFile;
    private FileConfiguration config;
    private static final String SECTION_GENERAL = "general";
    private static final String SECTION_PERFORMANCE = "performance";
    private static final String SECTION_STORAGE = "storage";
    private static final String SECTION_MESSAGES = "messages";
    private static final String KEY_DEBUG_MODE = "general.debug-mode";
    private static final String KEY_AUTO_SAVE_INTERVAL = "general.auto-save-interval";
    private static final String KEY_DEFAULT_LANGUAGE = "general.default-language";
    private static final String KEY_MAX_ARENA_SIZE = "performance.max-arena-size";
    private static final String KEY_MAX_SNAPSHOTS_PER_ARENA = "performance.max-snapshots-per-arena";
    private static final String KEY_ASYNC_OPERATIONS = "performance.async-operations";
    private static final String KEY_BATCH_SIZE = "performance.batch-size";
    private static final String KEY_MAX_CONCURRENT_RESTORES = "performance.max-concurrent-restores";
    private static final String KEY_STORAGE_TYPE = "storage.type";
    private static final String KEY_COMPRESSION_ENABLED = "storage.compression-enabled";
    private static final String KEY_COMPRESSION_LEVEL = "storage.compression-level";
    private static final String KEY_CLEANUP_INTERVAL = "storage.cleanup-interval";
    private static final String KEY_PREFIX = "messages.prefix";
    private static final String KEY_DATE_FORMAT = "messages.date-format";

    public ConfigManager(@NotNull JavaPlugin plugin) {
        this.plugin = plugin;
        this.logger = plugin.getLogger();
        this.configFile = new File(plugin.getDataFolder(), "config.yml");
    }

    public void loadConfig() {
        this.logger.info("Loading configuration...");
        try {
            if (!this.plugin.getDataFolder().exists()) {
                this.plugin.getDataFolder().mkdirs();
            }
            if (!this.configFile.exists()) {
                this.createDefaultConfig();
            }
            this.config = YamlConfiguration.loadConfiguration((File)this.configFile);
            this.validateConfiguration();
            this.saveConfig();
            this.logger.info("Configuration loaded successfully");
        }
        catch (Exception e) {
            this.logger.severe("Failed to load configuration: " + e.getMessage());
            throw new RuntimeException("Configuration loading failed", e);
        }
    }

    private void createDefaultConfig() {
        this.logger.info("Creating default configuration file...");
        try (InputStream inputStream = this.plugin.getResource("config.yml");){
            if (inputStream != null) {
                Files.copy(inputStream, this.configFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                this.logger.info("Default configuration created");
            } else {
                this.createMinimalConfig();
            }
        }
        catch (IOException e) {
            this.logger.warning("Failed to copy default config, creating minimal config: " + e.getMessage());
            this.createMinimalConfig();
        }
    }

    private void createMinimalConfig() {
        this.config = new YamlConfiguration();
        this.setDefaults();
        this.saveConfig();
    }

    private void setDefaults() {
        this.config.addDefault(KEY_DEBUG_MODE, (Object)false);
        this.config.addDefault(KEY_AUTO_SAVE_INTERVAL, (Object)300);
        this.config.addDefault(KEY_DEFAULT_LANGUAGE, (Object)"en");
        this.config.addDefault(KEY_MAX_ARENA_SIZE, (Object)1000000);
        this.config.addDefault(KEY_MAX_SNAPSHOTS_PER_ARENA, (Object)10);
        this.config.addDefault(KEY_ASYNC_OPERATIONS, (Object)true);
        this.config.addDefault(KEY_BATCH_SIZE, (Object)1000);
        this.config.addDefault(KEY_MAX_CONCURRENT_RESTORES, (Object)5);
        this.config.addDefault(KEY_STORAGE_TYPE, (Object)"file");
        this.config.addDefault(KEY_COMPRESSION_ENABLED, (Object)true);
        this.config.addDefault(KEY_COMPRESSION_LEVEL, (Object)6);
        this.config.addDefault(KEY_CLEANUP_INTERVAL, (Object)3600);
        this.config.addDefault(KEY_PREFIX, (Object)"\u00a78[\u00a7bRestoreIt\u00a78]\u00a7r ");
        this.config.addDefault(KEY_DATE_FORMAT, (Object)"yyyy-MM-dd HH:mm:ss");
        this.config.options().copyDefaults(true);
    }

    private void validateConfiguration() {
        this.logger.info("Validating configuration...");
        this.validateIntegerRange(KEY_AUTO_SAVE_INTERVAL, 30, 3600, "Auto-save interval must be between 30 and 3600 seconds");
        this.validateIntegerRange(KEY_MAX_ARENA_SIZE, 1000, 10000000, "Max arena size must be between 1,000 and 10,000,000 blocks");
        this.validateIntegerRange(KEY_MAX_SNAPSHOTS_PER_ARENA, 1, 100, "Max snapshots per arena must be between 1 and 100");
        this.validateIntegerRange(KEY_BATCH_SIZE, 100, 10000, "Batch size must be between 100 and 10,000");
        this.validateIntegerRange(KEY_MAX_CONCURRENT_RESTORES, 1, 20, "Max concurrent restores must be between 1 and 20");
        this.validateCompressionLevel();
        this.validateIntegerRange(KEY_CLEANUP_INTERVAL, 300, 86400, "Cleanup interval must be between 300 and 86400 seconds");
        String storageType = this.config.getString(KEY_STORAGE_TYPE, "file");
        if (!List.of("file", "database").contains(storageType.toLowerCase())) {
            this.logger.warning("Invalid storage type '" + storageType + "', defaulting to 'file'");
            this.config.set(KEY_STORAGE_TYPE, (Object)"file");
        }
        this.logger.info("Configuration validation completed");
    }

    private void validateIntegerRange(String key, int min, int max, String errorMessage) {
        int value = this.config.getInt(key);
        if (value < min || value > max) {
            this.logger.warning(errorMessage + " (was: " + value + ")");
            int defaultValue = (min + max) / 2;
            this.config.set(key, (Object)defaultValue);
            this.logger.info("Set " + key + " to default value: " + defaultValue);
        }
    }

    private void validateCompressionLevel() {
        int level = this.config.getInt(KEY_COMPRESSION_LEVEL);
        if (level < 0 || level > 9) {
            this.logger.warning("Compression level must be between 0 and 9 (was: " + level + ")");
            this.config.set(KEY_COMPRESSION_LEVEL, (Object)6);
            this.logger.info("Set compression level to default value: 6");
        }
    }

    public void saveConfig() {
        try {
            this.config.save(this.configFile);
            this.logger.fine("Configuration saved");
        }
        catch (IOException e) {
            this.logger.severe("Failed to save configuration: " + e.getMessage());
        }
    }

    public void reloadConfig() {
        this.logger.info("Reloading configuration...");
        this.loadConfig();
    }

    public boolean isDebugMode() {
        return this.config.getBoolean(KEY_DEBUG_MODE, false);
    }

    public int getAutoSaveInterval() {
        return this.config.getInt(KEY_AUTO_SAVE_INTERVAL, 300);
    }

    public String getDefaultLanguage() {
        return this.config.getString(KEY_DEFAULT_LANGUAGE, "en");
    }

    public int getMaxArenaSize() {
        return this.config.getInt(KEY_MAX_ARENA_SIZE, 1000000);
    }

    public int getMaxSnapshotsPerArena() {
        return this.config.getInt(KEY_MAX_SNAPSHOTS_PER_ARENA, 10);
    }

    public boolean isAsyncOperationsEnabled() {
        return this.config.getBoolean(KEY_ASYNC_OPERATIONS, true);
    }

    public int getBatchSize() {
        return this.config.getInt(KEY_BATCH_SIZE, 1000);
    }

    public int getMaxConcurrentRestores() {
        return this.config.getInt(KEY_MAX_CONCURRENT_RESTORES, 5);
    }

    public String getStorageType() {
        return this.config.getString(KEY_STORAGE_TYPE, "file");
    }

    public boolean isCompressionEnabled() {
        return this.config.getBoolean(KEY_COMPRESSION_ENABLED, true);
    }

    public int getCompressionLevel() {
        return this.config.getInt(KEY_COMPRESSION_LEVEL, 6);
    }

    public int getCleanupInterval() {
        return this.config.getInt(KEY_CLEANUP_INTERVAL, 3600);
    }

    public String getMessagePrefix() {
        return this.config.getString(KEY_PREFIX, "\u00a78[\u00a7bRestoreIt\u00a78]\u00a7r ");
    }

    public String getDateFormat() {
        return this.config.getString(KEY_DATE_FORMAT, "yyyy-MM-dd HH:mm:ss");
    }

    @NotNull
    public FileConfiguration getConfig() {
        return this.config;
    }
}

