/*
 * Decompiled with CFR 0.152.
 */
package com.restoreit.core;

import com.restoreit.config.ConfigManager;
import com.restoreit.integration.WorldEditIntegration;
import com.restoreit.model.Arena;
import com.restoreit.model.ArenaRegion;
import com.restoreit.storage.ArenaStorage;
import com.sk89q.worldedit.math.BlockVector3;
import com.sk89q.worldedit.regions.Region;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;
import org.bukkit.entity.Player;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class ArenaManager {
    private final WorldEditIntegration worldEditIntegration;
    private final ConfigManager configManager;
    private final ArenaStorage arenaStorage;
    private final Logger logger;
    private final Map<String, Arena> arenas;
    private final Map<String, Set<String>> arenasByWorld;

    public ArenaManager(@NotNull JavaPlugin plugin, @NotNull WorldEditIntegration worldEditIntegration, @NotNull ConfigManager configManager) {
        this.worldEditIntegration = worldEditIntegration;
        this.configManager = configManager;
        this.arenaStorage = new ArenaStorage(plugin);
        this.logger = plugin.getLogger();
        this.arenas = new ConcurrentHashMap<String, Arena>();
        this.arenasByWorld = new ConcurrentHashMap<String, Set<String>>();
        this.loadArenasFromStorage();
    }

    @Nullable
    public Arena createArena(@NotNull String name, @NotNull Player player) {
        try {
            if (name.trim().isEmpty()) {
                this.logger.warning("Arena name cannot be empty");
                return null;
            }
            if (this.arenas.containsKey(name.toLowerCase())) {
                return null;
            }
            Region selection = this.worldEditIntegration.getPlayerSelection(player);
            if (selection == null) {
                this.logger.info("Player " + player.getName() + " has no WorldEdit selection");
                return null;
            }
            if (!this.isValidSelection(selection)) {
                return null;
            }
            ArenaRegion arenaRegion = this.convertToArenaRegion(selection, player.getWorld().getName());
            Arena arena = Arena.builder(name).world(player.getWorld()).region(arenaRegion).createdBy(player.getName()).createdAt(LocalDateTime.now()).build();
            this.storeArena(arena);
            this.arenaStorage.saveArena(arena);
            return arena;
        }
        catch (Exception e) {
            this.logger.severe("Failed to create arena '" + name + "': " + e.getMessage());
            e.printStackTrace();
            return null;
        }
    }

    public boolean deleteArena(@NotNull String name) {
        try {
            String arenaKey = name.toLowerCase();
            Arena arena = this.arenas.get(arenaKey);
            if (arena == null) {
                this.logger.info("Arena '" + name + "' not found for deletion");
                return false;
            }
            this.removeArena(arena);
            this.arenaStorage.deleteArena(name);
            this.logger.info("Arena '" + name + "' deleted successfully");
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Failed to delete arena '" + name + "': " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    @Nullable
    public Arena getArena(@NotNull String name) {
        return this.arenas.get(name.toLowerCase());
    }

    public boolean arenaExists(@NotNull String name) {
        return this.arenas.containsKey(name.toLowerCase());
    }

    @NotNull
    public Collection<Arena> getAllArenas() {
        return Collections.unmodifiableCollection(this.arenas.values());
    }

    @NotNull
    public Collection<Arena> getArenasInWorld(@NotNull String worldName) {
        Set arenaNames = this.arenasByWorld.getOrDefault(worldName, Collections.emptySet());
        return arenaNames.stream().map(this.arenas::get).filter(Objects::nonNull).toList();
    }

    public int getArenaCount() {
        return this.arenas.size();
    }

    @NotNull
    public List<String> getArenaNames(@Nullable String pattern) {
        if (pattern == null || pattern.trim().isEmpty()) {
            return new ArrayList<String>(this.arenas.keySet());
        }
        String lowerPattern = pattern.toLowerCase();
        return this.arenas.keySet().stream().filter(name -> name.contains(lowerPattern)).sorted().toList();
    }

    private boolean isValidSelection(@NotNull Region selection) {
        if (!this.worldEditIntegration.isAvailable()) {
            this.logger.warning("WorldEdit integration not available for selection validation");
            return false;
        }
        long maxSize = this.configManager.getMaxArenaSize();
        if (!this.worldEditIntegration.isValidArenaRegion(selection, maxSize)) {
            this.logger.info("Selection failed arena region validation");
            return false;
        }
        long volume = selection.getVolume();
        if (volume < 1L) {
            this.logger.info("Selection is too small: " + volume + " blocks");
            return false;
        }
        if (volume > maxSize) {
            this.logger.info("Selection is too large: " + volume + " blocks (max: " + maxSize + ")");
            return false;
        }
        return true;
    }

    @NotNull
    private ArenaRegion convertToArenaRegion(@NotNull Region region, @NotNull String worldName) {
        BlockVector3 min = region.getMinimumPoint();
        BlockVector3 max = region.getMaximumPoint();
        return new ArenaRegion(worldName, min.getX(), min.getY(), min.getZ(), max.getX(), max.getY(), max.getZ());
    }

    private void storeArena(@NotNull Arena arena) {
        String arenaKey = arena.getName().toLowerCase();
        this.arenas.put(arenaKey, arena);
        this.arenasByWorld.computeIfAbsent(arena.getWorldName(), k -> ConcurrentHashMap.newKeySet()).add(arenaKey);
    }

    private void removeArena(@NotNull Arena arena) {
        String arenaKey = arena.getName().toLowerCase();
        this.arenas.remove(arenaKey);
        Set<String> worldArenas = this.arenasByWorld.get(arena.getWorldName());
        if (worldArenas != null) {
            worldArenas.remove(arenaKey);
            if (worldArenas.isEmpty()) {
                this.arenasByWorld.remove(arena.getWorldName());
            }
        }
    }

    public void updateArenaSnapshots(@NotNull String arenaName, @NotNull Set<String> snapshots) {
        String arenaKey = arenaName.toLowerCase();
        Arena arena = this.arenas.get(arenaKey);
        if (arena != null) {
            Arena updatedArena = arena.withSnapshots(snapshots);
            this.arenas.put(arenaKey, updatedArena);
        }
    }

    @NotNull
    public String getDebugInfo() {
        StringBuilder info = new StringBuilder();
        info.append("Arena Manager Debug Info:\n");
        info.append("  Total Arenas: ").append(this.arenas.size()).append("\n");
        info.append("  Worlds with Arenas: ").append(this.arenasByWorld.size()).append("\n");
        for (Map.Entry<String, Set<String>> entry : this.arenasByWorld.entrySet()) {
            info.append("    ").append(entry.getKey()).append(": ").append(entry.getValue().size()).append(" arenas\n");
        }
        return info.toString();
    }

    public void clearAll() {
        this.arenas.clear();
        this.arenasByWorld.clear();
        this.logger.info("All arenas cleared");
    }

    private void loadArenasFromStorage() {
        try {
            Map<String, Arena> loadedArenas = this.arenaStorage.loadArenas();
            for (Arena arena : loadedArenas.values()) {
                this.storeArena(arena);
            }
            this.logger.info("Loaded " + loadedArenas.size() + " arenas from storage");
        }
        catch (Exception e) {
            this.logger.severe("Failed to load arenas from storage: " + e.getMessage());
            e.printStackTrace();
        }
    }

    public boolean saveAllArenas() {
        try {
            return this.arenaStorage.saveArenas(this.arenas.values());
        }
        catch (Exception e) {
            this.logger.severe("Failed to save all arenas: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    @NotNull
    public List<String> findOverlappingArenas(@NotNull ArenaRegion newRegion, @Nullable String excludeArena) {
        ArrayList<String> overlapping = new ArrayList<String>();
        Collection<Arena> worldArenas = this.getArenasInWorld(newRegion.getWorldName());
        for (Arena arena : worldArenas) {
            if (excludeArena != null && arena.getName().equalsIgnoreCase(excludeArena) || !arena.getRegion().overlaps(newRegion)) continue;
            overlapping.add(arena.getName());
        }
        return overlapping;
    }

    @NotNull
    public ValidationResult validateArenaCreation(@NotNull String name, @NotNull ArenaRegion region) {
        if (name.trim().isEmpty()) {
            return ValidationResult.error("Arena name cannot be empty");
        }
        if (this.arenaExists(name)) {
            return ValidationResult.error("Arena '" + name + "' already exists");
        }
        long volume = region.getVolume();
        long maxSize = this.configManager.getMaxArenaSize();
        if (volume < 1L) {
            return ValidationResult.error("Arena region is too small: " + volume + " blocks");
        }
        if (volume > maxSize) {
            return ValidationResult.error("Arena region is too large: " + volume + " blocks (max: " + maxSize + ")");
        }
        List<String> overlapping = this.findOverlappingArenas(region, null);
        if (!overlapping.isEmpty()) {
            return ValidationResult.error("Arena would overlap with existing arenas: " + String.join((CharSequence)", ", overlapping));
        }
        return ValidationResult.success();
    }

    @NotNull
    public List<Arena> getArenasNear(@NotNull String worldName, int x, int y, int z, int radius) {
        return this.getArenasInWorld(worldName).stream().filter(arena -> arena.getRegion().isWithinDistance(x, y, z, radius)).sorted((a, b) -> Double.compare(a.getRegion().getDistanceFrom(x, y, z), b.getRegion().getDistanceFrom(x, y, z))).toList();
    }

    @NotNull
    public ArenaStatistics getStatistics() {
        long totalBlocks = this.arenas.values().stream().mapToLong(Arena::getBlockCount).sum();
        int totalSnapshots = this.arenas.values().stream().mapToInt(Arena::getSnapshotCount).sum();
        HashMap<String, Integer> arenasByWorldCount = new HashMap<String, Integer>();
        for (Map.Entry<String, Set<String>> entry : this.arenasByWorld.entrySet()) {
            arenasByWorldCount.put(entry.getKey(), entry.getValue().size());
        }
        return new ArenaStatistics(this.arenas.size(), totalBlocks, totalSnapshots, arenasByWorldCount);
    }

    public static final class ValidationResult {
        private final boolean success;
        private final String errorMessage;

        private ValidationResult(boolean success, @Nullable String errorMessage) {
            this.success = success;
            this.errorMessage = errorMessage;
        }

        public static ValidationResult success() {
            return new ValidationResult(true, null);
        }

        public static ValidationResult error(@NotNull String message) {
            return new ValidationResult(false, message);
        }

        public boolean isSuccess() {
            return this.success;
        }

        @Nullable
        public String getErrorMessage() {
            return this.errorMessage;
        }
    }

    public static final class ArenaStatistics {
        private final int totalArenas;
        private final long totalBlocks;
        private final int totalSnapshots;
        private final Map<String, Integer> arenasByWorld;

        public ArenaStatistics(int totalArenas, long totalBlocks, int totalSnapshots, @NotNull Map<String, Integer> arenasByWorld) {
            this.totalArenas = totalArenas;
            this.totalBlocks = totalBlocks;
            this.totalSnapshots = totalSnapshots;
            this.arenasByWorld = Collections.unmodifiableMap(new HashMap<String, Integer>(arenasByWorld));
        }

        public int getTotalArenas() {
            return this.totalArenas;
        }

        public long getTotalBlocks() {
            return this.totalBlocks;
        }

        public int getTotalSnapshots() {
            return this.totalSnapshots;
        }

        @NotNull
        public Map<String, Integer> getArenasByWorld() {
            return this.arenasByWorld;
        }
    }
}

