/*
 * Decompiled with CFR 0.152.
 */
package com.restoreit.core;

import com.restoreit.config.ConfigManager;
import com.restoreit.integration.WorldEditIntegration;
import com.restoreit.model.Arena;
import com.restoreit.model.Snapshot;
import com.restoreit.storage.SnapshotStorage;
import com.restoreit.util.MessageManager;
import com.sk89q.worldedit.EditSession;
import com.sk89q.worldedit.WorldEdit;
import com.sk89q.worldedit.bukkit.BukkitAdapter;
import com.sk89q.worldedit.extent.Extent;
import com.sk89q.worldedit.extent.clipboard.Clipboard;
import com.sk89q.worldedit.extent.clipboard.io.BuiltInClipboardFormat;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardFormat;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardFormats;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardReader;
import com.sk89q.worldedit.function.operation.Operation;
import com.sk89q.worldedit.function.operation.Operations;
import com.sk89q.worldedit.math.BlockVector3;
import com.sk89q.worldedit.session.ClipboardHolder;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.bukkit.Bukkit;
import org.bukkit.World;
import org.bukkit.command.CommandSender;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class RestoreManager {
    private final JavaPlugin plugin;
    private final WorldEditIntegration worldEditIntegration;
    private final ConfigManager configManager;
    private final MessageManager messageManager;
    private final SnapshotStorage snapshotStorage;
    private final Logger logger;
    private final Map<UUID, RestoreOperation> activeRestorations;

    public RestoreManager(@NotNull JavaPlugin plugin, @NotNull WorldEditIntegration worldEditIntegration, @NotNull ConfigManager configManager, @NotNull MessageManager messageManager, @NotNull SnapshotStorage snapshotStorage) {
        this.plugin = plugin;
        this.worldEditIntegration = worldEditIntegration;
        this.configManager = configManager;
        this.messageManager = messageManager;
        this.snapshotStorage = snapshotStorage;
        this.logger = plugin.getLogger();
        this.activeRestorations = new ConcurrentHashMap<UUID, RestoreOperation>();
    }

    @NotNull
    public CompletableFuture<RestoreResult> restoreArena(@NotNull Arena arena, @NotNull Snapshot snapshot, @NotNull CommandSender initiator) {
        UUID operationId = UUID.randomUUID();
        if (this.isArenaBeingRestored(arena)) {
            CompletableFuture<RestoreResult> future = new CompletableFuture<RestoreResult>();
            future.complete(RestoreResult.failure("Arena '" + arena.getName() + "' is already being restored"));
            return future;
        }
        RestoreOperation operation = new RestoreOperation(operationId, arena, snapshot, initiator, LocalDateTime.now());
        this.activeRestorations.put(operationId, operation);
        return CompletableFuture.supplyAsync(() -> {
            try {
                this.messageManager.sendInfo(initiator, "\u00a76Starting restoration of arena '\u00a7e" + arena.getName() + "\u00a76' to snapshot '\u00a7e" + snapshot.getName() + "\u00a76'...");
                operation.setStatus(RestoreStatus.LOADING_SNAPSHOT);
                Optional<Clipboard> clipboardOpt = this.loadSnapshotClipboard(snapshot);
                if (clipboardOpt.isEmpty()) {
                    RestoreResult restoreResult = RestoreResult.failure("Failed to load snapshot data");
                    return restoreResult;
                }
                Clipboard clipboard = clipboardOpt.get();
                operation.setStatus(RestoreStatus.RESTORING_BLOCKS);
                CompletableFuture mainThreadResult = new CompletableFuture();
                Bukkit.getScheduler().runTask((Plugin)this.plugin, () -> {
                    try {
                        this.performMainThreadRestoration(arena, clipboard, operation, mainThreadResult);
                    }
                    catch (Exception e) {
                        this.logger.log(Level.SEVERE, "Error during main thread restoration", e);
                        mainThreadResult.complete(RestoreResult.failure("Restoration failed: " + e.getMessage()));
                    }
                });
                RestoreResult result = (RestoreResult)mainThreadResult.get(30L, TimeUnit.SECONDS);
                if (result.isSuccess()) {
                    operation.setStatus(RestoreStatus.COMPLETED);
                    this.messageManager.sendSuccess(initiator, "\u00a7aArena '\u00a7e" + arena.getName() + "\u00a7a' successfully restored to snapshot '\u00a7e" + snapshot.getName() + "\u00a7a'!");
                    String duration = String.format("%.2f", operation.getDurationSeconds());
                    this.messageManager.sendInfo(initiator, "\u00a77Restoration completed in \u00a7f" + duration + "s");
                } else {
                    operation.setStatus(RestoreStatus.FAILED);
                    this.messageManager.sendError(initiator, "\u00a7cRestoration failed: " + result.getErrorMessage());
                }
                RestoreResult restoreResult = result;
                return restoreResult;
            }
            catch (Exception e) {
                this.logger.log(Level.SEVERE, "Error during arena restoration", e);
                operation.setStatus(RestoreStatus.FAILED);
                RestoreResult restoreResult = RestoreResult.failure("Restoration failed: " + e.getMessage());
                return restoreResult;
            }
            finally {
                this.activeRestorations.remove(operationId);
            }
        });
    }

    private void performMainThreadRestoration(@NotNull Arena arena, @NotNull Clipboard clipboard, @NotNull RestoreOperation operation, @NotNull CompletableFuture<RestoreResult> result) {
        try {
            if (!this.worldEditIntegration.isAvailable()) {
                result.complete(RestoreResult.failure("WorldEdit is not available"));
                return;
            }
            World bukkitWorld = Bukkit.getWorld((String)arena.getWorldName());
            if (bukkitWorld == null) {
                result.complete(RestoreResult.failure("World '" + arena.getWorldName() + "' not found"));
                return;
            }
            EditSession editSession = WorldEdit.getInstance().newEditSession(BukkitAdapter.adapt((World)bukkitWorld));
            int blockChangeLimit = this.configManager.getBatchSize() * 1000;
            editSession.setBlockChangeLimit(blockChangeLimit);
            BlockVector3 pastePosition = BlockVector3.at((int)arena.getRegion().getMinX(), (int)arena.getRegion().getMinY(), (int)arena.getRegion().getMinZ());
            ClipboardHolder clipboardHolder = new ClipboardHolder(clipboard);
            Operation pasteOperation = clipboardHolder.createPaste((Extent)editSession).to(pastePosition).ignoreAirBlocks(false).build();
            Operations.complete((Operation)pasteOperation);
            operation.setBlocksProcessed(clipboard.getRegion().getVolume());
            editSession.close();
            result.complete(RestoreResult.success(operation.getBlocksProcessed()));
        }
        catch (Exception e) {
            this.logger.log(Level.SEVERE, "Error during WorldEdit restoration", e);
            result.complete(RestoreResult.failure("WorldEdit operation failed: " + e.getMessage()));
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @NotNull
    private Optional<Clipboard> loadSnapshotClipboard(@NotNull Snapshot snapshot) {
        try {
            File snapshotFile = this.snapshotStorage.getSnapshotFile(snapshot);
            if (!snapshotFile.exists()) {
                this.logger.warning("Snapshot file not found: " + snapshotFile.getAbsolutePath());
                return Optional.empty();
            }
            try (FileInputStream fis = new FileInputStream(snapshotFile);){
                Optional<Clipboard> optional;
                block15: {
                    ClipboardFormat format = ClipboardFormats.findByFile((File)snapshotFile);
                    if (format == null) {
                        format = BuiltInClipboardFormat.SPONGE_SCHEMATIC;
                    }
                    ClipboardReader reader = format.getReader((InputStream)fis);
                    try {
                        Clipboard clipboard = reader.read();
                        this.logger.fine("Successfully loaded snapshot clipboard with " + clipboard.getRegion().getVolume() + " blocks");
                        optional = Optional.of(clipboard);
                        if (reader == null) break block15;
                    }
                    catch (Throwable throwable) {
                        if (reader != null) {
                            try {
                                reader.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    reader.close();
                }
                return optional;
            }
        }
        catch (IOException e) {
            this.logger.log(Level.WARNING, "Failed to load snapshot: " + snapshot.getFileName(), e);
            return Optional.empty();
        }
    }

    public boolean isArenaBeingRestored(@NotNull Arena arena) {
        return this.activeRestorations.values().stream().anyMatch(op -> op.getArena().getName().equals(arena.getName()) && op.getStatus() != RestoreStatus.COMPLETED && op.getStatus() != RestoreStatus.FAILED);
    }

    @NotNull
    public Optional<RestoreOperation> getActiveRestoration(@NotNull Arena arena) {
        return this.activeRestorations.values().stream().filter(op -> op.getArena().getName().equals(arena.getName()) && op.getStatus() != RestoreStatus.COMPLETED && op.getStatus() != RestoreStatus.FAILED).findFirst();
    }

    @NotNull
    public Map<UUID, RestoreOperation> getActiveRestorations() {
        return Map.copyOf(this.activeRestorations);
    }

    public boolean cancelRestoration(@NotNull UUID operationId) {
        RestoreOperation operation = this.activeRestorations.get(operationId);
        if (operation != null && operation.getStatus() == RestoreStatus.RESTORING_BLOCKS) {
            operation.setStatus(RestoreStatus.CANCELLED);
            this.activeRestorations.remove(operationId);
            this.logger.info("Cancelled restoration operation: " + String.valueOf(operationId));
            return true;
        }
        return false;
    }

    public void shutdown() {
        this.logger.info("Shutting down restore manager...");
        int cancelledCount = 0;
        for (UUID operationId : this.activeRestorations.keySet()) {
            if (!this.cancelRestoration(operationId)) continue;
            ++cancelledCount;
        }
        if (cancelledCount > 0) {
            this.logger.info("Cancelled " + cancelledCount + " active restoration operations");
        }
        this.activeRestorations.clear();
        this.logger.info("Restore manager shutdown complete");
    }

    public static final class RestoreResult {
        private final boolean success;
        private final String errorMessage;
        private final long blocksRestored;

        private RestoreResult(boolean success, @Nullable String errorMessage, long blocksRestored) {
            this.success = success;
            this.errorMessage = errorMessage;
            this.blocksRestored = blocksRestored;
        }

        @NotNull
        public static RestoreResult success(long blocksRestored) {
            return new RestoreResult(true, null, blocksRestored);
        }

        @NotNull
        public static RestoreResult failure(@NotNull String errorMessage) {
            return new RestoreResult(false, errorMessage, 0L);
        }

        public boolean isSuccess() {
            return this.success;
        }

        @Nullable
        public String getErrorMessage() {
            return this.errorMessage;
        }

        public long getBlocksRestored() {
            return this.blocksRestored;
        }
    }

    public static final class RestoreOperation {
        private final UUID operationId;
        private final Arena arena;
        private final Snapshot snapshot;
        private final CommandSender initiator;
        private final LocalDateTime startTime;
        private volatile RestoreStatus status;
        private volatile long blocksProcessed;
        private volatile LocalDateTime endTime;

        public RestoreOperation(@NotNull UUID operationId, @NotNull Arena arena, @NotNull Snapshot snapshot, @NotNull CommandSender initiator, @NotNull LocalDateTime startTime) {
            this.operationId = operationId;
            this.arena = arena;
            this.snapshot = snapshot;
            this.initiator = initiator;
            this.startTime = startTime;
            this.status = RestoreStatus.STARTING;
            this.blocksProcessed = 0L;
        }

        @NotNull
        public UUID getOperationId() {
            return this.operationId;
        }

        @NotNull
        public Arena getArena() {
            return this.arena;
        }

        @NotNull
        public Snapshot getSnapshot() {
            return this.snapshot;
        }

        @NotNull
        public CommandSender getInitiator() {
            return this.initiator;
        }

        @NotNull
        public LocalDateTime getStartTime() {
            return this.startTime;
        }

        @NotNull
        public RestoreStatus getStatus() {
            return this.status;
        }

        public long getBlocksProcessed() {
            return this.blocksProcessed;
        }

        @Nullable
        public LocalDateTime getEndTime() {
            return this.endTime;
        }

        public void setStatus(@NotNull RestoreStatus status) {
            this.status = status;
            if (status == RestoreStatus.COMPLETED || status == RestoreStatus.FAILED || status == RestoreStatus.CANCELLED) {
                this.endTime = LocalDateTime.now();
            }
        }

        public void setBlocksProcessed(long blocksProcessed) {
            this.blocksProcessed = blocksProcessed;
        }

        public double getDurationSeconds() {
            LocalDateTime end = this.endTime != null ? this.endTime : LocalDateTime.now();
            return (double)Duration.between(this.startTime, end).toMillis() / 1000.0;
        }
    }

    public static enum RestoreStatus {
        STARTING,
        LOADING_SNAPSHOT,
        RESTORING_BLOCKS,
        COMPLETED,
        FAILED,
        CANCELLED;

    }
}

