/*
 * Decompiled with CFR 0.152.
 */
package com.restoreit.core;

import com.restoreit.config.ConfigManager;
import com.restoreit.integration.WorldEditIntegration;
import com.restoreit.model.Arena;
import com.restoreit.model.Snapshot;
import com.restoreit.storage.SnapshotStorage;
import com.sk89q.worldedit.EditSession;
import com.sk89q.worldedit.WorldEdit;
import com.sk89q.worldedit.bukkit.BukkitAdapter;
import com.sk89q.worldedit.extent.Extent;
import com.sk89q.worldedit.extent.clipboard.BlockArrayClipboard;
import com.sk89q.worldedit.extent.clipboard.Clipboard;
import com.sk89q.worldedit.extent.clipboard.io.BuiltInClipboardFormat;
import com.sk89q.worldedit.extent.clipboard.io.ClipboardWriter;
import com.sk89q.worldedit.function.operation.ForwardExtentCopy;
import com.sk89q.worldedit.function.operation.Operation;
import com.sk89q.worldedit.function.operation.Operations;
import com.sk89q.worldedit.math.BlockVector3;
import com.sk89q.worldedit.regions.CuboidRegion;
import com.sk89q.worldedit.regions.Region;
import com.sk89q.worldedit.world.World;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;
import org.bukkit.Bukkit;
import org.bukkit.entity.Player;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.java.JavaPlugin;
import org.bukkit.scheduler.BukkitRunnable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class SnapshotManager {
    private final JavaPlugin plugin;
    private final WorldEditIntegration worldEditIntegration;
    private final ConfigManager configManager;
    private final SnapshotStorage snapshotStorage;
    private final Logger logger;
    private final Map<String, Map<String, Snapshot>> snapshots;
    private final Map<String, Snapshot> activeOperations;

    public SnapshotManager(@NotNull JavaPlugin plugin, @NotNull WorldEditIntegration worldEditIntegration, @NotNull ConfigManager configManager) {
        this.plugin = plugin;
        this.worldEditIntegration = worldEditIntegration;
        this.configManager = configManager;
        this.snapshotStorage = new SnapshotStorage(plugin, configManager);
        this.logger = plugin.getLogger();
        this.snapshots = new ConcurrentHashMap<String, Map<String, Snapshot>>();
        this.activeOperations = new ConcurrentHashMap<String, Snapshot>();
        this.loadSnapshotsFromStorage();
    }

    @NotNull
    public SnapshotStorage getSnapshotStorage() {
        return this.snapshotStorage;
    }

    @NotNull
    public CompletableFuture<Snapshot> createSnapshot(@NotNull Arena arena, @NotNull String snapshotName, @NotNull Player player, @Nullable String description) {
        CompletableFuture<Snapshot> future = new CompletableFuture<Snapshot>();
        try {
            int maxSnapshots;
            if (snapshotName.trim().isEmpty()) {
                future.completeExceptionally(new IllegalArgumentException("Snapshot name cannot be empty"));
                return future;
            }
            if (this.hasSnapshot(arena.getName(), snapshotName)) {
                future.completeExceptionally(new IllegalArgumentException("Snapshot '" + snapshotName + "' already exists in arena '" + arena.getName() + "'"));
                return future;
            }
            int currentCount = this.getSnapshotCount(arena.getName());
            if (currentCount >= (maxSnapshots = this.configManager.getMaxSnapshotsPerArena())) {
                future.completeExceptionally(new IllegalArgumentException("Arena has reached maximum snapshot limit (" + maxSnapshots + ")"));
                return future;
            }
            Snapshot snapshot = Snapshot.builder(snapshotName, arena.getName()).worldName(arena.getWorldName()).createdBy(player.getName()).description(description).status(Snapshot.SnapshotStatus.CREATING).build();
            String operationId = this.generateOperationId(arena.getName(), snapshotName);
            this.activeOperations.put(operationId, snapshot);
            if (this.configManager.isAsyncOperationsEnabled()) {
                this.createSnapshotAsync(arena, snapshot, future, operationId);
            } else {
                this.createSnapshotSync(arena, snapshot, future, operationId);
            }
        }
        catch (Exception e) {
            this.logger.severe("Failed to start snapshot creation: " + e.getMessage());
            future.completeExceptionally(e);
        }
        return future;
    }

    private void createSnapshotSync(@NotNull Arena arena, @NotNull Snapshot snapshot, @NotNull CompletableFuture<Snapshot> future, @NotNull String operationId) {
        try {
            Snapshot completedSnapshot = this.captureArenaBlocks(arena, snapshot);
            this.storeSnapshot(completedSnapshot);
            CompletableFuture.runAsync(() -> {
                try {
                    this.snapshotStorage.saveSnapshot(completedSnapshot);
                }
                catch (Exception e) {
                    this.logger.warning("Failed to save snapshot to persistent storage: " + e.getMessage());
                }
            });
            this.activeOperations.remove(operationId);
            future.complete(completedSnapshot);
        }
        catch (Exception e) {
            this.logger.severe("Failed to create snapshot '" + snapshot.getName() + "': " + e.getMessage());
            Snapshot errorSnapshot = snapshot.withStatus(Snapshot.SnapshotStatus.ERROR);
            this.storeSnapshot(errorSnapshot);
            this.activeOperations.remove(operationId);
            future.completeExceptionally(e);
        }
    }

    private void createSnapshotAsync(final @NotNull Arena arena, final @NotNull Snapshot snapshot, final @NotNull CompletableFuture<Snapshot> future, final @NotNull String operationId) {
        new BukkitRunnable(this){
            final /* synthetic */ SnapshotManager this$0;
            {
                this.this$0 = this$0;
            }

            public void run() {
                this.this$0.createSnapshotSync(arena, snapshot, future, operationId);
            }
        }.runTask((Plugin)this.plugin);
    }

    @NotNull
    private Snapshot captureArenaBlocks(@NotNull Arena arena, @NotNull Snapshot snapshot) throws Exception {
        if (!this.worldEditIntegration.isAvailable()) {
            throw new IllegalStateException("WorldEdit integration not available");
        }
        World world = BukkitAdapter.adapt((org.bukkit.World)Bukkit.getWorld((String)arena.getWorldName()));
        BlockVector3 min = BlockVector3.at((int)arena.getRegion().getMinX(), (int)arena.getRegion().getMinY(), (int)arena.getRegion().getMinZ());
        BlockVector3 max = BlockVector3.at((int)arena.getRegion().getMaxX(), (int)arena.getRegion().getMaxY(), (int)arena.getRegion().getMaxZ());
        CuboidRegion region = new CuboidRegion(world, min, max);
        BlockArrayClipboard clipboard = new BlockArrayClipboard((Region)region);
        try (EditSession editSession = WorldEdit.getInstance().newEditSession(world);){
            ForwardExtentCopy forwardExtentCopy = new ForwardExtentCopy((Extent)editSession, (Region)region, (Extent)clipboard, region.getMinimumPoint());
            Operations.complete((Operation)forwardExtentCopy);
        }
        File snapshotFile = this.snapshotStorage.getSnapshotFile(snapshot);
        long blockCount = region.getVolume();
        long fileSize = this.saveClipboardToFile((Clipboard)clipboard, snapshotFile);
        return Snapshot.builder(snapshot.getName(), snapshot.getArenaName()).worldName(snapshot.getWorldName()).createdAt(snapshot.getCreatedAt()).createdBy(snapshot.getCreatedBy()).description(snapshot.getDescription()).blockCount(blockCount).fileSize(fileSize).fileName(snapshot.getFileName()).status(Snapshot.SnapshotStatus.READY).build();
    }

    private long saveClipboardToFile(@NotNull Clipboard clipboard, @NotNull File file) throws IOException {
        file.getParentFile().mkdirs();
        BuiltInClipboardFormat format = BuiltInClipboardFormat.SPONGE_SCHEMATIC;
        try (FileOutputStream fos = new FileOutputStream(file);
             ClipboardWriter writer = format.getWriter((OutputStream)fos);){
            writer.write(clipboard);
        }
        return file.length();
    }

    public boolean deleteSnapshot(@NotNull String arenaName, @NotNull String snapshotName) {
        try {
            Map<String, Snapshot> arenaSnapshots = this.snapshots.get(arenaName.toLowerCase());
            if (arenaSnapshots == null) {
                return false;
            }
            Snapshot snapshot = arenaSnapshots.remove(snapshotName.toLowerCase());
            if (snapshot == null) {
                return false;
            }
            this.snapshotStorage.deleteSnapshot(snapshot);
            if (arenaSnapshots.isEmpty()) {
                this.snapshots.remove(arenaName.toLowerCase());
            }
            return true;
        }
        catch (Exception e) {
            this.logger.severe("Failed to delete snapshot '" + snapshotName + "': " + e.getMessage());
            return false;
        }
    }

    @Nullable
    public Snapshot getSnapshot(@NotNull String arenaName, @NotNull String snapshotName) {
        Map<String, Snapshot> arenaSnapshots = this.snapshots.get(arenaName.toLowerCase());
        if (arenaSnapshots == null) {
            return null;
        }
        return arenaSnapshots.get(snapshotName.toLowerCase());
    }

    public boolean hasSnapshot(@NotNull String arenaName, @NotNull String snapshotName) {
        return this.getSnapshot(arenaName, snapshotName) != null;
    }

    @NotNull
    public Collection<Snapshot> getSnapshots(@NotNull String arenaName) {
        Map<String, Snapshot> arenaSnapshots = this.snapshots.get(arenaName.toLowerCase());
        if (arenaSnapshots == null) {
            return Collections.emptyList();
        }
        return Collections.unmodifiableCollection(arenaSnapshots.values());
    }

    public int getSnapshotCount(@NotNull String arenaName) {
        Map<String, Snapshot> arenaSnapshots = this.snapshots.get(arenaName.toLowerCase());
        return arenaSnapshots != null ? arenaSnapshots.size() : 0;
    }

    public int getTotalSnapshotCount() {
        return this.snapshots.values().stream().mapToInt(Map::size).sum();
    }

    @NotNull
    public Collection<Snapshot> getAllSnapshots() {
        return this.snapshots.values().stream().flatMap(map -> map.values().stream()).toList();
    }

    private void storeSnapshot(@NotNull Snapshot snapshot) {
        this.snapshots.computeIfAbsent(snapshot.getArenaName().toLowerCase(), k -> new ConcurrentHashMap()).put(snapshot.getName().toLowerCase(), snapshot);
    }

    private void loadSnapshotsFromStorage() {
        try {
            Map<String, Map<String, Snapshot>> loadedSnapshots = this.snapshotStorage.loadSnapshots();
            for (Map.Entry<String, Map<String, Snapshot>> arenaEntry : loadedSnapshots.entrySet()) {
                for (Snapshot snapshot : arenaEntry.getValue().values()) {
                    this.storeSnapshot(snapshot);
                }
            }
        }
        catch (Exception e) {
            this.logger.severe("Failed to load snapshots from storage: " + e.getMessage());
            e.printStackTrace();
        }
    }

    @NotNull
    private String generateOperationId(@NotNull String arenaName, @NotNull String snapshotName) {
        return arenaName.toLowerCase() + ":" + snapshotName.toLowerCase() + ":" + System.currentTimeMillis();
    }

    @NotNull
    public Map<String, Snapshot> getActiveOperations() {
        return Collections.unmodifiableMap(this.activeOperations);
    }

    public boolean hasActiveOperations() {
        return !this.activeOperations.isEmpty();
    }

    @NotNull
    public String getDebugInfo() {
        StringBuilder info = new StringBuilder();
        info.append("Snapshot Manager Debug Info:\n");
        info.append("  Total Snapshots: ").append(this.getTotalSnapshotCount()).append("\n");
        info.append("  Active Operations: ").append(this.activeOperations.size()).append("\n");
        info.append("  Arenas with Snapshots: ").append(this.snapshots.size()).append("\n");
        for (Map.Entry<String, Map<String, Snapshot>> entry : this.snapshots.entrySet()) {
            info.append("    ").append(entry.getKey()).append(": ").append(entry.getValue().size()).append(" snapshots\n");
        }
        return info.toString();
    }

    public int cleanupOldSnapshots(@NotNull String arenaName) {
        Map<String, Snapshot> arenaSnapshots = this.snapshots.get(arenaName.toLowerCase());
        if (arenaSnapshots == null || arenaSnapshots.isEmpty()) {
            return 0;
        }
        int maxSnapshots = this.configManager.getMaxSnapshotsPerArena();
        if (arenaSnapshots.size() <= maxSnapshots) {
            return 0;
        }
        List<Snapshot> sortedSnapshots = arenaSnapshots.values().stream().sorted(Comparator.comparing(Snapshot::getCreatedAt)).toList();
        int toRemove = arenaSnapshots.size() - maxSnapshots;
        int removed = 0;
        for (int i = 0; i < toRemove && i < sortedSnapshots.size(); ++i) {
            Snapshot snapshot = sortedSnapshots.get(i);
            if (!this.deleteSnapshot(arenaName, snapshot.getName())) continue;
            ++removed;
        }
        return removed;
    }

    public void shutdown() {
        long timeout = System.currentTimeMillis() + 30000L;
        while (!this.activeOperations.isEmpty() && System.currentTimeMillis() < timeout) {
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
        if (!this.activeOperations.isEmpty()) {
            this.logger.warning("Shutdown with " + this.activeOperations.size() + " active operations remaining");
        }
    }
}

