/*
 * Decompiled with CFR 0.152.
 */
package com.restoreit.storage;

import com.restoreit.model.Arena;
import com.restoreit.model.ArenaRegion;
import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.bukkit.Bukkit;
import org.bukkit.World;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class ArenaStorage {
    private static final String ARENAS_FILE = "arenas.yml";
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ISO_LOCAL_DATE_TIME;
    private final File arenasFile;
    private final Logger logger;

    public ArenaStorage(@NotNull JavaPlugin plugin) {
        this.arenasFile = new File(plugin.getDataFolder(), ARENAS_FILE);
        this.logger = plugin.getLogger();
        if (!plugin.getDataFolder().exists()) {
            plugin.getDataFolder().mkdirs();
        }
    }

    @NotNull
    public Map<String, Arena> loadArenas() {
        HashMap<String, Arena> arenas = new HashMap<String, Arena>();
        if (!this.arenasFile.exists()) {
            return arenas;
        }
        try {
            YamlConfiguration config = YamlConfiguration.loadConfiguration((File)this.arenasFile);
            ConfigurationSection arenasSection = config.getConfigurationSection("arenas");
            if (arenasSection == null) {
                return arenas;
            }
            for (String arenaName : arenasSection.getKeys(false)) {
                try {
                    Arena arena = this.loadArena(arenasSection.getConfigurationSection(arenaName));
                    if (arena == null) continue;
                    arenas.put(arenaName.toLowerCase(), arena);
                }
                catch (Exception e) {
                    this.logger.severe("Failed to load arena '" + arenaName + "': " + e.getMessage());
                }
            }
        }
        catch (Exception e) {
            this.logger.severe("Failed to load arenas from storage: " + e.getMessage());
            e.printStackTrace();
        }
        return arenas;
    }

    public boolean saveArenas(@NotNull Collection<Arena> arenas) {
        try {
            YamlConfiguration config = new YamlConfiguration();
            for (Arena arena : arenas) {
                this.saveArena(config, arena);
            }
            config.save(this.arenasFile);
            return true;
        }
        catch (IOException e) {
            this.logger.severe("Failed to save arenas to storage: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    public boolean saveArena(@NotNull Arena arena) {
        try {
            YamlConfiguration config = this.arenasFile.exists() ? YamlConfiguration.loadConfiguration((File)this.arenasFile) : new YamlConfiguration();
            this.saveArena(config, arena);
            config.save(this.arenasFile);
            return true;
        }
        catch (IOException e) {
            this.logger.severe("Failed to save arena '" + arena.getName() + "': " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    public boolean deleteArena(@NotNull String arenaName) {
        try {
            if (!this.arenasFile.exists()) {
                return true;
            }
            YamlConfiguration config = YamlConfiguration.loadConfiguration((File)this.arenasFile);
            ConfigurationSection arenasSection = config.getConfigurationSection("arenas");
            if (arenasSection != null && arenasSection.contains(arenaName)) {
                arenasSection.set(arenaName, null);
                config.save(this.arenasFile);
                return true;
            }
            return false;
        }
        catch (IOException e) {
            this.logger.severe("Failed to delete arena '" + arenaName + "' from storage: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    @Nullable
    private Arena loadArena(@Nullable ConfigurationSection section) {
        if (section == null) {
            return null;
        }
        try {
            String name = section.getName();
            String worldName = section.getString("world");
            String createdBy = section.getString("created_by", "Unknown");
            String createdAtStr = section.getString("created_at");
            if (worldName == null || worldName.trim().isEmpty()) {
                this.logger.warning("Arena '" + name + "' has invalid world name");
                return null;
            }
            World world = Bukkit.getWorld((String)worldName);
            if (world == null) {
                this.logger.warning("Arena '" + name + "' references non-existent world: " + worldName);
                return null;
            }
            ConfigurationSection regionSection = section.getConfigurationSection("region");
            if (regionSection == null) {
                this.logger.warning("Arena '" + name + "' has no region data");
                return null;
            }
            ArenaRegion region = this.loadArenaRegion(regionSection, worldName);
            if (region == null) {
                this.logger.warning("Arena '" + name + "' has invalid region data");
                return null;
            }
            LocalDateTime createdAt = LocalDateTime.now();
            if (createdAtStr != null) {
                try {
                    createdAt = LocalDateTime.parse(createdAtStr, TIMESTAMP_FORMAT);
                }
                catch (DateTimeParseException e) {
                    this.logger.warning("Arena '" + name + "' has invalid timestamp: " + createdAtStr);
                }
            }
            List snapshotList = section.getStringList("snapshots");
            HashSet<String> snapshots = new HashSet<String>(snapshotList);
            return Arena.builder(name).world(world).region(region).createdBy(createdBy).createdAt(createdAt).addSnapshots(snapshots).build();
        }
        catch (Exception e) {
            this.logger.severe("Failed to load arena from section: " + e.getMessage());
            return null;
        }
    }

    @Nullable
    private ArenaRegion loadArenaRegion(@NotNull ConfigurationSection section, @NotNull String worldName) {
        try {
            int minX = section.getInt("min_x");
            int minY = section.getInt("min_y");
            int minZ = section.getInt("min_z");
            int maxX = section.getInt("max_x");
            int maxY = section.getInt("max_y");
            int maxZ = section.getInt("max_z");
            return new ArenaRegion(worldName, minX, minY, minZ, maxX, maxY, maxZ);
        }
        catch (Exception e) {
            this.logger.severe("Failed to load arena region: " + e.getMessage());
            return null;
        }
    }

    private void saveArena(@NotNull YamlConfiguration config, @NotNull Arena arena) {
        String path = "arenas." + arena.getName();
        config.set(path + ".world", (Object)arena.getWorldName());
        config.set(path + ".created_by", (Object)arena.getCreatedBy());
        config.set(path + ".created_at", (Object)arena.getCreatedAt().format(TIMESTAMP_FORMAT));
        ArenaRegion region = arena.getRegion();
        config.set(path + ".region.min_x", (Object)region.getMinX());
        config.set(path + ".region.min_y", (Object)region.getMinY());
        config.set(path + ".region.min_z", (Object)region.getMinZ());
        config.set(path + ".region.max_x", (Object)region.getMaxX());
        config.set(path + ".region.max_y", (Object)region.getMaxY());
        config.set(path + ".region.max_z", (Object)region.getMaxZ());
        config.set(path + ".snapshots", new ArrayList<String>(arena.getSnapshots()));
    }

    public boolean storageExists() {
        return this.arenasFile.exists();
    }

    @NotNull
    public String getStorageInfo() {
        if (!this.arenasFile.exists()) {
            return "Storage file does not exist";
        }
        return "Storage file: " + this.arenasFile.getAbsolutePath() + " (Size: " + this.arenasFile.length() + " bytes, Last modified: " + String.valueOf(new Date(this.arenasFile.lastModified())) + ")";
    }
}

