/*
 * Decompiled with CFR 0.152.
 */
package com.restoreit.storage;

import com.restoreit.config.ConfigManager;
import com.restoreit.model.Snapshot;
import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.bukkit.configuration.ConfigurationSection;
import org.bukkit.configuration.file.YamlConfiguration;
import org.bukkit.plugin.java.JavaPlugin;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class SnapshotStorage {
    private static final String SNAPSHOTS_FILE = "snapshots.yml";
    private static final String SNAPSHOTS_DIR = "snapshots";
    private static final DateTimeFormatter TIMESTAMP_FORMAT = DateTimeFormatter.ISO_LOCAL_DATE_TIME;
    private final File snapshotsFile;
    private final File snapshotsDir;
    private final Logger logger;

    public SnapshotStorage(@NotNull JavaPlugin plugin, @NotNull ConfigManager configManager) {
        this.snapshotsFile = new File(plugin.getDataFolder(), SNAPSHOTS_FILE);
        this.snapshotsDir = new File(plugin.getDataFolder(), SNAPSHOTS_DIR);
        this.logger = plugin.getLogger();
        if (!plugin.getDataFolder().exists()) {
            plugin.getDataFolder().mkdirs();
        }
        if (!this.snapshotsDir.exists()) {
            this.snapshotsDir.mkdirs();
        }
    }

    @NotNull
    public Map<String, Map<String, Snapshot>> loadSnapshots() {
        HashMap<String, Map<String, Snapshot>> snapshots = new HashMap<String, Map<String, Snapshot>>();
        if (!this.snapshotsFile.exists()) {
            return snapshots;
        }
        try {
            YamlConfiguration config = YamlConfiguration.loadConfiguration((File)this.snapshotsFile);
            ConfigurationSection snapshotsSection = config.getConfigurationSection(SNAPSHOTS_DIR);
            if (snapshotsSection == null) {
                return snapshots;
            }
            for (String arenaName : snapshotsSection.getKeys(false)) {
                ConfigurationSection arenaSection = snapshotsSection.getConfigurationSection(arenaName);
                if (arenaSection == null) continue;
                HashMap<String, Snapshot> arenaSnapshots = new HashMap<String, Snapshot>();
                for (String snapshotName : arenaSection.getKeys(false)) {
                    try {
                        Snapshot snapshot = this.loadSnapshot(arenaSection.getConfigurationSection(snapshotName));
                        if (snapshot == null) continue;
                        arenaSnapshots.put(snapshotName.toLowerCase(), snapshot);
                    }
                    catch (Exception e) {
                        this.logger.severe("Failed to load snapshot '" + arenaName + ":" + snapshotName + "': " + e.getMessage());
                    }
                }
                if (arenaSnapshots.isEmpty()) continue;
                snapshots.put(arenaName.toLowerCase(), arenaSnapshots);
            }
        }
        catch (Exception e) {
            this.logger.severe("Failed to load snapshots from storage: " + e.getMessage());
            e.printStackTrace();
        }
        return snapshots;
    }

    public boolean saveSnapshots(@NotNull Map<String, Map<String, Snapshot>> snapshots) {
        try {
            YamlConfiguration config = new YamlConfiguration();
            for (Map.Entry<String, Map<String, Snapshot>> arenaEntry : snapshots.entrySet()) {
                for (Snapshot snapshot : arenaEntry.getValue().values()) {
                    this.saveSnapshot(config, snapshot);
                }
            }
            config.save(this.snapshotsFile);
            return true;
        }
        catch (IOException e) {
            this.logger.severe("Failed to save snapshots to storage: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    public boolean saveSnapshot(@NotNull Snapshot snapshot) {
        try {
            YamlConfiguration config = this.snapshotsFile.exists() ? YamlConfiguration.loadConfiguration((File)this.snapshotsFile) : new YamlConfiguration();
            this.saveSnapshot(config, snapshot);
            config.save(this.snapshotsFile);
            return true;
        }
        catch (IOException e) {
            this.logger.severe("Failed to save snapshot '" + snapshot.getFullName() + "': " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    public boolean deleteSnapshot(@NotNull Snapshot snapshot) {
        try {
            File snapshotFile;
            String path;
            YamlConfiguration config;
            boolean configDeleted = false;
            boolean fileDeleted = false;
            if (this.snapshotsFile.exists() && (config = YamlConfiguration.loadConfiguration((File)this.snapshotsFile)).contains(path = "snapshots." + snapshot.getArenaName() + "." + snapshot.getName())) {
                config.set(path, null);
                ConfigurationSection arenaSection = config.getConfigurationSection("snapshots." + snapshot.getArenaName());
                if (arenaSection == null || arenaSection.getKeys(false).isEmpty()) {
                    config.set("snapshots." + snapshot.getArenaName(), null);
                }
                config.save(this.snapshotsFile);
                configDeleted = true;
            }
            if ((snapshotFile = this.getSnapshotFile(snapshot)).exists() && !(fileDeleted = snapshotFile.delete())) {
                this.logger.warning("Failed to delete snapshot file: " + snapshotFile.getAbsolutePath());
            }
            return configDeleted;
        }
        catch (IOException e) {
            this.logger.severe("Failed to delete snapshot '" + snapshot.getFullName() + "': " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    @NotNull
    public File getSnapshotFile(@NotNull Snapshot snapshot) {
        return new File(this.snapshotsDir, snapshot.getFileName());
    }

    public boolean snapshotFileExists(@NotNull Snapshot snapshot) {
        return this.getSnapshotFile(snapshot).exists();
    }

    public long getSnapshotFileSize(@NotNull Snapshot snapshot) {
        File file = this.getSnapshotFile(snapshot);
        return file.exists() ? file.length() : 0L;
    }

    @Nullable
    private Snapshot loadSnapshot(@Nullable ConfigurationSection section) {
        if (section == null) {
            return null;
        }
        try {
            String arenaName = section.getParent().getName();
            String snapshotName = section.getName();
            String worldName = section.getString("world");
            String createdBy = section.getString("created_by", "Unknown");
            String createdAtStr = section.getString("created_at");
            String description = section.getString("description");
            long blockCount = section.getLong("block_count", 0L);
            long fileSize = section.getLong("file_size", 0L);
            String fileName = section.getString("file_name");
            String statusStr = section.getString("status", "READY");
            if (worldName == null || worldName.trim().isEmpty()) {
                this.logger.warning("Snapshot '" + arenaName + ":" + snapshotName + "' has invalid world name");
                return null;
            }
            if (fileName == null || fileName.trim().isEmpty()) {
                this.logger.warning("Snapshot '" + arenaName + ":" + snapshotName + "' has invalid file name");
                return null;
            }
            LocalDateTime createdAt = LocalDateTime.now();
            if (createdAtStr != null) {
                try {
                    createdAt = LocalDateTime.parse(createdAtStr, TIMESTAMP_FORMAT);
                }
                catch (DateTimeParseException e) {
                    this.logger.warning("Snapshot '" + arenaName + ":" + snapshotName + "' has invalid timestamp: " + createdAtStr);
                }
            }
            Snapshot.SnapshotStatus status = Snapshot.SnapshotStatus.READY;
            try {
                status = Snapshot.SnapshotStatus.valueOf(statusStr);
            }
            catch (IllegalArgumentException e) {
                this.logger.warning("Snapshot '" + arenaName + ":" + snapshotName + "' has invalid status: " + statusStr);
            }
            ConfigurationSection metadataSection = section.getConfigurationSection("metadata");
            HashMap<String, Object> metadata = new HashMap<String, Object>();
            if (metadataSection != null) {
                for (String key : metadataSection.getKeys(false)) {
                    metadata.put(key, metadataSection.get(key));
                }
            }
            Snapshot.Builder builder = Snapshot.builder(snapshotName, arenaName).worldName(worldName).createdBy(createdBy).createdAt(createdAt).blockCount(blockCount).fileSize(fileSize).fileName(fileName).status(status);
            if (description != null) {
                builder.description(description);
            }
            for (Map.Entry entry : metadata.entrySet()) {
                builder.metadata((String)entry.getKey(), entry.getValue());
            }
            return builder.build();
        }
        catch (Exception e) {
            this.logger.severe("Failed to load snapshot from section: " + e.getMessage());
            return null;
        }
    }

    private void saveSnapshot(@NotNull YamlConfiguration config, @NotNull Snapshot snapshot) {
        String path = "snapshots." + snapshot.getArenaName() + "." + snapshot.getName();
        config.set(path + ".world", (Object)snapshot.getWorldName());
        config.set(path + ".created_by", (Object)snapshot.getCreatedBy());
        config.set(path + ".created_at", (Object)snapshot.getCreatedAt().format(TIMESTAMP_FORMAT));
        config.set(path + ".block_count", (Object)snapshot.getBlockCount());
        config.set(path + ".file_size", (Object)snapshot.getFileSize());
        config.set(path + ".file_name", (Object)snapshot.getFileName());
        config.set(path + ".status", (Object)snapshot.getStatus().name());
        if (snapshot.getDescription() != null) {
            config.set(path + ".description", (Object)snapshot.getDescription());
        }
        if (!snapshot.getMetadata().isEmpty()) {
            for (Map.Entry<String, Object> entry : snapshot.getMetadata().entrySet()) {
                config.set(path + ".metadata." + entry.getKey(), entry.getValue());
            }
        }
    }

    public boolean storageExists() {
        return this.snapshotsFile.exists();
    }

    @NotNull
    public String getStorageInfo() {
        File[] files;
        long totalSize = 0L;
        int fileCount = 0;
        if (this.snapshotsDir.exists() && this.snapshotsDir.isDirectory() && (files = this.snapshotsDir.listFiles()) != null) {
            for (File file : files) {
                if (!file.isFile() || !file.getName().endsWith(".schem") && !file.getName().endsWith(".snapshot")) continue;
                totalSize += file.length();
                ++fileCount;
            }
        }
        Object sizeStr = totalSize < 1024L ? totalSize + " B" : (totalSize < 0x100000L ? String.format("%.1f KB", (double)totalSize / 1024.0) : (totalSize < 0x40000000L ? String.format("%.1f MB", (double)totalSize / 1048576.0) : String.format("%.1f GB", (double)totalSize / 1.073741824E9)));
        return "Storage directory: " + this.snapshotsDir.getAbsolutePath() + " (" + fileCount + " files, " + (String)sizeStr + " total)";
    }

    public int cleanupOrphanedFiles(@NotNull Set<String> validSnapshots) {
        if (!this.snapshotsDir.exists() || !this.snapshotsDir.isDirectory()) {
            return 0;
        }
        File[] files = this.snapshotsDir.listFiles();
        if (files == null) {
            return 0;
        }
        int cleaned = 0;
        for (File file : files) {
            if (!file.isFile() || !file.getName().endsWith(".schem") && !file.getName().endsWith(".snapshot") || validSnapshots.contains(file.getName())) continue;
            if (file.delete()) {
                ++cleaned;
                continue;
            }
            this.logger.warning("Failed to delete orphaned snapshot file: " + file.getName());
        }
        return cleaned;
    }
}

