/*
 * Decompiled with CFR 0.152.
 */
package org.texboobcat.catQueue.core;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.texboobcat.catQueue.config.ConfigManager;

public class QueueManager {
    private final Logger logger;
    private final ConfigManager config;
    private final NavigableMap<Integer, Queue<QueueEntry>> queues = new TreeMap(Comparator.reverseOrder());
    private final Map<UUID, Integer> playerPriority = new ConcurrentHashMap<UUID, Integer>();
    private final Map<UUID, QueueEntry> entriesByUuid = new ConcurrentHashMap<UUID, QueueEntry>();
    private final AtomicLong totalJoined = new AtomicLong();
    private volatile double emaRatePerMinute = 0.0;
    private volatile long lastServiceEpochSec = 0L;
    private final Map<String, Double> emaRatePerMinuteByServer = new ConcurrentHashMap<String, Double>();
    private final Map<String, Long> lastServiceEpochSecByServer = new ConcurrentHashMap<String, Long>();
    private final Map<UUID, Integer> lastDecayMinuteApplied = new ConcurrentHashMap<UUID, Integer>();
    private final Map<UUID, GraceRec> grace = new ConcurrentHashMap<UUID, GraceRec>();

    public QueueManager(Logger logger, ConfigManager config) {
        this.logger = logger;
        this.config = config;
        this.queues.putIfAbsent(0, new ConcurrentLinkedQueue());
    }

    public boolean isQueued(UUID uuid) {
        return this.playerPriority.containsKey(uuid);
    }

    public int getPosition(UUID uuid) {
        Integer p = this.playerPriority.get(uuid);
        if (p == null) {
            return -1;
        }
        int pos = 0;
        for (Map.Entry e : this.queues.entrySet()) {
            int pr = (Integer)e.getKey();
            Queue q = (Queue)e.getValue();
            if (pr > p) {
                pos += q.size();
                continue;
            }
            if (pr != p) continue;
            int i = 0;
            for (QueueEntry qe : q) {
                if (Objects.equals(qe.uuid, uuid)) {
                    return pos + i + 1;
                }
                ++i;
            }
        }
        return -1;
    }

    public Duration estimateWait(UUID uuid, int slotRatePerMinute, String targetServerName) {
        double perTarget;
        int position = this.getPosition(uuid);
        if (position <= 0 || slotRatePerMinute <= 0) {
            return Duration.ZERO;
        }
        double d = perTarget = targetServerName == null ? 0.0 : this.emaRatePerMinuteByServer.getOrDefault(targetServerName, 0.0);
        double rate = perTarget > 0.0 ? perTarget : (this.emaRatePerMinute > 0.0 ? this.emaRatePerMinute : (double)slotRatePerMinute);
        long seconds = (long)Math.ceil((double)position * 60.0 / Math.max(0.001, rate));
        return Duration.ofSeconds(seconds);
    }

    public Duration estimateWait(UUID uuid, int slotRatePerMinute) {
        return this.estimateWait(uuid, slotRatePerMinute, null);
    }

    public int resolvePriority(boolean isStaff, boolean isVip) {
        int base = 0;
        if (this.config.priorityEnabled()) {
            if (isStaff) {
                base = Math.max(base, 100);
            }
            if (isVip) {
                base = Math.max(base, 10);
            }
        }
        return base;
    }

    public void enqueue(UUID uuid, int priority) {
        if (this.isQueued(uuid)) {
            return;
        }
        QueueEntry entry = new QueueEntry(uuid, priority);
        this.queues.computeIfAbsent(priority, k -> new ConcurrentLinkedQueue()).add(entry);
        this.playerPriority.put(uuid, priority);
        this.entriesByUuid.put(uuid, entry);
        this.totalJoined.incrementAndGet();
        this.logger.debug("Enqueued player {} with priority {}. Queue size now {}", new Object[]{uuid, priority, this.size()});
    }

    public void enqueueRestored(UUID uuid, int priority, Instant joinedAt) {
        if (this.isQueued(uuid)) {
            return;
        }
        QueueEntry entry = new QueueEntry(uuid, priority, joinedAt);
        this.queues.computeIfAbsent(priority, k -> new ConcurrentLinkedQueue()).add(entry);
        this.playerPriority.put(uuid, priority);
        this.entriesByUuid.put(uuid, entry);
        this.totalJoined.incrementAndGet();
        this.logger.debug("Restored player {} with priority {} (joinedAt {}). Queue size now {}", new Object[]{uuid, priority, entry.joinedAt, this.size()});
    }

    public boolean remove(UUID uuid) {
        Integer p = this.playerPriority.remove(uuid);
        if (p == null) {
            return false;
        }
        Queue q = (Queue)this.queues.get(p);
        if (q == null) {
            return false;
        }
        boolean removed = q.removeIf(e -> e.uuid.equals(uuid));
        if (removed) {
            this.entriesByUuid.remove(uuid);
            this.logger.debug("Removed player {} from queue. Queue size now {}", (Object)uuid, (Object)this.size());
        }
        return removed;
    }

    public Optional<UUID> peekNext() {
        for (Queue q : this.queues.values()) {
            QueueEntry e = (QueueEntry)q.peek();
            if (e == null) continue;
            return Optional.of(e.uuid);
        }
        return Optional.empty();
    }

    public Optional<UUID> pollNext() {
        for (Queue q : this.queues.values()) {
            QueueEntry e = (QueueEntry)q.poll();
            if (e == null) continue;
            this.playerPriority.remove(e.uuid);
            this.entriesByUuid.remove(e.uuid);
            this.recordService();
            return Optional.of(e.uuid);
        }
        return Optional.empty();
    }

    public Optional<UUID> pollNext(String targetServerName) {
        for (Queue q : this.queues.values()) {
            QueueEntry e = (QueueEntry)q.poll();
            if (e == null) continue;
            this.playerPriority.remove(e.uuid);
            this.entriesByUuid.remove(e.uuid);
            this.recordService(targetServerName);
            return Optional.of(e.uuid);
        }
        return Optional.empty();
    }

    public int size() {
        return this.playerPriority.size();
    }

    public List<UUID> listQueued() {
        return new ArrayList<UUID>(this.playerPriority.keySet());
    }

    public Optional<QueueEntry> getEntry(UUID uuid) {
        return Optional.ofNullable(this.entriesByUuid.get(uuid));
    }

    public boolean promote(UUID uuid, int deltaPriority) {
        Integer p = this.playerPriority.get(uuid);
        if (p == null) {
            return false;
        }
        Queue q = (Queue)this.queues.get(p);
        if (q == null) {
            return false;
        }
        QueueEntry target = null;
        for (QueueEntry e : q) {
            if (!e.uuid.equals(uuid)) continue;
            target = e;
            break;
        }
        if (target == null) {
            return false;
        }
        q.remove(target);
        int newPriority = p + Math.max(1, deltaPriority);
        this.playerPriority.put(uuid, newPriority);
        this.entriesByUuid.put(uuid, new QueueEntry(uuid, newPriority, target.joinedAt));
        this.queues.computeIfAbsent(newPriority, k -> new ConcurrentLinkedQueue()).add(this.entriesByUuid.get(uuid));
        this.logger.debug("Promoted {} from priority {} to {}", new Object[]{uuid, p, newPriority});
        return true;
    }

    public void recordService() {
        long now = System.currentTimeMillis() / 1000L;
        if (this.lastServiceEpochSec == 0L) {
            this.lastServiceEpochSec = now;
            if (this.emaRatePerMinute <= 0.0) {
                this.emaRatePerMinute = 1.0;
            }
            return;
        }
        long dt = Math.max(1L, now - this.lastServiceEpochSec);
        this.lastServiceEpochSec = now;
        double instRatePerMin = 60.0 / (double)dt;
        double alpha = 0.2;
        this.emaRatePerMinute = alpha * instRatePerMin + (1.0 - alpha) * Math.max(0.0, this.emaRatePerMinute);
    }

    public void recordService(String targetServerName) {
        if (targetServerName == null || targetServerName.isBlank()) {
            this.recordService();
            return;
        }
        long now = System.currentTimeMillis() / 1000L;
        long last = this.lastServiceEpochSecByServer.getOrDefault(targetServerName, 0L);
        if (last == 0L) {
            this.lastServiceEpochSecByServer.put(targetServerName, now);
            this.emaRatePerMinuteByServer.putIfAbsent(targetServerName, 1.0);
            return;
        }
        long dt = Math.max(1L, now - last);
        this.lastServiceEpochSecByServer.put(targetServerName, now);
        double instRatePerMin = 60.0 / (double)dt;
        double alpha = 0.2;
        double prev = Math.max(0.0, this.emaRatePerMinuteByServer.getOrDefault(targetServerName, 0.0));
        double updated = alpha * instRatePerMin + (1.0 - alpha) * prev;
        this.emaRatePerMinuteByServer.put(targetServerName, updated);
        this.recordService();
    }

    public int applyPriorityDecay() {
        if (!this.config.decayEnabled()) {
            return 0;
        }
        int minutesPerStep = Math.max(1, this.config.decayMinutesPerStep());
        int minPriority = this.config.decayMinPriority();
        int amountPerStep = Math.max(1, this.config.decayAmountPerStep());
        int affected = 0;
        for (Map.Entry<UUID, QueueEntry> ent : new ArrayList<Map.Entry<UUID, QueueEntry>>(this.entriesByUuid.entrySet())) {
            UUID uuid = ent.getKey();
            QueueEntry qe = ent.getValue();
            Integer curPr = this.playerPriority.get(uuid);
            if (curPr == null) continue;
            long minutes = Duration.between(qe.joinedAt, Instant.now()).toMinutes();
            int lastApplied = this.lastDecayMinuteApplied.getOrDefault(uuid, -1);
            if (minutes < (long)minutesPerStep || (int)(minutes / (long)minutesPerStep) <= lastApplied) continue;
            if (curPr > minPriority) {
                Queue q = (Queue)this.queues.get(curPr);
                if (q != null) {
                    q.removeIf(e -> e.uuid.equals(uuid));
                }
                int newPriority = Math.max(minPriority, curPr - amountPerStep);
                this.playerPriority.put(uuid, newPriority);
                this.entriesByUuid.put(uuid, new QueueEntry(uuid, newPriority, qe.joinedAt));
                this.queues.computeIfAbsent(newPriority, k -> new ConcurrentLinkedQueue()).add(this.entriesByUuid.get(uuid));
                ++affected;
                this.logger.debug("Priority decay: {} {}->{} (minutes since join: {})", new Object[]{uuid, curPr, newPriority, minutes});
            }
            this.lastDecayMinuteApplied.put(uuid, (int)(minutes / (long)minutesPerStep));
        }
        return affected;
    }

    public void saveGraceOnDisconnect(UUID uuid, int minutes) {
        QueueEntry e = this.entriesByUuid.get(uuid);
        Integer p = this.playerPriority.get(uuid);
        if (e == null || p == null) {
            return;
        }
        long expire = System.currentTimeMillis() / 1000L + (long)Math.max(1, minutes) * 60L;
        this.grace.put(uuid, new GraceRec(p, e.joinedAt, expire));
        this.remove(uuid);
    }

    public boolean tryRestoreFromGrace(UUID uuid) {
        GraceRec rec = this.grace.remove(uuid);
        if (rec == null) {
            return false;
        }
        long now = System.currentTimeMillis() / 1000L;
        if (now > rec.expireAtEpochSec) {
            return false;
        }
        this.enqueueRestored(uuid, rec.priority, rec.joinedAt);
        this.promote(uuid, 1);
        return true;
    }

    public void cleanupGrace() {
        long now = System.currentTimeMillis() / 1000L;
        this.grace.entrySet().removeIf(e -> now > ((GraceRec)e.getValue()).expireAtEpochSec);
    }

    public static class QueueEntry {
        public final UUID uuid;
        public final Instant joinedAt;
        public final int priority;

        public QueueEntry(UUID uuid, int priority) {
            this.uuid = uuid;
            this.priority = priority;
            this.joinedAt = Instant.now();
        }

        public QueueEntry(UUID uuid, int priority, Instant joinedAt) {
            this.uuid = uuid;
            this.priority = priority;
            this.joinedAt = joinedAt == null ? Instant.now() : joinedAt;
        }
    }

    private static class GraceRec {
        final int priority;
        final Instant joinedAt;
        final long expireAtEpochSec;

        GraceRec(int priority, Instant joinedAt, long expireAtEpochSec) {
            this.priority = priority;
            this.joinedAt = joinedAt;
            this.expireAtEpochSec = expireAtEpochSec;
        }
    }
}

