#version 120
uniform int isEyeInWater;
uniform sampler2D texture;
uniform sampler2D lightmap;
uniform sampler2D normals;
uniform sampler2D specular;

uniform vec3 sunPosition;
uniform vec3 moonPosition;
uniform vec3 cameraPosition;
uniform float frameTimeCounter;
uniform int worldTime;
uniform float rainStrength;
uniform float wetness;

uniform ivec2 eyeBrightnessSmooth;
uniform float near;
uniform float far;

varying vec2 texcoord;
varying vec2 lmcoord;
varying vec4 glcolor;
varying vec3 normal;
varying vec3 worldPos;
varying vec3 viewPos;
varying float blockId;
varying float waveEffect;

vec3 saturation(vec3 color, float sat) {
    float gray = dot(color, vec3(0.299, 0.587, 0.114));
    return mix(vec3(gray), color, sat);
}

vec3 contrast(vec3 color, float cont) {
    return (color - 0.6) * cont + 0.5;
}

vec3 screenSpaceReflection(vec3 worldPos, vec3 normal, vec3 viewDir) {
    vec3 reflectDir = reflect(viewDir, normal);
    vec3 reflectionColor = vec3(0.0);
    
    float fresnel = pow(1.0 - max(dot(-viewDir, normal), 0.0), 1.0);
    
    vec3 skyColor = mix(vec3(0.5, 0.7, 1.0), vec3(1.0, 0.9, 0.7), abs(reflectDir.y));
    reflectionColor = skyColor * fresnel * 0.1;
    
    return reflectionColor;
}

vec3 calculateAdvancedLighting(vec3 color, vec3 normal, vec3 lightDir, vec3 viewDir, vec3 worldPos) {
    float NdotL = max(dot(normal, lightDir), 0.0);
    float NdotV = max(dot(normal, -viewDir), 0.0);

    vec3 ambient = vec3(0.4, 0.4, 0.4);
    
    vec3 diffuse = vec3(1.2, 1.1, 0.9) * NdotL;
    
    vec3 halfDir = normalize(lightDir - viewDir);
    
    vec3 reflection = screenSpaceReflection(worldPos, normal, viewDir);
    
    vec3 bouncedLight = vec3(0.2, 0.25, 0.3) * (1.0 - NdotL) * 0.5;
    
    return color * (ambient + diffuse + bouncedLight) + reflection;
}

vec3 getDayNightColor() {
    float dayTime = float(worldTime) / 24000.0;
    
    // Define colors for different times
    vec3 morningColor = vec3(0.8, 0.7, 0.8);
    vec3 dayColor = vec3(0.9, 0.8, 0.9);
    vec3 nightColor = vec3(0.6, 0.75, 0.7);
    
    if (worldTime <= 6000) {
        float t = smoothstep(1000.0, 6000.0, float(worldTime));
        return mix(morningColor, dayColor, t);
    } else if (worldTime <= 13000) {
        float t = smoothstep(6000.0, 13000.0, float(worldTime));
        return mix(dayColor, nightColor, t);
    } else if (worldTime <= 24000) {
        float t = smoothstep(13000.0, 16000.0, float(worldTime));
        return mix(nightColor, morningColor, t);
    }
}
void main() {
    vec2 coord = texcoord;
    vec4 color = texture2D(texture, texcoord) * glcolor;
    vec3 lightmapColor = texture2D(lightmap, lmcoord).rgb;
    
    if (color.a < 0.1) {
        discard;
    }
    
    vec3 dayNightTint = getDayNightColor();
    color.rgb *= dayNightTint;
    
    float minBrightness = 0.1;
    float currentBrightness = dot(color.rgb, vec3(0.299, 0.587, 0.114));
    if (currentBrightness < minBrightness) {
        color.rgb = mix(color.rgb, color.rgb * (minBrightness / max(currentBrightness, 0.001)), 0.4);
    }
   

    
    vec3 lightDir = normalize(sunPosition - worldPos);
    vec3 viewDir = normalize(viewPos);
    color.rgb = calculateAdvancedLighting(color.rgb, normal, lightDir, viewDir, worldPos);
    
    vec3 enhancedLightmap = pow(lightmapColor, vec3(0.6)) * 1.2;
    color.rgb *= enhancedLightmap;
    
    float torchLight = max(lmcoord.x - 0.5, 0.0) * 2.0;
    color.rgb += vec3(1.0, 0.6, 0.3) * torchLight * 0.4;
    
    float distance = length(viewPos);
    float fogFactor = exp(-distance * 0.01);
    
    vec3 fogColor = dayNightTint * 0.6;
    if (rainStrength > 0.0) {
        fogColor = mix(fogColor, vec3(0.7, 0.7, 0.8), rainStrength);
    }
    
    color.rgb = mix(fogColor, color.rgb, fogFactor);
    
    if (rainStrength > 0.0) {
        color.rgb *= 1.0 + wetness * 0.3;
        color.rgb = saturation(color.rgb, 1.0 - rainStrength * 0.3);
    }


    if (isEyeInWater == 1) {
        vec2 center = coord - 0.5;
        
        // Mavi ton
        color.rgb *= vec3(0.6, 0.75, 2.3);
        
        // Derinlik efekti
        float depthFade = exp(-distance * 0.025);
        color.rgb = mix(vec3(0.2, 0.4, 0.8), color.rgb, depthFade);
    }
    
    // Kontrast ve renk düzeltmeleri (daha parlak)
    color.rgb = contrast(color.rgb, 1.2);
    color.rgb = saturation(color.rgb, 1.3);
    
    color.rgb *= 1.2;
    
    color.rgb = pow(color.rgb, vec3(1.0));
    
    gl_FragColor = color;
}