package kr.toxicity.model.api.data.renderer;

import kr.toxicity.model.api.BetterModel;
import kr.toxicity.model.api.bone.BoneName;
import kr.toxicity.model.api.bone.BoneTag;
import kr.toxicity.model.api.data.blueprint.BlueprintChildren;
import kr.toxicity.model.api.data.blueprint.NamedBoundingBox;
import kr.toxicity.model.api.bone.BoneMovement;
import kr.toxicity.model.api.bone.RenderedBone;
import kr.toxicity.model.api.mount.MountController;
import kr.toxicity.model.api.mount.MountControllers;
import kr.toxicity.model.api.player.PlayerLimb;
import kr.toxicity.model.api.tracker.TrackerModifier;
import kr.toxicity.model.api.util.MathUtil;
import kr.toxicity.model.api.util.TransformedItemStack;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.entity.ItemDisplay;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemStack;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;
import org.joml.Vector3f;

import java.util.Map;
import java.util.stream.Collectors;

/**
 * A group of models.
 */
@RequiredArgsConstructor
public final class RendererGroup {

    @Getter
    private final BoneName name;
    @Getter
    private final BlueprintChildren.BlueprintGroup parent;
    @Getter
    private final Vector3f position;
    private final Vector3f rotation;
    private final TransformedItemStack itemStack;
    @Getter
    @Unmodifiable
    private final Map<BoneName, RendererGroup> children;
    @Getter
    private final @Nullable NamedBoundingBox hitBox;

    @Getter
    private final Vector3f center;
    @Getter
    private final @Nullable PlayerLimb limb;

    @Getter
    private final @NotNull MountController mountController;

    /**
     * Creates group instance.
     * @param name name
     * @param scale scale
     * @param itemStack item
     * @param group parent
     * @param children children
     * @param box hit-box
     * @param limb player limb
     */
    public RendererGroup(
            @NotNull BoneName name,
            float scale,
            @Nullable ItemStack itemStack,
            @NotNull BlueprintChildren.BlueprintGroup group,
            @NotNull Map<BoneName, RendererGroup> children,
            @Nullable NamedBoundingBox box,
            @Nullable PlayerLimb limb
    ) {
        this.name = name;
        this.limb = limb;
        this.parent = group;
        this.children = children;
        this.itemStack = new TransformedItemStack(
                new Vector3f(),
                new Vector3f(scale),
                itemStack != null ? itemStack : new ItemStack(Material.AIR)
        );
        position = MathUtil.blockBenchToDisplay(group.origin().toVector()
                .div(16));
        this.hitBox = box;
        rotation = group.rotation().toVector();
        center = hitBox != null ? hitBox.centerPoint() : new Vector3f();
        if (name.tagged(BoneTag.SEAT)) {
            mountController = BetterModel.inst().configManager().defaultMountController();
        } else if (name.tagged(BoneTag.SUB_SEAT)) {
            mountController = MountControllers.NONE;
        } else mountController = MountControllers.INVALID;
    }

    /**
     * Creates entity.
     * @param player player
     * @param location location
     * @return entity
     */
    public @NotNull RenderedBone create(@Nullable Player player, @NotNull TrackerModifier modifier, @NotNull Location location) {
        return create(player, modifier, null, location);
    }
    private @NotNull RenderedBone create(@Nullable Player player, @NotNull TrackerModifier modifier, @Nullable RenderedBone entityParent, @NotNull Location location) {
        return new RenderedBone(
                this,
                entityParent,
                getItem(player),
                limb != null ? limb.getTransform() : ItemDisplay.ItemDisplayTransform.FIXED,
                location,
                new BoneMovement(
                        entityParent != null ? new Vector3f(position).sub(entityParent.getGroup().position) : new Vector3f(),
                        new Vector3f(1),
                        MathUtil.toQuaternion(MathUtil.blockBenchToDisplay(rotation)),
                        rotation
                ),
                modifier,
                parent1 -> children.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().create(player, modifier, parent1, location)))
        );
    }

    @NotNull
    private TransformedItemStack getItem(@Nullable Player player) {
        if (player != null) {
            return limb != null ? limb.createItem(player) : itemStack.asAir();
        }
        return itemStack;
    }

    /**
     * Gets display item.
     * @return item
     */
    public @NotNull TransformedItemStack getItemStack() {
        return itemStack.copy();
    }
}
