package kr.toxicity.model.api;

import kr.toxicity.model.api.data.renderer.ModelRenderer;
import kr.toxicity.model.api.event.PluginEndReloadEvent;
import kr.toxicity.model.api.event.PluginStartReloadEvent;
import kr.toxicity.model.api.tracker.EntityTracker;
import kr.toxicity.model.api.util.EventUtil;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Unmodifiable;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import static kr.toxicity.model.api.util.ReflectionUtil.*;

/**
 * A dummy class for BetterModel plugin instance.
 */
public final class BetterModel {

    /**
     * Private initializer
     */
    private BetterModel() {
        throw new RuntimeException();
    }

    /**
     * Check a running platform is Folia.
     */
    public static final boolean IS_FOLIA = classExists("io.papermc.paper.threadedregions.RegionizedServer");
    /**
     * Check a running platform is Purpur.
     */
    public static final boolean IS_PURPUR = classExists("org.purpurmc.purpur.PurpurConfig");
    /**
     * Check a running platform is Paper.
     */
    public static final boolean IS_PAPER = IS_PURPUR || IS_FOLIA || classExists("io.papermc.paper.configuration.PaperConfigurations");

    /**
     * Plugin instance.
     */
    private static BetterModelPlugin instance;

    public static @NotNull Optional<ModelRenderer> model(@NotNull String name) {
        return Optional.ofNullable(inst().modelManager().renderer(name));
    }
    public static @NotNull Optional<ModelRenderer> limb(@NotNull String name) {
        return Optional.ofNullable(inst().playerManager().limb(name));
    }
    public static @NotNull @Unmodifiable List<ModelRenderer> models() {
        return inst().modelManager().renderers();
    }
    public static @NotNull @Unmodifiable List<ModelRenderer> limbs() {
        return inst().playerManager().limbs();
    }

    /**
     * Gets plugin instance of BetterModel.
     * @see org.bukkit.plugin.java.JavaPlugin
     * @return instance
     */
    public static @NotNull BetterModelPlugin inst() {
        return Objects.requireNonNull(instance, "BetterModel hasn't been initialized yet!");
    }

    /**
     * Sets plugin instance of BetterModel.
     * @param instance instance
     */
    @ApiStatus.Internal
    public static void inst(@NotNull BetterModelPlugin instance) {
        if (BetterModel.instance != null) throw new RuntimeException();
        instance.addReloadStartHandler(() -> EventUtil.call(new PluginStartReloadEvent()));
        instance.addReloadEndHandler(t -> EntityTracker.reload());
        instance.addReloadEndHandler(t -> EventUtil.call(new PluginEndReloadEvent(t)));
        BetterModel.instance = Objects.requireNonNull(instance);
    }
}
