package com.provismet.provihealth.hud;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import com.provismet.provihealth.ProviHealthClient;
import com.provismet.provihealth.api.ProviHealthApi;
import com.provismet.provihealth.config.Options;
import com.provismet.provihealth.config.resources.EntityOptions;
import com.provismet.provihealth.config.resources.TagOptions;
import net.fabricmc.fabric.api.resource.SimpleSynchronousResourceReloadListener;
import net.minecraft.class_1299;
import net.minecraft.class_1309;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public class ElementRegistry implements SimpleSynchronousResourceReloadListener {
    // Cached Elements
    private static final Map<class_1299<?>, class_2960> borderCache = new HashMap<>();
    private static final Map<class_1299<?>, class_1799> iconCache = new HashMap<>();
    private static final Map<class_1299<?>, class_2960> barCache = new HashMap<>();
    private static final Map<class_1299<?>, Options.HUDType> hudCache = new HashMap<>();
    private static final Map<class_1299<?>, EntityOptions> entityOptionCache = new HashMap<>(); // Preferred
    private static final Map<class_6862<class_1299<?>>, TagOptions> tagOptionsCache = new HashMap<>(); // Used to feed the others data

    // Prioritised HUD elements from dependent mods using the API
    private static final Map<class_6862<class_1299<?>>, BorderPriority> tagBorderPriorities = new HashMap<>();
    private static final Map<class_6862<class_1299<?>>, ItemPriority> tagIconPriorities = new HashMap<>();
    private static final Map<class_1299<?>, BorderPriority> typeBorderPriorities = new HashMap<>();
    private static final Map<class_1299<?>, ItemPriority> typeIconPriorities = new HashMap<>();

    private static final List<TitlePriority> orderedTitles = new ArrayList<>();

    public static final class_2960 DEFAULT_BORDER = ProviHealthClient.identifier("textures/gui/healthbars/default.png");
    public static final class_2960 DEFAULT_BARS = ProviHealthClient.identifier("textures/gui/healthbars/bars.png");

    @Override
    public class_2960 getFabricId () {
        return ProviHealthClient.identifier("asset_listener");
    }

    @Override
    public void method_14491 (class_3300 manager) {
        entityOptionCache.clear();
        tagOptionsCache.clear();
        borderCache.clear();
        iconCache.clear();

        class_7923.field_41177.method_29722().forEach(entry -> {
            class_2960 resourceLocation = entry.getKey().method_29177().method_45138(ProviHealthClient.MODID + "/entity/").method_48331(".json");
            Optional<class_3298> resource = manager.method_14486(resourceLocation);
            if (resource.isPresent()) {
                try (InputStream stream = resource.get().method_14482()) {
                    String text = new String(stream.readAllBytes(), StandardCharsets.UTF_8);
                    DataResult<Pair<EntityOptions, JsonElement>> dataResult = EntityOptions.CODEC.decode(JsonOps.INSTANCE, JsonParser.parseString(text));
                    EntityOptions resolvedOptions = dataResult.getOrThrow().getFirst();
                    entityOptionCache.put(entry.getValue(), resolvedOptions);
                    ProviHealthClient.LOGGER.info("Found entity option with ID: {}", resourceLocation);
                }
                catch (Throwable e) {
                    ProviHealthClient.LOGGER.error("ProviHealth encountered an error reading file {} from pack {}", resourceLocation, resource.get().method_14480(), e);
                }
            }
        });

        Map<class_2960, class_3298> tagOptions = manager.method_14488(ProviHealthClient.MODID + "/tag", identifier -> identifier.method_12832().endsWith(".json"));
        for (Map.Entry<class_2960, class_3298> tagEntry : tagOptions.entrySet()) {
            String path = tagEntry.getKey().method_12832().replace(".json", "").replaceFirst("provihealth/tag/", "");
            class_2960 tagId = class_2960.method_60655(tagEntry.getKey().method_12836(), path);
            class_6862<class_1299<?>> tagKey = class_6862.method_40092(class_7924.field_41266, tagId);
            ProviHealthClient.LOGGER.info("Found tag options file {} for entity tag {}", tagEntry.getKey(), tagId);

            try (InputStream stream = tagEntry.getValue().method_14482()) {
                String text = new String(stream.readAllBytes(), StandardCharsets.UTF_8);
                DataResult<Pair<TagOptions, JsonElement>> dataResult = TagOptions.CODEC.decode(JsonOps.INSTANCE, JsonParser.parseString(text));
                TagOptions resolvedOptions = dataResult.getOrThrow().getFirst();
                tagOptionsCache.put(tagKey, resolvedOptions);
            }
            catch (Throwable e) {
                ProviHealthClient.LOGGER.error("ProviHealth encountered an error reading file {} from pack {}", tagEntry.getKey(), tagEntry.getValue().method_14480(), e);
            }
        }
    }

    public static boolean registerBorder (class_6862<class_1299<?>> entityTag, @Nullable class_2960 border, int priority) {
        if (entityTag == null) {
            ProviHealthClient.LOGGER.error("Attempted to register a null object to the border registry.");
            return false;
        }
        else if (tagBorderPriorities.containsKey(entityTag) && priority <= tagBorderPriorities.get(entityTag).priority()) {
            return false;
        }
        tagBorderPriorities.put(entityTag, new BorderPriority(border, priority));
        return true;
    }

    public static boolean registerItem (class_6862<class_1299<?>> entityTag, @Nullable class_1799 item, int priority) {
        if (entityTag == null) {
            ProviHealthClient.LOGGER.error("Attempted to register a null EntityGroup to the icon registry.");
            return false;
        }
        else if (tagIconPriorities.containsKey(entityTag) && priority <= tagIconPriorities.get(entityTag).priority()) {
            return false;
        }
        tagIconPriorities.put(entityTag, new ItemPriority(item, priority));
        return true;
    }

    public static boolean registerBorder (class_1299<?> type, @Nullable class_2960 border, int priority) {
        if (type == null) {
            ProviHealthClient.LOGGER.error("Attempted to register a null EntityType to the border registry.");
            return false;
        }
        else if (typeBorderPriorities.containsKey(type) && priority <= typeBorderPriorities.get(type).priority()) {
            return false;
        }
        typeBorderPriorities.put(type, new BorderPriority(border, priority));
        return true;
    }

    public static boolean registerItem (class_1299<?> type, @Nullable class_1799 item, int priority) {
        if (type == null) {
            ProviHealthClient.LOGGER.error("Attempted to register a null EntityType to the icon registry.");
            return false;
        }
        else if (typeIconPriorities.containsKey(type) && priority <= typeIconPriorities.get(type).priority()) {
            return false;
        }
        typeIconPriorities.put(type, new ItemPriority(item, priority));
        return true;
    }

    public static void registerTitle (ProviHealthApi.TitleGenerator titleGen, int order) {
        orderedTitles.add(new TitlePriority(titleGen, order));
    }

    public static void sortTitles () {
        orderedTitles.sort(Comparator.comparingInt(TitlePriority::order));
    }

    public static EntityOptions getEntityOptions (@Nullable class_1309 entity) {
        if (entity == null) return EntityOptions.DEFAULT;
        return entityOptionCache.getOrDefault(entity.method_5864(), EntityOptions.DEFAULT);
    }

    // Only used as a fallback when EntityOptions doesn't have it.
    @NotNull
    public static class_2960 getOrCacheBorder (@Nullable class_1309 entity) {
        if (entity == null || !Options.useCustomHudPortraits) return DEFAULT_BORDER;
        else {
            if (borderCache.containsKey(entity.method_5864())) return borderCache.get(entity.method_5864());

            int maxPriority = Integer.MIN_VALUE;
            class_2960 bestBorder = DEFAULT_BORDER;
            // Read from assets
            for (Map.Entry<class_6862<class_1299<?>>, TagOptions> entry : tagOptionsCache.entrySet()) {
                if (entity.method_5864().method_20210(entry.getKey()) && entry.getValue().getPriority() > maxPriority && entry.getValue().getBorder() != null) {
                    bestBorder = entry.getValue().getBorder();
                    maxPriority = entry.getValue().getPriority();
                }
            }

            // Read from mod addons
            for (class_6862<class_1299<?>> entityTag : tagBorderPriorities.keySet()) {
                if (entity.method_5864().method_20210(entityTag) && tagBorderPriorities.get(entityTag).priority() > maxPriority) {
                    bestBorder = tagBorderPriorities.get(entityTag).borderId();
                    maxPriority = tagBorderPriorities.get(entityTag).priority();
                }
            }

            for (class_1299<?> type : typeBorderPriorities.keySet()) {
                if (entity.method_5864() == type && typeBorderPriorities.get(type).priority() > maxPriority) {
                    bestBorder = typeBorderPriorities.get(type).borderId();
                    maxPriority = typeBorderPriorities.get(type).priority();
                }
            }

            borderCache.put(entity.method_5864(), bestBorder);
            return bestBorder;
        }
    }

    // Only used as a fallback when EntityOptions doesn't have it.
    @Nullable
    public static class_1799 getOrCacheIcon (class_1309 entity) {
        if (entity == null) return null;
        else if (iconCache.containsKey(entity.method_5864())) return iconCache.get(entity.method_5864());

        class_1799 bestIcon = null;
        int maxPriority = Integer.MIN_VALUE;

        // Read from assets
        for (Map.Entry<class_6862<class_1299<?>>, TagOptions> entry : tagOptionsCache.entrySet()) {
            if (entity.method_5864().method_20210(entry.getKey()) && entry.getValue().getPriority() > maxPriority && entry.getValue().getIcon() != null) {
                bestIcon = entry.getValue().getIcon();
                maxPriority = entry.getValue().getPriority();
            }
        }

        // Read from mod addons
        for (class_6862<class_1299<?>> entityTag : tagIconPriorities.keySet()) {
            if (entity.method_5864().method_20210(entityTag) && tagIconPriorities.get(entityTag).priority() > maxPriority) {
                bestIcon = tagIconPriorities.get(entityTag).itemStack();
                maxPriority = tagIconPriorities.get(entityTag).priority();
            }
        }

        for (class_1299<?> type : typeIconPriorities.keySet()) {
            if (entity.method_5864() == type && typeIconPriorities.get(type).priority() > maxPriority) {
                bestIcon = typeIconPriorities.get(type).itemStack();
                maxPriority = typeIconPriorities.get(type).priority();
            }
        }
        iconCache.put(entity.method_5864(), bestIcon);
        return bestIcon;
    }

    @NotNull
    public static class_2960 getOrCacheHealthBar (class_1309 entity) {
        if (entity == null) return DEFAULT_BARS;
        if (barCache.containsKey(entity.method_5864())) return barCache.get(entity.method_5864());

        class_2960 bestBars = DEFAULT_BARS;
        int maxPriority = Integer.MIN_VALUE;

        // Read from assets
        for (Map.Entry<class_6862<class_1299<?>>, TagOptions> entry : tagOptionsCache.entrySet()) {
            if (entity.method_5864().method_20210(entry.getKey()) && entry.getValue().getPriority() > maxPriority && entry.getValue().getHealthBar() != null) {
                bestBars = entry.getValue().getHealthBar();
                maxPriority = entry.getValue().getPriority();
            }
        }

        barCache.put(entity.method_5864(), bestBars);
        return bestBars;
    }

    @Nullable
    public static Options.HUDType getOrCacheHudType (class_1309 entity) {
        if (entity == null) return null;
        if (hudCache.containsKey(entity.method_5864())) return hudCache.get(entity.method_5864());

        Options.HUDType bestHud = null;
        int maxPriority = Integer.MIN_VALUE;

        // Read from assets
        for (Map.Entry<class_6862<class_1299<?>>, TagOptions> entry : tagOptionsCache.entrySet()) {
            if (entity.method_5864().method_20210(entry.getKey()) && entry.getValue().getPriority() > maxPriority && entry.getValue().getHudType() != null) {
                bestHud = entry.getValue().getHudType();
                maxPriority = entry.getValue().getPriority();
            }
        }

        hudCache.put(entity.method_5864(), bestHud);
        return bestHud;
    }

    public static List<class_2561> getTitle (class_1309 entity, boolean world, boolean hud) {
        if (entity == null) return null;

        return orderedTitles.stream().map(title -> title.titleGetter().apply(entity, world, hud)).filter(Objects::nonNull).toList();
    }

    private record ItemPriority (class_1799 itemStack, int priority) {}
    private record BorderPriority (class_2960 borderId, int priority) {}
    private record TitlePriority (ProviHealthApi.TitleGenerator titleGetter, int order) {}
}
