/*
 * Decompiled with CFR 0.152.
 */
package com.winss.dustlab.media;

import com.winss.dustlab.DustLab;
import com.winss.dustlab.media.AnimatedModel;
import com.winss.dustlab.media.FrameData;
import com.winss.dustlab.media.PixelToParticleMapper;
import com.winss.dustlab.packed.PackedParticleArray;
import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.metadata.IIOMetadataNode;
import javax.imageio.stream.ImageInputStream;
import org.bukkit.plugin.Plugin;

public class MediaProcessor {
    private static final int DEFAULT_FRAME_DELAY = 50;
    private static final int MIN_FRAME_DELAY = 40;
    private static final int MAX_FRAME_DELAY = 1000;
    private static final int DOWNLOAD_TIMEOUT = 20000;
    private static final ExecutorService MEDIA_PROCESSING_EXECUTOR = Executors.newFixedThreadPool(Math.max(2, Runtime.getRuntime().availableProcessors() / 2), new ThreadFactory(){
        private final AtomicInteger threadNumber = new AtomicInteger(1);

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r, "DustLab-MediaProcessor-" + this.threadNumber.getAndIncrement());
            t.setDaemon(false);
            t.setPriority(5);
            return t;
        }
    });
    private final Plugin plugin;

    public MediaProcessor(Plugin plugin) {
        this.plugin = plugin;
    }

    public static void shutdown() {
        if (!MEDIA_PROCESSING_EXECUTOR.isShutdown()) {
            MEDIA_PROCESSING_EXECUTOR.shutdown();
            try {
                while (!MEDIA_PROCESSING_EXECUTOR.awaitTermination(5L, TimeUnit.SECONDS)) {
                    Logger.getLogger(MediaProcessor.class.getName()).info("Waiting for media processing tasks to finish...");
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public static void submitAsync(Runnable task) {
        MEDIA_PROCESSING_EXECUTOR.submit(task);
    }

    public static Future<?> submitAsyncFuture(Runnable task) {
        return MEDIA_PROCESSING_EXECUTOR.submit(task);
    }

    public CompletableFuture<AnimatedModel> processMediaUrl(String url, String modelName, int blockWidth, int blockHeight, int maxParticleCount, boolean persistent) {
        return this.processMediaUrl(url, modelName, blockWidth, blockHeight, maxParticleCount, persistent, 1);
    }

    public CompletableFuture<AnimatedModel> processMediaUrl(String url, String modelName, int blockWidth, int blockHeight, int maxParticleCount, boolean persistent, int frameSkip) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                byte[] mediaData = this.downloadMedia(url);
                if (mediaData == null) {
                    throw new RuntimeException("Failed to download media from URL");
                }
                AnimatedModel result = this.processMediaData(mediaData, modelName, blockWidth, blockHeight, maxParticleCount, url, persistent, frameSkip);
                return result;
            }
            catch (Exception e) {
                this.plugin.getLogger().severe("Error in async processing: " + e.getMessage());
                e.printStackTrace();
                throw new RuntimeException("Media processing failed: " + e.getMessage());
            }
        }, MEDIA_PROCESSING_EXECUTOR);
    }

    private byte[] downloadMedia(String urlString) throws IOException {
        URL url = new URL(urlString);
        URLConnection connection = url.openConnection();
        connection.setConnectTimeout(20000);
        connection.setReadTimeout(20000);
        connection.setRequestProperty("User-Agent", "DustLab-MediaProcessor/1.0");
        int contentLength = connection.getContentLength();
        long maxBytes = ((DustLab)this.plugin).getDustLabConfig().getMediaMaxFileSizeBytes();
        if (contentLength > 0 && (long)contentLength > maxBytes) {
            throw new IOException("File too large: " + contentLength + " bytes (max: " + maxBytes + " bytes)");
        }
        try (InputStream inputStream = connection.getInputStream();){
            byte[] byArray;
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream(contentLength > 0 ? contentLength : 8192);){
                int bytesRead;
                byte[] buffer = new byte[16384];
                int totalSize = 0;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    if ((long)(totalSize += bytesRead) > maxBytes) {
                        throw new IOException("File too large during download (exceeded " + maxBytes + " bytes)");
                    }
                    baos.write(buffer, 0, bytesRead);
                }
                byArray = baos.toByteArray();
            }
            return byArray;
        }
    }

    private BufferedImage processFrameOptimized(BufferedImage frame, BufferedImage ignoredPreviousFrame, boolean isGif, int targetWidth, int targetHeight) {
        if (frame.getType() == 2 && frame.getWidth() == targetWidth && frame.getHeight() == targetHeight) {
            return frame;
        }
        if ((frame = this.ensureARGBFormat(frame)).getWidth() != targetWidth || frame.getHeight() != targetHeight) {
            BufferedImage resized = this.resizeImageFast(frame, targetWidth, targetHeight);
            if (frame != resized) {
                frame.flush();
            }
            return resized;
        }
        return frame;
    }

    private BufferedImage simpleFrameComposition(BufferedImage currentFrame, BufferedImage previousFrame) {
        BufferedImage result = new BufferedImage(currentFrame.getWidth(), currentFrame.getHeight(), 2);
        Graphics2D g = result.createGraphics();
        g.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
        if (previousFrame != null) {
            g.drawImage((Image)previousFrame, 0, 0, null);
        }
        g.setComposite(AlphaComposite.SrcOver);
        g.drawImage((Image)currentFrame, 0, 0, null);
        g.dispose();
        return result;
    }

    private BufferedImage resizeImageFast(BufferedImage original, int targetWidth, int targetHeight) {
        BufferedImage resized = new BufferedImage(targetWidth, targetHeight, 2);
        Graphics2D g = resized.createGraphics();
        g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_SPEED);
        g.setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
        g.drawImage(original, 0, 0, targetWidth, targetHeight, null);
        g.dispose();
        return resized;
    }

    private int getFrameDelayOptimized(ImageReader reader, int frameIndex, int frameSkip, int totalFrames) {
        try {
            int delay = this.getFrameDelaySimple(reader, frameIndex);
            if (frameSkip > 1) {
                int totalDelay = delay;
                for (int i = 1; i < frameSkip && frameIndex + i < totalFrames; ++i) {
                    try {
                        totalDelay += this.getFrameDelaySimple(reader, frameIndex + i);
                        continue;
                    }
                    catch (Exception e) {
                        totalDelay += delay;
                    }
                }
                delay = totalDelay;
            }
            int clamped = Math.max(40, Math.min(1000, delay));
            return this.quantizeFrameDelay(clamped);
        }
        catch (Exception e) {
            return 50;
        }
    }

    private int quantizeFrameDelay(int delayMs) {
        if (delayMs >= 45 && delayMs <= 55) {
            return 50;
        }
        return delayMs;
    }

    private AnimatedModel processMediaData(byte[] mediaData, String modelName, int blockWidth, int blockHeight, int maxParticleCount, String sourceUrl, boolean persistent, int frameSkip) throws IOException {
        ArrayList<FrameData> frames = new ArrayList<FrameData>();
        try (ImageInputStream imageInputStream = ImageIO.createImageInputStream(new ByteArrayInputStream(mediaData));){
            int numFrames;
            Iterator<ImageReader> readers = ImageIO.getImageReaders(imageInputStream);
            if (!readers.hasNext()) {
                throw new IOException("No suitable image reader found for this media format");
            }
            ImageReader reader = readers.next();
            reader.setInput(imageInputStream);
            try {
                numFrames = reader.getNumImages(true);
                if (numFrames <= 0) {
                    numFrames = 1;
                }
            }
            catch (Exception e) {
                numFrames = 1;
            }
            int actualOriginalFrames = numFrames;
            int maxFrames = ((DustLab)this.plugin).getDustLabConfig().getMediaMaxFrames();
            int targetFrameCount = Math.min(maxFrames, (actualOriginalFrames + frameSkip - 1) / frameSkip);
            boolean isGif = "gif".equalsIgnoreCase(reader.getFormatName());
            this.plugin.getLogger().info("Processing " + targetFrameCount + " frames from " + actualOriginalFrames + " total");
            int targetWidth = blockWidth * 16;
            int targetHeight = blockHeight * 16;
            int processedFrames = 0;
            BufferedImage previousFrame = null;
            for (int frameIndex = 0; frameIndex < numFrames && processedFrames < maxFrames; frameIndex += frameSkip) {
                BufferedImage frame;
                try {
                    frame = reader.read(frameIndex);
                }
                catch (Exception e) {
                    this.plugin.getLogger().warning("Could not read frame " + frameIndex + ", stopping at " + frames.size() + " frames");
                    break;
                }
                if (frame == null) {
                    this.plugin.getLogger().warning("Frame " + frameIndex + " is null, stopping processing");
                    break;
                }
                frame = this.processFrameOptimized(frame, null, isGif, targetWidth, targetHeight);
                int frameDelay = this.getFrameDelayOptimized(reader, frameIndex, frameSkip, numFrames);
                float particleScale = ((DustLab)this.plugin).getDustLabConfig().getMediaParticleScale();
                PackedParticleArray packedParticles = PixelToParticleMapper.imageToParticlesOptimized(frame, blockWidth, blockHeight, maxParticleCount, particleScale);
                frames.add(new FrameData(packedParticles, frameIndex, frameDelay));
                ++processedFrames;
                if (previousFrame != null && previousFrame != frame) {
                    previousFrame.flush();
                }
                previousFrame = frame;
                if (processedFrames == 1 || processedFrames == targetFrameCount || processedFrames % Math.max(1, targetFrameCount / 10) == 0) {
                    double progressPercent = (double)processedFrames / (double)targetFrameCount * 100.0;
                    String progressBar = this.createProgressBar(progressPercent);
                    this.plugin.getLogger().info(String.format("Processing frames: %s %.0f%% (%d/%d)", progressBar, progressPercent, processedFrames, targetFrameCount));
                }
                if (processedFrames % 10 != 0) continue;
                Thread.yield();
                if (processedFrames % 50 != 0 || targetFrameCount <= 100) continue;
                System.gc();
            }
            reader.dispose();
        }
        if (frames.isEmpty()) {
            throw new IOException("No frames could be processed from the media");
        }
        boolean looping = frames.size() > 1;
        AnimatedModel model = new AnimatedModel(modelName, frames, looping, sourceUrl, blockWidth, blockHeight, maxParticleCount);
        HashMap<String, Object> metadata = new HashMap<String, Object>();
        metadata.put("generatedBy", "DustLab v1.1");
        metadata.put("website", "https://winss.xyz/dustlab");
        metadata.put("generatedOn", Instant.now().toString());
        metadata.put("sourceFile", sourceUrl);
        metadata.put("particleCount", model.getTotalParticleCount());
        metadata.put("frameCount", frames.size());
        metadata.put("globalParticleSize", Double.valueOf(((DustLab)this.plugin).getDustLabConfig().getMediaParticleScale()));
        model.setMetadata(metadata);
        this.plugin.getLogger().info("Model creation complete! " + frames.size() + " frames, " + model.getTotalParticleCount() + " particles");
        return model;
    }

    private BufferedImage resizeImage(BufferedImage original, int targetWidth, int targetHeight) {
        double scaleX = (double)targetWidth / (double)original.getWidth();
        double scaleY = (double)targetHeight / (double)original.getHeight();
        double scale = Math.min(scaleX, scaleY);
        int newWidth = (int)((double)original.getWidth() * scale);
        int newHeight = (int)((double)original.getHeight() * scale);
        BufferedImage resized = new BufferedImage(newWidth, newHeight, 2);
        Graphics2D g2d = resized.createGraphics();
        g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
        g2d.drawImage(original, 0, 0, newWidth, newHeight, null);
        g2d.dispose();
        return resized;
    }

    private BufferedImage resizeImageHighQuality(BufferedImage original, int targetWidth, int targetHeight) {
        if (original == null) {
            return null;
        }
        double scaleX = (double)targetWidth / (double)original.getWidth();
        double scaleY = (double)targetHeight / (double)original.getHeight();
        double scale = Math.min(scaleX, scaleY);
        int newWidth = (int)((double)original.getWidth() * scale);
        int newHeight = (int)((double)original.getHeight() * scale);
        BufferedImage current = original;
        while (current.getWidth() > newWidth * 2 || current.getHeight() > newHeight * 2) {
            int tempWidth = Math.max(newWidth, current.getWidth() / 2);
            int tempHeight = Math.max(newHeight, current.getHeight() / 2);
            BufferedImage temp = new BufferedImage(tempWidth, tempHeight, 2);
            Graphics2D g2d = temp.createGraphics();
            g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
            g2d.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
            g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g2d.drawImage(current, 0, 0, tempWidth, tempHeight, null);
            g2d.dispose();
            current = temp;
        }
        if (current.getWidth() != newWidth || current.getHeight() != newHeight) {
            BufferedImage resized = new BufferedImage(newWidth, newHeight, 2);
            Graphics2D g2d = resized.createGraphics();
            g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
            g2d.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
            g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g2d.setRenderingHint(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
            g2d.drawImage(current, 0, 0, newWidth, newHeight, null);
            g2d.dispose();
            return resized;
        }
        return current;
    }

    private BufferedImage ensureARGBFormat(BufferedImage original) {
        if (original == null) {
            return null;
        }
        if (original.getType() == 2) {
            return this.fixTransparencyIssues(original);
        }
        BufferedImage argbImage = new BufferedImage(original.getWidth(), original.getHeight(), 2);
        Graphics2D g2d = argbImage.createGraphics();
        g2d.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
        if (original.getColorModel().hasAlpha()) {
            g2d.setComposite(AlphaComposite.Src);
        } else {
            g2d.setComposite(AlphaComposite.SrcOver);
            g2d.setColor(new Color(0, 0, 0, 0));
            g2d.fillRect(0, 0, argbImage.getWidth(), argbImage.getHeight());
        }
        g2d.drawImage((Image)original, 0, 0, null);
        g2d.dispose();
        return this.fixTransparencyIssues(argbImage);
    }

    private BufferedImage fixTransparencyIssues(BufferedImage image) {
        if (image == null) {
            return null;
        }
        BufferedImage fixed = new BufferedImage(image.getWidth(), image.getHeight(), 2);
        for (int y = 0; y < image.getHeight(); ++y) {
            for (int x = 0; x < image.getWidth(); ++x) {
                int rgb = image.getRGB(x, y);
                int alpha = rgb >> 24 & 0xFF;
                int red = rgb >> 16 & 0xFF;
                int green = rgb >> 8 & 0xFF;
                int blue = rgb & 0xFF;
                boolean shouldBeTransparent = false;
                if (red == 0 && green == 0 && blue == 0 && alpha < 128) {
                    shouldBeTransparent = true;
                }
                if (alpha < 25 && red < 20 && green < 20 && blue < 20) {
                    shouldBeTransparent = true;
                }
                if (alpha < 10) {
                    shouldBeTransparent = true;
                }
                if (shouldBeTransparent) {
                    fixed.setRGB(x, y, 0);
                    continue;
                }
                if (alpha > 0 && alpha < 50) {
                    alpha = Math.max(alpha, 100);
                    int newRgb = alpha << 24 | red << 16 | green << 8 | blue;
                    fixed.setRGB(x, y, newRgb);
                    continue;
                }
                fixed.setRGB(x, y, rgb);
            }
        }
        return fixed;
    }

    private int getFrameDelaySimple(ImageReader reader, int frameIndex) {
        try {
            String formatName;
            IIOMetadata metadata = reader.getImageMetadata(frameIndex);
            if (metadata != null && "gif".equalsIgnoreCase(formatName = reader.getFormatName())) {
                int delay = this.getGifFrameDelay(metadata);
                return delay;
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().warning("Could not read frame delay for frame " + frameIndex + ": " + e.getMessage());
        }
        return 50;
    }

    private int getGifFrameDelay(IIOMetadata metadata) {
        try {
            String[] formatNames;
            for (String formatName : formatNames = metadata.getMetadataFormatNames()) {
                String delayTime;
                IIOMetadataNode root;
                IIOMetadataNode graphicsControlExtension;
                if (!"javax_imageio_gif_image_1.0".equals(formatName) || (graphicsControlExtension = this.getChild(root = (IIOMetadataNode)metadata.getAsTree(formatName), "GraphicControlExtension")) == null || (delayTime = graphicsControlExtension.getAttribute("delayTime")) == null || delayTime.isEmpty()) continue;
                try {
                    int centiseconds = Integer.parseInt(delayTime);
                    int delay = centiseconds * 10;
                    if (delay <= 0) {
                        delay = 50;
                    } else if (delay < 20) {
                        delay = Math.max(delay, 40);
                    }
                    return delay;
                }
                catch (NumberFormatException e) {
                    this.plugin.getLogger().warning("Invalid delay time format: " + delayTime);
                }
            }
        }
        catch (Exception e) {
            this.plugin.getLogger().fine("Could not extract GIF frame delay: " + e.getMessage());
        }
        return 50;
    }

    private int getFrameDelay(ImageReader reader, int frameIndex) {
        return this.getFrameDelaySimple(reader, frameIndex);
    }

    private IIOMetadataNode getChild(IIOMetadataNode parent, String childName) {
        for (int i = 0; i < parent.getLength(); ++i) {
            if (!childName.equals(parent.item(i).getNodeName())) continue;
            return (IIOMetadataNode)parent.item(i);
        }
        return null;
    }

    private String createProgressBar(double percent) {
        int i;
        int totalBars = 10;
        double clamped = Math.max(0.0, Math.min(100.0, percent));
        int filledBars = (int)Math.round(clamped / 100.0 * 10.0);
        if (filledBars > 10) {
            filledBars = 10;
        }
        StringBuilder progressBar = new StringBuilder();
        progressBar.append("[");
        progressBar.append("\u001b[34m");
        for (i = 0; i < filledBars; ++i) {
            progressBar.append("\u2588");
        }
        progressBar.append("\u001b[0m");
        progressBar.append("\u001b[37m");
        for (i = filledBars; i < 10; ++i) {
            progressBar.append("\u2591");
        }
        progressBar.append("\u001b[0m");
        progressBar.append("]");
        return progressBar.toString();
    }
}

