/*
 * Decompiled with CFR 0.152.
 */
package com.github.ob_yekt.simpleskills.managers;

import com.github.ob_yekt.simpleskills.Simpleskills;
import com.github.ob_yekt.simpleskills.Skills;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.minecraft.class_5218;
import net.minecraft.server.MinecraftServer;

public class DatabaseManager {
    private static final String DATABASE_NAME = "simpleskills.db";
    private static final List<String> SKILLS = Arrays.stream(Skills.values()).map(Skills::getId).toList();
    private static DatabaseManager instance;
    private Connection connection;
    private Path currentDatabasePath;
    private static final Map<String, Map<String, SkillData>> skillCache;
    private static final Map<String, Boolean> ironmanCache;
    private static final Map<String, Integer> prestigeCache;
    private static final Map<String, Integer> totalLevelCache;

    private DatabaseManager() {
    }

    public static DatabaseManager getInstance() {
        if (instance == null) {
            instance = new DatabaseManager();
        }
        return instance;
    }

    public void initializeDatabase(MinecraftServer server) {
        Path worldDirectory = server.method_27050(class_5218.field_24188).resolve("data");
        Path newDatabasePath = worldDirectory.resolve(DATABASE_NAME);
        if (this.currentDatabasePath != null && this.currentDatabasePath.equals(newDatabasePath) && this.isConnectionValid()) {
            return;
        }
        this.closeConnection();
        try {
            Files.createDirectories(worldDirectory, new FileAttribute[0]);
            this.connection = DriverManager.getConnection("jdbc:sqlite:" + String.valueOf(newDatabasePath));
            this.connection.setAutoCommit(true);
            try (PreparedStatement pragmaStmt = this.connection.prepareStatement("PRAGMA journal_mode=WAL;");){
                pragmaStmt.execute();
            }
            this.currentDatabasePath = newDatabasePath;
            this.createTables();
            Simpleskills.LOGGER.info("Connected to SQLite database at: {}", (Object)newDatabasePath);
        }
        catch (IOException | SQLException e) {
            Simpleskills.LOGGER.error("Failed to initialize database at {}.", (Object)newDatabasePath, (Object)e);
            throw new DatabaseException("Database initialization failed", e);
        }
    }

    private boolean isConnectionValid() {
        try {
            return this.connection != null && !this.connection.isClosed() && this.connection.isValid(5);
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Error checking database connection validity: {}", (Object)e.getMessage());
            return false;
        }
    }

    private void createTables() {
        String createPlayersTable = "    CREATE TABLE IF NOT EXISTS players (\n        player_uuid TEXT PRIMARY KEY,\n        is_ironman INTEGER DEFAULT 0,\n        is_tab_menu_visible INTEGER DEFAULT 1,\n        prestige INTEGER DEFAULT 0\n    )\n";
        String createSkillsTable = "    CREATE TABLE IF NOT EXISTS player_skills (\n        player_uuid TEXT,\n        skill_id TEXT,\n        xp INTEGER DEFAULT 0,\n        level INTEGER DEFAULT 1,\n        PRIMARY KEY (player_uuid, skill_id)\n    )\n";
        String createPlayerNamesTable = "    CREATE TABLE IF NOT EXISTS player_names (\n        uuid TEXT,\n        name TEXT,\n        last_seen INTEGER,\n        PRIMARY KEY (uuid, last_seen)\n    )\n";
        String createIndex = "    CREATE INDEX IF NOT EXISTS idx_player_skills_uuid ON player_skills (player_uuid)\n";
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute(createPlayersTable);
            stmt.execute(createSkillsTable);
            stmt.execute(createPlayerNamesTable);
            stmt.execute(createIndex);
            try {
                stmt.execute("ALTER TABLE players ADD COLUMN prestige INTEGER DEFAULT 0");
            }
            catch (SQLException sQLException) {
                // empty catch block
            }
            Simpleskills.LOGGER.debug("Created database tables and indexes if they didn't exist.");
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to create database tables or indexes: {}", (Object)e.getMessage());
            throw new DatabaseException("Failed to create tables", e);
        }
    }

    public void initializePlayer(String playerUuid) {
        this.checkConnection();
        String insertPlayerSql = "INSERT OR IGNORE INTO players (player_uuid, is_ironman, is_tab_menu_visible, prestige) VALUES (?, 0, 1, 0)";
        String insertSkillSql = "INSERT OR IGNORE INTO player_skills (player_uuid, skill_id, xp, level) VALUES (?, ?, 0, 1)";
        try {
            this.connection.setAutoCommit(false);
            try (PreparedStatement playerStmt = this.connection.prepareStatement(insertPlayerSql);){
                playerStmt.setString(1, playerUuid);
                playerStmt.executeUpdate();
            }
            try (PreparedStatement skillStmt = this.connection.prepareStatement(insertSkillSql);){
                for (String skill : SKILLS) {
                    skillStmt.setString(1, playerUuid);
                    skillStmt.setString(2, skill);
                    skillStmt.executeUpdate();
                }
            }
            this.connection.commit();
            HashMap<String, SkillData> defaultSkills = new HashMap<String, SkillData>();
            for (String skill : SKILLS) {
                defaultSkills.put(skill, new SkillData(0, 1));
            }
            skillCache.put(playerUuid, defaultSkills);
            ironmanCache.put(playerUuid, false);
            prestigeCache.put(playerUuid, 0);
            totalLevelCache.put(playerUuid, SKILLS.size());
            Simpleskills.LOGGER.debug("Initialized player data for UUID: {}", (Object)playerUuid);
        }
        catch (SQLException e) {
            try {
                this.connection.rollback();
            }
            catch (SQLException rollbackEx) {
                Simpleskills.LOGGER.error("Failed to rollback transaction: {}", (Object)rollbackEx.getMessage());
            }
            Simpleskills.LOGGER.error("Failed to initialize player UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to initialize player", e);
        }
        finally {
            try {
                this.connection.setAutoCommit(true);
            }
            catch (SQLException e) {
                Simpleskills.LOGGER.error("Failed to restore auto-commit: {}", (Object)e.getMessage());
            }
        }
    }

    public void ensurePlayerInitialized(String playerUuid) {
        this.checkConnection();
        String sql = "SELECT COUNT(*) FROM players WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, playerUuid);
            try (ResultSet result = statement.executeQuery();){
                if (result.next() && result.getInt(1) == 0) {
                    this.initializePlayer(playerUuid);
                }
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to check player initialization for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to check player initialization", e);
        }
    }

    public void updatePlayerName(String playerUuid, String playerName) {
        this.checkConnection();
        String sql = "INSERT OR REPLACE INTO player_names (uuid, name, last_seen) VALUES (?, ?, ?)";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, playerUuid);
            statement.setString(2, playerName);
            statement.setLong(3, System.currentTimeMillis() / 1000L);
            statement.executeUpdate();
            Simpleskills.LOGGER.debug("Updated player name for UUID {}: {}", (Object)playerUuid, (Object)playerName);
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to update player name for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to update player name", e);
        }
    }

    public Map<String, SkillData> getAllSkills(String playerUuid) {
        Map<String, SkillData> map;
        block17: {
            this.checkConnection();
            if (skillCache.containsKey(playerUuid)) {
                return Collections.unmodifiableMap(skillCache.get(playerUuid));
            }
            HashMap<String, SkillData> skills = new HashMap<String, SkillData>();
            String sql = "SELECT skill_id, xp, level FROM player_skills WHERE player_uuid = ?";
            PreparedStatement statement = this.connection.prepareStatement(sql);
            try {
                statement.setString(1, playerUuid);
                try (ResultSet result = statement.executeQuery();){
                    while (result.next()) {
                        String skillId = result.getString("skill_id");
                        int xp = result.getInt("xp");
                        int level = result.getInt("level");
                        skills.put(skillId, new SkillData(xp, level));
                    }
                }
                skillCache.put(playerUuid, skills);
                map = Collections.unmodifiableMap(skills);
                if (statement == null) break block17;
            }
            catch (Throwable throwable) {
                try {
                    if (statement != null) {
                        try {
                            statement.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (SQLException e) {
                    Simpleskills.LOGGER.error("Failed to retrieve skills for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
                    throw new DatabaseException("Failed to retrieve skills", e);
                }
            }
            statement.close();
        }
        return map;
    }

    public void savePlayerSkill(String playerUuid, String skillId, int xp, int level) {
        this.checkConnection();
        String sql = "INSERT OR REPLACE INTO player_skills (player_uuid, skill_id, xp, level) VALUES (?, ?, ?, ?)";
        try {
            this.connection.setAutoCommit(false);
            try (PreparedStatement statement = this.connection.prepareStatement(sql);){
                statement.setString(1, playerUuid);
                statement.setString(2, skillId);
                statement.setInt(3, xp);
                statement.setInt(4, level);
                statement.executeUpdate();
            }
            this.connection.commit();
            if (skillCache.containsKey(playerUuid)) {
                skillCache.get(playerUuid).put(skillId, new SkillData(xp, level));
                int total = 0;
                for (SkillData data : skillCache.get(playerUuid).values()) {
                    total += data.level();
                }
                totalLevelCache.put(playerUuid, total);
            } else {
                HashMap<String, SkillData> skills = new HashMap<String, SkillData>();
                skills.put(skillId, new SkillData(xp, level));
                skillCache.put(playerUuid, skills);
            }
            Simpleskills.LOGGER.debug("Saved skill {} for UUID {}: {} XP, level {}", new Object[]{skillId, playerUuid, xp, level});
        }
        catch (SQLException e) {
            try {
                this.connection.rollback();
            }
            catch (SQLException rollbackEx) {
                Simpleskills.LOGGER.error("Failed to rollback transaction: {}", (Object)rollbackEx.getMessage());
            }
            Simpleskills.LOGGER.error("Failed to save skill {} for UUID {}: {}", new Object[]{skillId, playerUuid, e.getMessage()});
            throw new DatabaseException("Failed to save skill data", e);
        }
        finally {
            try {
                this.connection.setAutoCommit(true);
            }
            catch (SQLException e) {
                Simpleskills.LOGGER.error("Failed to restore auto-commit: {}", (Object)e.getMessage());
            }
        }
    }

    public void resetPlayerSkills(String playerUuid) {
        this.checkConnection();
        String sql = "UPDATE player_skills SET xp = 0, level = 1 WHERE player_uuid = ?";
        try {
            this.connection.setAutoCommit(false);
            try (PreparedStatement statement = this.connection.prepareStatement(sql);){
                statement.setString(1, playerUuid);
                statement.executeUpdate();
            }
            this.connection.commit();
            if (skillCache.containsKey(playerUuid)) {
                Map<String, SkillData> cachedSkills = skillCache.get(playerUuid);
                for (String skill : SKILLS) {
                    cachedSkills.put(skill, new SkillData(0, 1));
                }
                totalLevelCache.put(playerUuid, SKILLS.size());
            }
            Simpleskills.LOGGER.debug("Reset skills for UUID: {}", (Object)playerUuid);
        }
        catch (SQLException e) {
            try {
                this.connection.rollback();
            }
            catch (SQLException rollbackEx) {
                Simpleskills.LOGGER.error("Failed to rollback transaction: {}", (Object)rollbackEx.getMessage());
            }
            Simpleskills.LOGGER.error("Failed to reset skills for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to reset skills", e);
        }
        finally {
            try {
                this.connection.setAutoCommit(true);
            }
            catch (SQLException e) {
                Simpleskills.LOGGER.error("Failed to restore auto-commit: {}", (Object)e.getMessage());
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getPrestige(String playerUuid) {
        this.checkConnection();
        if (prestigeCache.containsKey(playerUuid)) {
            return prestigeCache.get(playerUuid);
        }
        String sql = "SELECT prestige FROM players WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, playerUuid);
            try (ResultSet result = statement.executeQuery();){
                if (!result.next()) return 0;
                int prestige = result.getInt("prestige");
                prestigeCache.put(playerUuid, prestige);
                int n = prestige;
                return n;
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to get prestige for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to get prestige", e);
        }
    }

    public void setPrestige(String playerUuid, int prestige) {
        this.checkConnection();
        String sql = "UPDATE players SET prestige = ? WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setInt(1, prestige);
            statement.setString(2, playerUuid);
            statement.executeUpdate();
            prestigeCache.put(playerUuid, prestige);
            Simpleskills.LOGGER.debug("Set prestige to {} for UUID: {}", (Object)prestige, (Object)playerUuid);
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to set prestige for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to set prestige", e);
        }
    }

    public void incrementPrestige(String playerUuid) {
        this.setPrestige(playerUuid, this.getPrestige(playerUuid) + 1);
    }

    public void setIronmanMode(String playerUuid, boolean isIronman) {
        this.checkConnection();
        String sql = "UPDATE players SET is_ironman = ? WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setInt(1, isIronman ? 1 : 0);
            statement.setString(2, playerUuid);
            statement.executeUpdate();
            ironmanCache.put(playerUuid, isIronman);
            Simpleskills.LOGGER.debug("Set Ironman mode to {} for UUID: {}", (Object)isIronman, (Object)playerUuid);
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to set Ironman mode for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to set Ironman mode", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean isPlayerInIronmanMode(String playerUuid) {
        this.checkConnection();
        if (ironmanCache.containsKey(playerUuid)) {
            return ironmanCache.get(playerUuid);
        }
        String sql = "SELECT is_ironman FROM players WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, playerUuid);
            try (ResultSet result = statement.executeQuery();){
                if (!result.next()) return false;
                boolean isIronman = result.getInt("is_ironman") == 1;
                ironmanCache.put(playerUuid, isIronman);
                boolean bl = isIronman;
                return bl;
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to check Ironman mode for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to check Ironman mode", e);
        }
    }

    public void setTabMenuVisibility(String playerUuid, boolean isVisible) {
        this.checkConnection();
        String sql = "UPDATE players SET is_tab_menu_visible = ? WHERE player_uuid = ?";
        try {
            this.connection.setAutoCommit(false);
            try (PreparedStatement statement = this.connection.prepareStatement(sql);){
                statement.setInt(1, isVisible ? 1 : 0);
                statement.setString(2, playerUuid);
                statement.executeUpdate();
                Simpleskills.LOGGER.debug("Set tab menu visibility to {} for UUID: {}", (Object)isVisible, (Object)playerUuid);
            }
            this.connection.commit();
        }
        catch (SQLException e) {
            try {
                this.connection.rollback();
            }
            catch (SQLException rollbackEx) {
                Simpleskills.LOGGER.error("Failed to rollback transaction: {}", (Object)rollbackEx.getMessage());
            }
            Simpleskills.LOGGER.error("Failed to set tab menu visibility for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to set tab menu visibility", e);
        }
        finally {
            try {
                this.connection.setAutoCommit(true);
            }
            catch (SQLException e) {
                Simpleskills.LOGGER.error("Failed to restore auto-commit: {}", (Object)e.getMessage());
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean isTabMenuVisible(String playerUuid) {
        this.checkConnection();
        String sql = "SELECT is_tab_menu_visible FROM players WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, playerUuid);
            try (ResultSet result = statement.executeQuery();){
                if (!result.next()) return true;
                boolean bl = result.getInt("is_tab_menu_visible") == 1;
                return bl;
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to check tab menu visibility for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to check tab menu visibility", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int getTotalSkillLevel(String playerUuid) {
        this.checkConnection();
        if (totalLevelCache.containsKey(playerUuid)) {
            return totalLevelCache.get(playerUuid);
        }
        String sql = "SELECT SUM(level) as total_level FROM player_skills WHERE player_uuid = ?";
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, playerUuid);
            try (ResultSet result = statement.executeQuery();){
                if (!result.next()) return 0;
                int total = result.getInt("total_level");
                totalLevelCache.put(playerUuid, total);
                int n = total;
                return n;
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to get total skill level for UUID {}: {}", (Object)playerUuid, (Object)e.getMessage());
            throw new DatabaseException("Failed to get total skill level", e);
        }
    }

    public List<LeaderboardEntry> getSkillLeaderboard(String skillId, int limit) {
        this.checkConnection();
        String sql = "    SELECT p.player_uuid, ps.level, ps.xp, COALESCE(n.player_name, p.player_uuid) as player_name, p.prestige\n    FROM player_skills ps\n    JOIN players p ON ps.player_uuid = p.player_uuid\n    LEFT JOIN (\n        SELECT uuid, name as player_name\n        FROM player_names\n        WHERE (uuid, last_seen) IN (\n            SELECT uuid, MAX(last_seen)\n            FROM player_names\n            GROUP BY uuid\n        )\n    ) n ON p.player_uuid = n.uuid\n    WHERE ps.skill_id = ?\n    ORDER BY p.prestige DESC, ps.level DESC, ps.xp DESC\n    LIMIT ?\n";
        ArrayList<LeaderboardEntry> leaderboard = new ArrayList<LeaderboardEntry>();
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, skillId);
            statement.setInt(2, limit);
            try (ResultSet result = statement.executeQuery();){
                while (result.next()) {
                    String playerUuid = result.getString("player_uuid");
                    String playerName = result.getString("player_name");
                    int level = result.getInt("level");
                    int xp = result.getInt("xp");
                    int prestige = result.getInt("prestige");
                    leaderboard.add(new LeaderboardEntry(playerUuid, playerName, level, xp, prestige));
                }
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to retrieve leaderboard for skill {}: {}", (Object)skillId, (Object)e.getMessage());
            throw new DatabaseException("Failed to retrieve skill leaderboard", e);
        }
        return leaderboard;
    }

    public List<LeaderboardEntry> getTotalLevelLeaderboard(int limit) {
        this.checkConnection();
        String sql = "    SELECT p.player_uuid, SUM(ps.level) as total_level, COALESCE(n.player_name, p.player_uuid) as player_name, p.prestige\n    FROM player_skills ps\n    JOIN players p ON ps.player_uuid = p.player_uuid\n    LEFT JOIN (\n        SELECT uuid, name as player_name\n        FROM player_names\n        WHERE (uuid, last_seen) IN (\n            SELECT uuid, MAX(last_seen)\n            FROM player_names\n            GROUP BY uuid\n        )\n    ) n ON p.player_uuid = n.uuid\n    GROUP BY p.player_uuid\n    ORDER BY p.prestige DESC, total_level DESC\n    LIMIT ?\n";
        ArrayList<LeaderboardEntry> leaderboard = new ArrayList<LeaderboardEntry>();
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setInt(1, limit);
            try (ResultSet result = statement.executeQuery();){
                while (result.next()) {
                    String playerUuid = result.getString("player_uuid");
                    String playerName = result.getString("player_name");
                    int totalLevel = result.getInt("total_level");
                    int prestige = result.getInt("prestige");
                    leaderboard.add(new LeaderboardEntry(playerUuid, playerName, totalLevel, 0, prestige));
                }
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to retrieve total level leaderboard: {}", (Object)e.getMessage());
            throw new DatabaseException("Failed to retrieve total level leaderboard", e);
        }
        return leaderboard;
    }

    public List<LeaderboardEntry> getIronmanSkillLeaderboard(String skillId, int limit) {
        this.checkConnection();
        String sql = "    SELECT p.player_uuid, ps.level, ps.xp, COALESCE(n.player_name, p.player_uuid) as player_name, p.prestige\n    FROM player_skills ps\n    JOIN players p ON ps.player_uuid = p.player_uuid\n    LEFT JOIN (\n        SELECT uuid, name as player_name\n        FROM player_names\n        WHERE (uuid, last_seen) IN (\n            SELECT uuid, MAX(last_seen)\n            FROM player_names\n            GROUP BY uuid\n        )\n    ) n ON p.player_uuid = n.uuid\n    WHERE ps.skill_id = ? AND p.is_ironman = 1\n    ORDER BY p.prestige DESC, ps.level DESC, ps.xp DESC\n    LIMIT ?\n";
        ArrayList<LeaderboardEntry> leaderboard = new ArrayList<LeaderboardEntry>();
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setString(1, skillId);
            statement.setInt(2, limit);
            try (ResultSet result = statement.executeQuery();){
                while (result.next()) {
                    String playerUuid = result.getString("player_uuid");
                    String playerName = result.getString("player_name");
                    int level = result.getInt("level");
                    int xp = result.getInt("xp");
                    int prestige = result.getInt("prestige");
                    leaderboard.add(new LeaderboardEntry(playerUuid, playerName, level, xp, prestige));
                }
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to retrieve Ironman leaderboard for skill {}: {}", (Object)skillId, (Object)e.getMessage());
            throw new DatabaseException("Failed to retrieve Ironman skill leaderboard", e);
        }
        return leaderboard;
    }

    public List<LeaderboardEntry> getIronmanTotalLevelLeaderboard(int limit) {
        this.checkConnection();
        String sql = "    SELECT p.player_uuid, SUM(ps.level) as total_level, COALESCE(n.player_name, p.player_uuid) as player_name, p.prestige\n    FROM player_skills ps\n    JOIN players p ON ps.player_uuid = p.player_uuid\n    LEFT JOIN (\n        SELECT uuid, name as player_name\n        FROM player_names\n        WHERE (uuid, last_seen) IN (\n            SELECT uuid, MAX(last_seen)\n            FROM player_names\n            GROUP BY uuid\n        )\n    ) n ON p.player_uuid = n.uuid\n    WHERE p.is_ironman = 1\n    GROUP BY p.player_uuid\n    ORDER BY p.prestige DESC, total_level DESC\n    LIMIT ?\n";
        ArrayList<LeaderboardEntry> leaderboard = new ArrayList<LeaderboardEntry>();
        try (PreparedStatement statement = this.connection.prepareStatement(sql);){
            statement.setInt(1, limit);
            try (ResultSet result = statement.executeQuery();){
                while (result.next()) {
                    String playerUuid = result.getString("player_uuid");
                    String playerName = result.getString("player_name");
                    int totalLevel = result.getInt("total_level");
                    int prestige = result.getInt("prestige");
                    leaderboard.add(new LeaderboardEntry(playerUuid, playerName, totalLevel, 0, prestige));
                }
            }
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to retrieve Ironman total level leaderboard: {}", (Object)e.getMessage());
            throw new DatabaseException("Failed to retrieve Ironman total level leaderboard", e);
        }
        return leaderboard;
    }

    private void reconnectIfNeeded() {
        if (!this.isConnectionValid() && this.currentDatabasePath != null) {
            Simpleskills.LOGGER.info("Reconnecting to database at {}", (Object)this.currentDatabasePath);
            this.closeConnection();
            try {
                this.connection = DriverManager.getConnection("jdbc:sqlite:" + String.valueOf(this.currentDatabasePath));
                this.connection.setAutoCommit(true);
                try (PreparedStatement pragmaStmt = this.connection.prepareStatement("PRAGMA journal_mode=WAL;");){
                    pragmaStmt.execute();
                }
                this.createTables();
                Simpleskills.LOGGER.info("Successfully reconnected to the database.");
            }
            catch (SQLException e) {
                Simpleskills.LOGGER.error("Failed to reconnect to the database: {}", (Object)e.getMessage());
                throw new DatabaseException("Database reconnection failed", e);
            }
        }
    }

    private void checkConnection() {
        this.reconnectIfNeeded();
        if (this.connection == null) {
            Simpleskills.LOGGER.error("No database connection available.");
            throw new DatabaseException("No database connection available", null);
        }
    }

    private void closeConnection() {
        if (this.connection != null) {
            try {
                this.connection.close();
                Simpleskills.LOGGER.debug("Database connection closed.");
            }
            catch (SQLException e) {
                Simpleskills.LOGGER.error("Failed to close database connection: {}", (Object)e.getMessage());
            }
        }
    }

    public void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                try (PreparedStatement optimizeStmt = this.connection.prepareStatement("PRAGMA optimize;");){
                    optimizeStmt.execute();
                    Simpleskills.LOGGER.debug("Ran PRAGMA optimize on database.");
                }
            }
            this.closeConnection();
            this.currentDatabasePath = null;
            skillCache.clear();
            ironmanCache.clear();
            totalLevelCache.clear();
        }
        catch (SQLException e) {
            Simpleskills.LOGGER.error("Failed to optimize or close database: {}", (Object)e.getMessage());
        }
    }

    static {
        skillCache = new HashMap<String, Map<String, SkillData>>();
        ironmanCache = new HashMap<String, Boolean>();
        prestigeCache = new HashMap<String, Integer>();
        totalLevelCache = new HashMap<String, Integer>();
    }

    public static class DatabaseException
    extends RuntimeException {
        public DatabaseException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public record SkillData(int xp, int level) {
    }

    public record LeaderboardEntry(String playerUuid, String playerName, int level, int xp, int prestige) {
    }
}

