package in.northwestw.shortcircuit.registries.blockentities;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import in.northwestw.shortcircuit.Constants;
import in.northwestw.shortcircuit.config.Config;
import in.northwestw.shortcircuit.data.CircuitSavedData;
import in.northwestw.shortcircuit.data.Octolet;
import in.northwestw.shortcircuit.properties.DirectionHelper;
import in.northwestw.shortcircuit.properties.RelativeDirection;
import in.northwestw.shortcircuit.registries.BlockEntities;
import in.northwestw.shortcircuit.registries.Blocks;
import in.northwestw.shortcircuit.registries.blockentities.common.CommonCircuitBlockEntity;
import in.northwestw.shortcircuit.registries.blocks.CircuitBlock;
import in.northwestw.shortcircuit.registries.blocks.CircuitBoardBlock;
import net.minecraft.class_1923;
import net.minecraft.class_2248;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2383;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2512;
import net.minecraft.class_2520;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_3218;
import net.minecraft.class_7924;
import net.minecraft.server.MinecraftServer;
import org.apache.commons.lang3.tuple.Pair;

import java.util.*;

public class CircuitBlockEntity extends CommonCircuitBlockEntity {
    private static final long MAX_TAG_BYTE_SIZE = 2097152L; // copied from NbtAccounter
    private UUID runtimeUuid, ownerUuid;
    private short blockSize, ticks;
    private boolean fake;
    private byte[] powers, inputs;
    public Map<class_2338, class_2680> blocks; // 8x8x8
    // chunking
    private long chunkedOffset;
    private boolean chunked;

    public CircuitBlockEntity(class_2338 pos, class_2680 state) {
        super(BlockEntities.CIRCUIT.get(), pos, state);
        this.blocks = Maps.newTreeMap();
        this.runtimeUuid = UUID.randomUUID();
        this.powers = new byte[6];
        this.inputs = new byte[6];
    }

    @Override
    public void tick() {
        super.tick();
        if (this.shouldTick())
            this.updateInnerBlocks();
        // This should trigger getUpdateTag, which will figure out if inner blocks are chunked
        if (this.chunked && this.chunkedOffset != 0)
            this.field_11863.method_8413(this.method_11016(), this.method_11010(), this.method_11010(), class_2248.field_31028);
    }

    public boolean shouldTick() {
        this.ticks = (short) ((this.ticks + 1) % 100);
        return this.ticks == 1;
    }

    public void updateInnerBlocks() {
        if (this.hidden) return;
        MinecraftServer server = this.field_11863.method_8503();
        if (server == null) return;
        class_3218 runtimeLevel = this.field_11863.method_8503().method_3847(Constants.RUNTIME_DIMENSION);
        if (runtimeLevel == null) return;
        CircuitSavedData data = CircuitSavedData.getRuntimeData(runtimeLevel);
        Octolet octolet = data.getParentOctolet(this.runtimeUuid);
        if (octolet == null) return;
        class_2338 startingPos = data.getCircuitStartingPos(this.runtimeUuid);
        this.blocks.clear();
        for (int ii = 1; ii < octolet.blockSize - 1; ii++) {
            for (int jj = 1; jj < octolet.blockSize - 1; jj++) {
                for (int kk = 1; kk < octolet.blockSize - 1; kk++) {
                    class_2680 blockState = runtimeLevel.method_8320(startingPos.method_10069(ii, jj, kk));
                    if (!blockState.method_26215()) {
                        // ShortCircuit.LOGGER.debug("{} at {}, {}, {}", blockState, ii, jj, kk);
                        this.blocks.put(new class_2338(ii - 1, jj - 1, kk - 1), blockState);
                    }
                }
            }
        }
        field_11863.method_8413(this.method_11016(), this.method_11010(), this.method_11010(), class_2248.field_31028);
    }

    @Override
    public boolean isValid() {
        return super.isValid() && !this.fake;
    }

    public boolean isFake() {
        return fake;
    }

    public void resetRuntime() {
        this.runtimeUuid = UUID.randomUUID();
    }

    public RuntimeReloadResult reloadRuntime() {
        return this.reloadRuntime(Sets.newHashSet());
    }

    public RuntimeReloadResult reloadRuntime(Set<UUID> recurrence) {
        return this.reloadRuntimeAndModeMap(recurrence).getLeft();
    }

    public Pair<RuntimeReloadResult, Map<RelativeDirection, CircuitBoardBlock.Mode>> reloadRuntimeAndModeMap(Set<UUID> recurrence) {
        if (this.uuid == null) return this.emptyMapResult(RuntimeReloadResult.FAIL_NOT_EXIST);
        MinecraftServer server = this.field_11863.method_8503();
        if (server == null) return this.emptyMapResult(RuntimeReloadResult.FAIL_NO_SERVER);
        class_3218 circuitBoardLevel = field_11863.method_8503().method_3847(Constants.CIRCUIT_BOARD_DIMENSION);
        class_3218 runtimeLevel = field_11863.method_8503().method_3847(Constants.RUNTIME_DIMENSION);
        if (circuitBoardLevel == null || runtimeLevel == null) return this.emptyMapResult(RuntimeReloadResult.FAIL_NO_SERVER);
        CircuitSavedData boardData = CircuitSavedData.getCircuitBoardData(circuitBoardLevel);
        CircuitSavedData runtimeData = CircuitSavedData.getRuntimeData(runtimeLevel);
        class_2338 boardPos = boardData.getCircuitStartingPos(this.uuid);
        if (boardPos == null) return this.emptyMapResult(RuntimeReloadResult.FAIL_NOT_EXIST); // circuit doesn't exist yet. use the poking stick on it
        this.blockSize = boardData.getParentOctolet(this.uuid).blockSize;
        Octolet octolet = runtimeData.getParentOctolet(this.runtimeUuid);
        int octoletIndex = runtimeData.octoletIndexForSize(blockSize);
        if (octolet == null) {
            if (!runtimeData.octolets.containsKey(octoletIndex)) runtimeData.addOctolet(octoletIndex, new Octolet(this.blockSize));
            runtimeData.addCircuit(this.runtimeUuid, octoletIndex);
            octolet = runtimeData.octolets.get(octoletIndex);
        }
        recurrence.add(this.uuid);
        class_2338 start = Octolet.getOctoletPos(octoletIndex);
        for (class_1923 pos : octolet.getLoadedChunks())
            runtimeLevel.method_17988(start.method_10263() / 16 + pos.field_9181, start.method_10260() / 16 + pos.field_9180, true);
        class_2338 runtimePos = runtimeData.getCircuitStartingPos(this.runtimeUuid);
        Map<RelativeDirection, CircuitBoardBlock.Mode> modeMap = Maps.newHashMap();
        List<class_2338> outputBlockPos = Lists.newArrayList();
        for (int ii = 0; ii < this.blockSize; ii++) {
            for (int jj = 0; jj < this.blockSize; jj++) {
                for (int kk = 0; kk < this.blockSize; kk++) {
                    class_2338 oldPos = boardPos.method_10069(ii, jj, kk);
                    class_2338 newPos = runtimePos.method_10069(ii, jj, kk);
                    class_2680 oldState = circuitBoardLevel.method_8320(oldPos);
                    runtimeLevel.method_8652(newPos, oldState, class_2248.field_31031 | class_2248.field_31028); // no neighbor update to prevent things from breaking
                    class_2586 oldBlockEntity = circuitBoardLevel.method_8321(oldPos);
                    if (oldBlockEntity != null) {
                        class_2487 save = oldBlockEntity.method_38244();
                        class_2586 be = runtimeLevel.method_8321(newPos);
                        be.method_11014(save);
                        if (be instanceof CircuitBoardBlockEntity blockEntity) {
                            RelativeDirection dir = oldState.method_11654(CircuitBoardBlock.DIRECTION);
                            CircuitBoardBlock.Mode mode = oldState.method_11654(CircuitBoardBlock.MODE);
                            if (modeMap.containsKey(dir)) {
                                CircuitBoardBlock.Mode existingMode = modeMap.get(dir);
                                if (mode != CircuitBoardBlock.Mode.NONE && existingMode != mode) {
                                    this.removeRuntime();
                                    return this.emptyMapResult(RuntimeReloadResult.FAIL_MULTI_MODE);
                                }
                            } else if (mode != CircuitBoardBlock.Mode.NONE)
                                modeMap.put(dir, mode);
                            blockEntity.setConnection(this.field_11863.method_27983(), this.method_11016(), this.runtimeUuid);
                            if (mode == CircuitBoardBlock.Mode.OUTPUT) outputBlockPos.add(newPos);
                        } else if (be instanceof CircuitBlockEntity blockEntity) {
                            if (recurrence.contains(blockEntity.getUuid())) {
                                this.removeRuntime();
                                return this.emptyMapResult(RuntimeReloadResult.FAIL_RECURRENCE);
                            } else {
                                blockEntity.resetRuntime();
                                RuntimeReloadResult result = blockEntity.reloadRuntime(recurrence);
                                if (!result.isGood()) {
                                    this.removeRuntime();
                                    return this.emptyMapResult(result);
                                }
                            }
                        }
                    }
                }
            }
        }
        // tick everything inside once
        for (int ii = 1; ii < this.blockSize - 1; ii++) {
            for (int jj = 1; jj < this.blockSize - 1; jj++) {
                for (int kk = 1; kk < this.blockSize - 1; kk++) {
                    class_2338 pos = runtimePos.method_10069(ii, jj, kk);
                    class_2680 state = runtimeLevel.method_8320(pos);
                    if (!state.method_26215()) runtimeLevel.method_8408(pos, state.method_26204());
                }
            }
        }
        this.updateInputs();
        outputBlockPos.forEach(pos -> {
            class_2680 state = runtimeLevel.method_8320(pos);
            runtimeLevel.method_8492(pos, state.method_26204(), pos.method_10093(DirectionHelper.circuitBoardFixedDirection(state.method_11654(CircuitBoardBlock.DIRECTION))));
        });
        this.chunkedOffset = 0;
        this.chunked = false;
        this.updateInnerBlocks();
        return Pair.of(RuntimeReloadResult.SUCCESS, modeMap);
    }

    private Pair<RuntimeReloadResult, Map<RelativeDirection, CircuitBoardBlock.Mode>> emptyMapResult(RuntimeReloadResult result) {
        return Pair.of(result, Maps.newHashMap());
    }

    public void updateRuntimeBlock(int signal, RelativeDirection direction) {
        MinecraftServer server = field_11863.method_8503();
        if (server == null) return;
        class_3218 runtimeLevel = server.method_3847(Constants.RUNTIME_DIMENSION);
        if (runtimeLevel == null) return;
        CircuitSavedData data = CircuitSavedData.getRuntimeData(runtimeLevel);
        if (!data.circuits.containsKey(this.runtimeUuid)) return;
        int octoletIndex = data.octoletIndexForSize(blockSize);
        if (!data.octolets.containsKey(octoletIndex)) data.addOctolet(octoletIndex, new Octolet(blockSize));
        class_2338 startingPos = data.getCircuitStartingPos(this.runtimeUuid);
        if (startingPos == null) return;
        for (int ii = 0; ii < this.blockSize; ii++) {
            for (int jj = 0; jj < this.blockSize; jj++) {
                class_2338 pos = this.twoDimensionalRelativeDirectionOffset(startingPos, ii, jj, direction);
                class_2680 state = runtimeLevel.method_8320(pos);
                if (state.method_27852(Blocks.CIRCUIT_BOARD.get()) && state.method_11654(CircuitBoardBlock.MODE) == CircuitBoardBlock.Mode.INPUT)
                    runtimeLevel.method_8501(pos, state.method_11657(CircuitBoardBlock.POWER, signal));
            }
        }
    }

    public void removeRuntime() {
        if (this.uuid == null) return;
        MinecraftServer server = this.field_11863.method_8503();
        if (server == null) return;
        class_3218 runtimeLevel = field_11863.method_8503().method_3847(Constants.RUNTIME_DIMENSION);
        if (runtimeLevel == null) return;
        CircuitSavedData runtimeData = CircuitSavedData.getRuntimeData(runtimeLevel);
        Octolet octolet = runtimeData.getParentOctolet(this.runtimeUuid);
        if (octolet != null && octolet.blocks.containsKey(this.runtimeUuid)) {
            // we need to remove all recurrence, so may as well remove the blocks
            class_2338 runtimePos = runtimeData.getCircuitStartingPos(this.runtimeUuid);
            for (int ii = 0; ii < this.blockSize; ii++) {
                for (int jj = 0; jj < this.blockSize; jj++) {
                    for (int kk = 0; kk < this.blockSize; kk++) {
                        class_2338 pos = runtimePos.method_10069(ii, jj, kk);
                        if (runtimeLevel.method_8321(pos) instanceof CircuitBlockEntity blockEntity)
                            blockEntity.removeRuntime();
                        runtimeLevel.method_8652(pos, net.minecraft.class_2246.field_10124.method_9564(), class_2248.field_31031 | class_2248.field_31028); // no neighbor update to prevent things from breaking
                    }
                }
            }
            Set<class_1923> chunks = octolet.getBlockChunk(octolet.blocks.get(this.runtimeUuid));
            class_2338 start = Octolet.getOctoletPos(runtimeData.circuits.get(this.runtimeUuid));
            runtimeData.removeCircuit(this.runtimeUuid);
            Set<class_1923> newChunks = octolet.getLoadedChunks();
            for (class_1923 chunk : chunks) {
                if (!newChunks.contains(chunk))
                    runtimeLevel.method_17988(start.method_10263() / 16 + chunk.field_9181, start.method_10260() / 16 + chunk.field_9180, false);
            }
        }
    }

    private class_2338 twoDimensionalRelativeDirectionOffset(class_2338 pos, int ii, int jj, RelativeDirection direction) {
        return switch (direction) {
            case UP -> pos.method_10069(ii, this.blockSize - 1, jj);
            case DOWN -> pos.method_10069(ii, 0, jj);
            case RIGHT -> pos.method_10069(ii, jj, 0);
            case LEFT -> pos.method_10069(ii, jj, this.blockSize - 1);
            case FRONT -> pos.method_10069(0, ii, jj);
            case BACK -> pos.method_10069(this.blockSize - 1, ii, jj);
        };
    }

    @Override
    public void method_11014(class_2487 tag) {
        super.method_11014(tag);
        this.runtimeUuid = tag.method_25926("runtimeUuid");
        this.blockSize = tag.method_10568("blockSize");
        this.fake = tag.method_10577("fake");
        this.powers = tag.method_10547("powers");
        if (this.powers.length != 6) this.powers = new byte[6];
        this.inputs = tag.method_10547("inputs");
        if (this.inputs.length != 6) this.inputs = new byte[6];
        if (!this.hidden) this.loadExtraFromData(tag);
    }

    @Override
    protected void method_11007(class_2487 tag) {
        super.method_11007(tag);
        tag.method_25927("runtimeUuid", this.runtimeUuid);
        tag.method_10575("blockSize", this.blockSize);
        tag.method_10556("fake", this.fake);
        tag.method_10570("powers", this.powers);
        tag.method_10570("inputs", this.inputs);
    }

    @Override
    public class_2487 method_16887() {
        class_2487 tag = super.method_16887();
        if (this.hidden) return tag;
        class_2499 list = new class_2499(), testList = new class_2499();
        long size = tag.method_47988() + (48 + 28 + 2 * 6 + 36) + (48 + 28 + 2 * 7 + 36 + 9);
        boolean broke = false, oldChunked = this.chunked;
        int ii = 0;
        for (Map.Entry<class_2338, class_2680> entry : this.blocks.entrySet()) {
            if (ii++ < this.chunkedOffset) continue;
            class_2487 tuple = new class_2487();
            tuple.method_10566("pos", class_2512.method_10692(entry.getKey()));
            tuple.method_10566("block", class_2512.method_10686(entry.getValue()));
            testList.add(tuple);

            this.chunkedOffset++;
            if (testList.method_47988() + size >= MAX_TAG_BYTE_SIZE) {
                this.chunked = true;
                broke = true;
                break;
            } else {
                list.add(tuple);
            }
        }
        // keep chunked true. will reset on reload
        if (!broke)
            this.chunkedOffset = 0;
        tag.method_10566("blocks", list);
        // chunked only changes from false to true after first update of reload. we want to clear out old blocks
        tag.method_10556("chunked", oldChunked == this.chunked && this.chunked);
        return tag;
    }

    public void loadExtraFromData(class_2487 tag) {
        if (this.field_11863 == null) return;
        this.chunked = tag.method_10577("chunked");
        Map<class_2338, class_2680> blocks = Maps.newHashMap();
        for (class_2520 t : tag.method_10554("blocks", class_2520.field_33260)) {
            class_2487 tuple = (class_2487) t;
            if (!tuple.method_10573("pos", class_2520.field_33260)) continue;
            class_2338 pos = class_2512.method_10691(tuple.method_10562("pos"));
            class_2680 state = class_2512.method_10681(this.field_11863.method_45448(class_7924.field_41254), tuple.method_10562("block"));
            blocks.put(pos, state);
        }
        if (!this.chunked) this.blocks = blocks;
        else this.blocks.putAll(blocks);
    }

    public UUID getRuntimeUuid() {
        return runtimeUuid;
    }

    public UUID getOwnerUuid() {
        return ownerUuid;
    }

    public void setOwnerUuid(UUID ownerUuid) {
        this.ownerUuid = ownerUuid;
    }

    public short getBlockSize() {
        return blockSize;
    }

    public void setBlockSize(short blockSize) {
        // If block size is cheesed, set to smallest
        if (blockSize > Config.MAX_CIRCUIT_SIZE) blockSize = 4;
        this.blockSize = blockSize;
        this.method_5431();
    }

    public void setFake(boolean fake) {
        this.fake = fake;
        this.method_5431();
    }

    public boolean matchRuntimeUuid(UUID uuid) {
        return this.runtimeUuid.equals(uuid);
    }

    public boolean setPower(int power, RelativeDirection direction) {
        byte oldPower = this.powers[direction.getId()];
        if (oldPower == power) return false;
        this.powers[direction.getId()] = (byte) power;
        class_2680 state = this.method_11010();
        boolean powered = false;
        for (byte pow : this.powers) {
            if (pow > 0) {
                powered = true;
                break;
            }
        }
        state = state.method_11657(CircuitBlock.POWERED, powered);
        this.field_11863.method_8652(this.method_11016(), state, class_2248.field_31028);
        this.method_5431();
        this.updateInnerBlocks();
        return true;
    }

    public int getPower(class_2350 direction) {
        switch (direction) {
            // this is so stupid. why is the direction of signals flipped!?
            case field_11036: return this.powers[RelativeDirection.DOWN.getId()];
            case field_11033: return this.powers[RelativeDirection.UP.getId()];
        }
        int data2d = this.method_11010().method_11654(class_2383.field_11177).method_10161();
        int offset = direction.method_10161() - data2d;
        if (offset < 0) offset += 4;
        return switch (offset) {
            case 0 -> this.powers[RelativeDirection.BACK.getId()];
            case 1 -> this.powers[RelativeDirection.LEFT.getId()];
            case 2 -> this.powers[RelativeDirection.FRONT.getId()];
            case 3 -> this.powers[RelativeDirection.RIGHT.getId()];
            default -> 0;
        };
    }

    public int getRelativePower(RelativeDirection direction) {
        return this.powers[direction.getId()];
    }

    @Override
    public void updateInputs() {
        // stop infinite updates when a side has ticked over n times
        if (this.maxUpdateReached()) return;
        class_2338 pos = this.method_11016();
        class_2680 state = this.method_11010();
        for (class_2350 direction : class_2350.values()) {
            RelativeDirection relDir = DirectionHelper.directionToRelativeDirection(state.method_11654(class_2383.field_11177), direction);
            int signal = field_11863.method_49808(pos.method_10093(direction), direction);
            if (this.inputs[relDir.getId()] != signal) {
                this.sideUpdated(relDir);
                this.inputs[relDir.getId()] = (byte) signal;
                this.updateRuntimeBlock(signal, relDir);
            }
        }
    }

    public enum RuntimeReloadResult {
        SUCCESS("action.circuit.reload.success", true),
        FAIL_NO_SERVER("action.circuit.reload.fail.no_server", false),
        FAIL_NOT_EXIST("action.circuit.reload.fail.not_exist", false),
        FAIL_RECURRENCE("action.circuit.reload.fail.recurrence", false),
        FAIL_MULTI_MODE("action.circuit.reload.fail.multi_mode", false);

        final String translationKey;
        final boolean good;

        RuntimeReloadResult(String translationKey, boolean good) {
            this.translationKey = translationKey;
            this.good = good;
        }

        public String getTranslationKey() {
            return translationKey;
        }

        public boolean isGood() {
            return good;
        }
    }
}
