/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.features.aging;

import com.mojang.logging.LogUtils;
import java.util.Map;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.api.distmarker.Dist;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.fml.event.lifecycle.FMLClientSetupEvent;
import net.neoforged.neoforge.event.entity.player.PlayerEvent;
import org.slf4j.Logger;
import world.landfall.persona.client.gui.input.CharacterCreationInputRegistry;
import world.landfall.persona.config.Config;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.features.aging.AgingInputProvider;

@EventBusSubscriber(modid="persona")
public class AgingManager {
    private static final Logger LOGGER = LogUtils.getLogger();
    public static final ResourceLocation AGING_DATA_KEY = ResourceLocation.parse((String)"persona:aging_data");
    public static final String CREATION_TIMESTAMP_KEY = "CreationTimestampMillis";
    public static final String STARTING_AGE_KEY = "StartingAge";
    private static final double MILLIS_PER_REAL_DAY = 8.64E7;
    private static final ResourceLocation AGING_INPUT_PROVIDER_ID = ResourceLocation.parse((String)"persona:aging_input");

    private static boolean isAgingEnabled() {
        return (Boolean)Config.ENABLE_AGING_SYSTEM.get();
    }

    private static double getRealLifeDaysPerGameYearRatio() {
        double ratio = (Double)Config.TIME_PASSING_RATIO.get();
        if (ratio <= 0.0) {
            LOGGER.warn("Configured TIME_PASSING_RATIO is invalid ({}). Defaulting to 1.0.", (Object)ratio);
            return 1.0;
        }
        return ratio;
    }

    public static void initializeCharacterAging(CharacterProfile profile, CompoundTag agingInputData) {
        double startingAge;
        if (!AgingManager.isAgingEnabled()) {
            LOGGER.debug("Aging system disabled. Skipping aging initialization for character {}.", (Object)profile.getId());
            return;
        }
        CompoundTag agingDataTag = profile.getModData(AGING_DATA_KEY);
        if (agingDataTag == null) {
            agingDataTag = new CompoundTag();
        }
        if (agingInputData != null && agingInputData.contains(STARTING_AGE_KEY, 99)) {
            startingAge = agingInputData.getDouble(STARTING_AGE_KEY);
            double minAge = (Double)Config.MIN_CHARACTER_AGE.get();
            double maxAge = (Double)Config.MAX_CHARACTER_AGE.get();
            if (startingAge < minAge) {
                LOGGER.warn("Starting age {} for character {} is below minimum age {}. Using minimum age.", new Object[]{startingAge, profile.getId(), minAge});
                startingAge = minAge;
            } else if (startingAge > maxAge) {
                LOGGER.warn("Starting age {} for character {} is above maximum age {}. Using maximum age.", new Object[]{startingAge, profile.getId(), maxAge});
                startingAge = maxAge;
            }
        } else {
            startingAge = (Double)Config.DEFAULT_CHARACTER_AGE.get();
            LOGGER.debug("No starting age provided for character {}. Using default age: {}", (Object)profile.getDisplayName(), (Object)startingAge);
        }
        double realLifeDaysPerGameYear = AgingManager.getRealLifeDaysPerGameYearRatio();
        long backdateMillis = (long)(startingAge * realLifeDaysPerGameYear * 8.64E7);
        long effectiveTimestampMillis = System.currentTimeMillis() - backdateMillis;
        LOGGER.debug("Character {} created with age: {:.2f} game years. Effective creation timestamp: {}. (Ratio: {:.2f} real days/game year)", new Object[]{profile.getDisplayName(), startingAge, effectiveTimestampMillis, realLifeDaysPerGameYear});
        agingDataTag.putLong(CREATION_TIMESTAMP_KEY, effectiveTimestampMillis);
        profile.setModData(AGING_DATA_KEY, agingDataTag);
        LOGGER.debug("Set aging data for {}: {}", (Object)profile.getDisplayName(), (Object)agingDataTag);
    }

    public static void handleCharacterCreationWithModData(CharacterProfile newProfile, Map<ResourceLocation, CompoundTag> modData) {
        if (!AgingManager.isAgingEnabled()) {
            LOGGER.debug("Aging system disabled. Skipping character creation aging handler for {}.", (Object)newProfile.getDisplayName());
            return;
        }
        LOGGER.debug("Handling character creation aging for {} with modData keys: {}", (Object)newProfile.getDisplayName(), modData != null ? modData.keySet() : "null");
        CompoundTag agingInput = null;
        if (modData != null && modData.containsKey(AGING_INPUT_PROVIDER_ID)) {
            agingInput = modData.get(AGING_INPUT_PROVIDER_ID);
            LOGGER.debug("Found aging input data via AGING_INPUT_PROVIDER_ID for {}: {}", (Object)newProfile.getDisplayName(), (Object)agingInput);
        }
        AgingManager.initializeCharacterAging(newProfile, agingInput);
    }

    public static void calculateAndUpdateAge(CharacterProfile profile) {
        if (!AgingManager.isAgingEnabled()) {
            return;
        }
        if (profile == null) {
            LOGGER.warn("calculateAndUpdateAge called with null profile.");
            return;
        }
        double currentAge = AgingManager.getCalculatedAge(profile);
        LOGGER.debug("Character {} ({}): Current age is approx. {:.2f} game years.", new Object[]{profile.getDisplayName(), profile.getId(), currentAge});
    }

    public static void triggerAgeUpdate(ServerPlayer player, CharacterProfile profile) {
        if (!AgingManager.isAgingEnabled()) {
            return;
        }
        if (profile == null) {
            LOGGER.warn("triggerAgeUpdate called with null profile for player {}.", (Object)(player != null ? player.getName().getString() : "unknown"));
            return;
        }
        LOGGER.debug("Triggering age update for character: {} (Player: {})", (Object)profile.getDisplayName(), (Object)(player != null ? player.getName().getString() : "N/A"));
        AgingManager.calculateAndUpdateAge(profile);
    }

    public static double getCalculatedAge(CharacterProfile profile) {
        if (!AgingManager.isAgingEnabled()) {
            return 0.0;
        }
        if (profile == null) {
            LOGGER.warn("getCalculatedAge called with null profile. Returning 0.");
            return 0.0;
        }
        CompoundTag agingData = profile.getModData(AGING_DATA_KEY);
        if (agingData == null || !agingData.contains(CREATION_TIMESTAMP_KEY, 4)) {
            LOGGER.warn("Aging data or creation timestamp not found for character {}. Cannot calculate age. Returning 0.", (Object)profile.getId());
            return 0.0;
        }
        long creationTimestampMillis = agingData.getLong(CREATION_TIMESTAMP_KEY);
        long currentTimeMillis = System.currentTimeMillis();
        double realLifeDaysPerGameYear = AgingManager.getRealLifeDaysPerGameYearRatio();
        if (currentTimeMillis < creationTimestampMillis) {
            LOGGER.warn("Current time ({}) is before creation timestamp ({}) for character {}. Age might be negative or zero (indicates clock issue or bad backdating).", new Object[]{currentTimeMillis, creationTimestampMillis, profile.getId()});
        }
        double elapsedMillis = currentTimeMillis - creationTimestampMillis;
        double elapsedRealDays = elapsedMillis / 8.64E7;
        double gameYears = elapsedRealDays / realLifeDaysPerGameYear;
        return gameYears;
    }

    @SubscribeEvent
    public static void onPlayerLogin(PlayerEvent.PlayerLoggedInEvent event) {
        Player player = event.getEntity();
        if (!(player instanceof ServerPlayer)) {
            return;
        }
        ServerPlayer player2 = (ServerPlayer)player;
        if (!AgingManager.isAgingEnabled()) {
            return;
        }
        PlayerCharacterData data = (PlayerCharacterData)player2.getData(PlayerCharacterCapability.CHARACTER_DATA);
        if (data == null || data.getActiveCharacterId() == null) {
            LOGGER.debug("Player {} logged in, but no active character data. No aging actions taken.", (Object)player2.getName().getString());
            return;
        }
        CharacterProfile profile = data.getCharacter(data.getActiveCharacterId());
        if (profile != null) {
            boolean needsInitialization;
            LOGGER.debug("Player {} logged in. Checking/updating age for character {}.", (Object)player2.getName().getString(), (Object)profile.getDisplayName());
            CompoundTag agingData = profile.getModData(AGING_DATA_KEY);
            boolean bl = needsInitialization = agingData == null || !agingData.contains(CREATION_TIMESTAMP_KEY, 4);
            if (needsInitialization) {
                LOGGER.warn("Aging data for character {} of player {} is missing or incomplete on login. Initializing timestamp with current time.", (Object)profile.getDisplayName(), (Object)player2.getName().getString());
                AgingManager.initializeCharacterAging(profile, null);
            }
            AgingManager.calculateAndUpdateAge(profile);
        } else {
            LOGGER.debug("Player {} logged in, but no active Persona profile found. No aging actions taken.", (Object)player2.getName().getString());
        }
    }

    @EventBusSubscriber(modid="persona", value={Dist.CLIENT}, bus=EventBusSubscriber.Bus.MOD)
    public static class ClientRegistration {
        @SubscribeEvent
        public static void onClientSetup(FMLClientSetupEvent event) {
            event.enqueueWork(() -> {
                CharacterCreationInputRegistry.register(new AgingInputProvider());
                LOGGER.debug("[Persona] Registered AgingInputProvider for character creation GUI.");
            });
        }
    }
}

