/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.data;

import com.mojang.logging.LogUtils;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import org.slf4j.Logger;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.registry.GlobalCharacterRegistry;

public class CharacterFileStorage {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final String CHARACTERS_DIR = "characters";
    private static final String CHARACTER_FILE_EXTENSION = ".dat";
    private static Path charactersDirectory;
    private static final ReentrantReadWriteLock storageLock;
    private static final Map<Long, CharacterProfile> characterCache;

    public static void initialize(Path worldPath) {
        storageLock.writeLock().lock();
        try {
            Path personaDir = worldPath.resolve("persona").normalize();
            charactersDirectory = personaDir.resolve(CHARACTERS_DIR).normalize();
            Files.createDirectories(charactersDirectory, new FileAttribute[0]);
            LOGGER.info("[CharacterFileStorage] Initialized character storage at: {}", (Object)charactersDirectory);
        }
        catch (IOException e) {
            LOGGER.error("[CharacterFileStorage] Failed to initialize character storage", (Throwable)e);
            throw new RuntimeException("Failed to initialize character file storage", e);
        }
        finally {
            storageLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean saveCharacter(CharacterProfile character) {
        if (character == null) {
            LOGGER.warn("[CharacterFileStorage] Cannot save null character");
            return false;
        }
        if (charactersDirectory == null) {
            LOGGER.debug("[CharacterFileStorage] Storage not initialized - this should only be called on server side");
            return false;
        }
        long characterId = character.getId();
        Path characterFile = CharacterFileStorage.getCharacterFilePath(characterId);
        storageLock.writeLock().lock();
        try {
            CompoundTag characterData = character.serialize();
            NbtIo.writeCompressed((CompoundTag)characterData, (Path)characterFile);
            characterCache.put(characterId, character);
            LOGGER.debug("[CharacterFileStorage] Saved character {} to file: {}", (Object)character.getDisplayName(), (Object)characterFile.getFileName());
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            LOGGER.error("[CharacterFileStorage] Failed to save character {} ({})", new Object[]{character.getDisplayName(), characterId, e});
            boolean bl = false;
            return bl;
        }
        finally {
            storageLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static CharacterProfile loadCharacter(Long characterId) {
        if (characterId == null) {
            LOGGER.warn("[CharacterFileStorage] Cannot load character with null ID");
            return null;
        }
        if (charactersDirectory == null) {
            LOGGER.debug("[CharacterFileStorage] Storage not initialized - this should only be called on server side");
            return null;
        }
        CharacterProfile cached = characterCache.get(characterId);
        if (cached != null) {
            return cached;
        }
        Path characterFile = CharacterFileStorage.getCharacterFilePath(characterId);
        storageLock.readLock().lock();
        try {
            if (!Files.exists(characterFile, new LinkOption[0])) {
                LOGGER.debug("[CharacterFileStorage] Character file not found: {}", (Object)characterFile);
                CharacterProfile characterProfile = null;
                return characterProfile;
            }
            CompoundTag characterData = NbtIo.readCompressed((Path)characterFile, (NbtAccounter)NbtAccounter.unlimitedHeap());
            if (characterData == null) {
                LOGGER.warn("[CharacterFileStorage] Failed to read character data from file: {}", (Object)characterFile);
                CharacterProfile characterProfile = null;
                return characterProfile;
            }
            CharacterProfile character = CharacterProfile.deserialize(characterData);
            characterCache.put(characterId, character);
            LOGGER.debug("[CharacterFileStorage] Loaded character {} from file: {}", (Object)character.getDisplayName(), (Object)characterFile.getFileName());
            CharacterProfile characterProfile = character;
            return characterProfile;
        }
        catch (IOException e) {
            LOGGER.error("[CharacterFileStorage] Failed to load character {}", (Object)characterId, (Object)e);
            CharacterProfile characterProfile = null;
            return characterProfile;
        }
        finally {
            storageLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean deleteCharacter(Long characterId) {
        if (characterId == null) {
            LOGGER.warn("[CharacterFileStorage] Cannot delete character with null ID");
            return false;
        }
        Path characterFile = CharacterFileStorage.getCharacterFilePath(characterId);
        storageLock.writeLock().lock();
        try {
            characterCache.remove(characterId);
            if (!Files.exists(characterFile, new LinkOption[0])) {
                LOGGER.debug("[CharacterFileStorage] Character file already doesn't exist: {}", (Object)characterFile);
                boolean bl = true;
                return bl;
            }
            Files.delete(characterFile);
            LOGGER.debug("[CharacterFileStorage] Deleted character file: {}", (Object)characterFile.getFileName());
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            LOGGER.error("[CharacterFileStorage] Failed to delete character {}", (Object)characterId, (Object)e);
            boolean bl = false;
            return bl;
        }
        finally {
            storageLock.writeLock().unlock();
        }
    }

    public static boolean characterExists(Long characterId) {
        if (characterId == null) {
            return false;
        }
        if (characterCache.containsKey(characterId)) {
            return true;
        }
        Path characterFile = CharacterFileStorage.getCharacterFilePath(characterId);
        storageLock.readLock().lock();
        try {
            boolean bl = Files.exists(characterFile, new LinkOption[0]);
            return bl;
        }
        finally {
            storageLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Map<Long, String> loadPlayerCharacterIds(UUID playerId) {
        if (playerId == null) {
            LOGGER.warn("[CharacterFileStorage] Cannot load characters for null player ID");
            return new HashMap<Long, String>();
        }
        HashMap<Long, String> playerCharacters = new HashMap<Long, String>();
        storageLock.readLock().lock();
        try {
            if (!Files.exists(charactersDirectory, new LinkOption[0])) {
                LOGGER.debug("[CharacterFileStorage] Characters directory doesn't exist yet");
                HashMap<Long, String> hashMap = playerCharacters;
                return hashMap;
            }
            Map<Long, UUID> characterToPlayerMap = GlobalCharacterRegistry.getCharacterToPlayerMap();
            for (Map.Entry<Long, UUID> entry : characterToPlayerMap.entrySet()) {
                CharacterProfile character;
                Long characterId = entry.getKey();
                UUID characterPlayerId = entry.getValue();
                if (!playerId.equals(characterPlayerId) || (character = CharacterFileStorage.loadCharacter(characterId)) == null) continue;
                playerCharacters.put(characterId, character.getDisplayName());
            }
            LOGGER.debug("[CharacterFileStorage] Found {} characters for player {}", (Object)playerCharacters.size(), (Object)playerId);
            HashMap<Long, String> hashMap = playerCharacters;
            return hashMap;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterFileStorage] Failed to load character IDs for player {}", (Object)playerId, (Object)e);
            HashMap<Long, String> hashMap = new HashMap<Long, String>();
            return hashMap;
        }
        finally {
            storageLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Map<Long, CharacterProfile> findOrphanedCharacters(UUID playerId) {
        if (playerId == null) {
            LOGGER.warn("[CharacterFileStorage] Cannot find orphaned characters for null player ID");
            return new HashMap<Long, CharacterProfile>();
        }
        if (charactersDirectory == null) {
            LOGGER.debug("[CharacterFileStorage] Storage not initialized - this should only be called on server side");
            return new HashMap<Long, CharacterProfile>();
        }
        HashMap<Long, CharacterProfile> orphanedCharacters = new HashMap<Long, CharacterProfile>();
        storageLock.readLock().lock();
        try {
            if (!Files.exists(charactersDirectory, new LinkOption[0])) {
                LOGGER.debug("[CharacterFileStorage] Characters directory doesn't exist yet");
                HashMap<Long, CharacterProfile> hashMap = orphanedCharacters;
                return hashMap;
            }
            Map<Long, UUID> characterToPlayerMap = GlobalCharacterRegistry.getCharacterToPlayerMap();
            Files.list(charactersDirectory).filter(path -> path.toString().endsWith(CHARACTER_FILE_EXTENSION)).forEach(path -> {
                try {
                    CharacterProfile character;
                    String filename = path.getFileName().toString();
                    String idStr = filename.replace(CHARACTER_FILE_EXTENSION, "");
                    Long characterId = Long.parseLong(idStr);
                    UUID registeredPlayerId = (UUID)characterToPlayerMap.get(characterId);
                    if ((registeredPlayerId == null || registeredPlayerId.equals(playerId)) && (character = CharacterFileStorage.loadCharacter(characterId)) != null) {
                        orphanedCharacters.put(characterId, character);
                    }
                }
                catch (NumberFormatException e) {
                    LOGGER.warn("[CharacterFileStorage] Skipping file with invalid character ID format: {}", (Object)path.getFileName());
                }
                catch (Exception e) {
                    LOGGER.error("[CharacterFileStorage] Error processing file {} for orphan detection", path, (Object)e);
                }
            });
            if (!orphanedCharacters.isEmpty()) {
                LOGGER.debug("[CharacterFileStorage] Found {} orphaned character files for player {}", (Object)orphanedCharacters.size(), (Object)playerId);
            }
            HashMap<Long, CharacterProfile> hashMap = orphanedCharacters;
            return hashMap;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterFileStorage] Failed to find orphaned characters for player {}", (Object)playerId, (Object)e);
            HashMap<Long, CharacterProfile> hashMap = new HashMap<Long, CharacterProfile>();
            return hashMap;
        }
        finally {
            storageLock.readLock().unlock();
        }
    }

    public static void clearCache() {
        storageLock.writeLock().lock();
        try {
            characterCache.clear();
            LOGGER.debug("[CharacterFileStorage] Cleared character cache");
        }
        finally {
            storageLock.writeLock().unlock();
        }
    }

    private static Path getCharacterFilePath(Long characterId) {
        if (charactersDirectory == null) {
            throw new IllegalStateException("CharacterFileStorage is not initialized. This should only be called on the server side.");
        }
        String filename = characterId + CHARACTER_FILE_EXTENSION;
        Path filePath = charactersDirectory.resolve(filename);
        try {
            Path normalizedPath = filePath.normalize();
            if (!normalizedPath.startsWith(charactersDirectory)) {
                throw new SecurityException("Path traversal attempt detected: " + filename);
            }
            return normalizedPath;
        }
        catch (Exception e) {
            throw new SecurityException("Invalid character file path", e);
        }
    }

    public static Path getCharactersDirectory() {
        return charactersDirectory;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int cleanupDeceasedCharacters(long gracePeriodHours) {
        if (charactersDirectory == null) {
            LOGGER.debug("[CharacterFileStorage] Storage not initialized - cleanup only runs on server side");
            return 0;
        }
        long gracePeriodMillis = gracePeriodHours * 60L * 60L * 1000L;
        long currentTime = System.currentTimeMillis();
        ArrayList cleanedCharacters = new ArrayList();
        storageLock.writeLock().lock();
        try {
            if (!Files.exists(charactersDirectory, new LinkOption[0])) {
                int n = 0;
                return n;
            }
            Files.list(charactersDirectory).filter(path -> path.toString().endsWith(CHARACTER_FILE_EXTENSION)).forEach(path -> {
                try {
                    long timeSinceSync;
                    String filename = path.getFileName().toString();
                    String idStr = filename.replace(CHARACTER_FILE_EXTENSION, "");
                    Long characterId = Long.parseLong(idStr);
                    CharacterProfile character = CharacterFileStorage.loadCharacter(characterId);
                    if (character != null && character.isDeceased() && !character.hasInventory() && (timeSinceSync = currentTime - character.getLastSyncedAt()) > gracePeriodMillis && Files.deleteIfExists(path)) {
                        characterCache.remove(characterId);
                        cleanedCharacters.add(characterId);
                        LOGGER.info("[CharacterFileStorage] Cleaned up deceased character {} (ID: {})", (Object)character.getDisplayName(), (Object)characterId);
                    }
                }
                catch (Exception e) {
                    LOGGER.error("[CharacterFileStorage] Error processing file {} for cleanup", path, (Object)e);
                }
            });
            int cleanedCount = cleanedCharacters.size();
            if (cleanedCount > 0) {
                LOGGER.info("[CharacterFileStorage] Cleanup completed, removed {} deceased characters", (Object)cleanedCount);
            }
            int n = cleanedCount;
            return n;
        }
        catch (IOException e) {
            LOGGER.error("[CharacterFileStorage] Failed to perform deceased character cleanup", (Throwable)e);
            int n = 0;
            return n;
        }
        finally {
            storageLock.writeLock().unlock();
        }
    }

    public static int cleanupDeceasedCharactersImmediate() {
        return CharacterFileStorage.cleanupDeceasedCharacters(0L);
    }

    static {
        storageLock = new ReentrantReadWriteLock();
        characterCache = new ConcurrentHashMap<Long, CharacterProfile>();
    }
}

