/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.data;

import com.mojang.logging.LogUtils;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtAccounter;
import net.minecraft.nbt.NbtIo;
import net.minecraft.nbt.Tag;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.bus.api.Event;
import net.neoforged.neoforge.common.NeoForge;
import net.neoforged.neoforge.server.ServerLifecycleHooks;
import org.slf4j.Logger;
import world.landfall.persona.data.CharacterFileStorage;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.registry.GlobalCharacterRegistry;
import world.landfall.persona.registry.PersonaEvents;
import world.landfall.persona.registry.PersonaNetworking;
import world.landfall.persona.statecraft.StatecraftAPI;

public class CharacterTransactionManager {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final String TRANSACTION_DIR = "transactions";
    private static final String TRANSACTION_FILE_EXTENSION = ".txn";
    private static final ConcurrentHashMap<Long, ReentrantLock> characterLocks = new ConcurrentHashMap();
    private static final ConcurrentHashMap<Long, Long> lockLastUsedTime = new ConcurrentHashMap();
    private static Path transactionDirectory;

    public static void initialize(Path worldPath) {
        try {
            Path personaDir = worldPath.resolve("persona").normalize();
            transactionDirectory = personaDir.resolve(TRANSACTION_DIR).normalize();
            Files.createDirectories(transactionDirectory, new FileAttribute[0]);
            CharacterTransactionManager.cleanupOldTransactions();
            CharacterTransactionManager.cleanupOldLocks();
            LOGGER.info("[CharacterTransactionManager] Initialized transaction manager at: {}", (Object)transactionDirectory);
        }
        catch (IOException e) {
            LOGGER.error("[CharacterTransactionManager] Failed to initialize transaction manager", (Throwable)e);
            throw new RuntimeException("Failed to initialize character transaction manager", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static long createCharacter(UUID playerId, String characterName, CharacterProfile profile) {
        if (playerId == null || characterName == null || profile == null) {
            LOGGER.error("[CharacterTransactionManager] Cannot create character with null parameters");
            return -1L;
        }
        Long originalCharacterId = profile.getId();
        ReentrantLock lock = CharacterTransactionManager.getCharacterLock(originalCharacterId);
        lock.lock();
        try {
            Transaction txn = new Transaction(TransactionType.CREATE, originalCharacterId, playerId, characterName);
            txn.characterData = profile;
            if (!CharacterTransactionManager.prepareCreate(txn)) {
                long l = -1L;
                return l;
            }
            if (CharacterTransactionManager.commitCreate(txn)) {
                if (!txn.characterId.equals(originalCharacterId)) {
                    characterLocks.remove(txn.characterId);
                }
                long l = txn.characterId;
                return l;
            }
            long l = -1L;
            return l;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to create character", (Throwable)e);
            long l = -1L;
            return l;
        }
        finally {
            lock.unlock();
            characterLocks.remove(originalCharacterId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean deleteCharacter(UUID playerId, Long characterId, String characterName) {
        if (playerId == null || characterId == null || characterName == null) {
            LOGGER.error("[CharacterTransactionManager] Cannot delete character with null parameters");
            return false;
        }
        ReentrantLock lock = CharacterTransactionManager.getCharacterLock(characterId);
        lock.lock();
        try {
            Transaction txn = new Transaction(TransactionType.DELETE, characterId, playerId, characterName);
            if (!CharacterTransactionManager.prepareDelete(txn)) {
                boolean bl = false;
                return bl;
            }
            boolean bl = CharacterTransactionManager.commitDelete(txn);
            return bl;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to delete character", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            lock.unlock();
            characterLocks.remove(characterId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean updateCharacterName(Long characterId, String oldName, String newName) {
        if (characterId == null || oldName == null || newName == null) {
            LOGGER.error("[CharacterTransactionManager] Cannot update character with null parameters");
            return false;
        }
        ReentrantLock lock = CharacterTransactionManager.getCharacterLock(characterId);
        lock.lock();
        try {
            UUID playerId = GlobalCharacterRegistry.getPlayerForCharacter(characterId).orElse(null);
            if (playerId == null) {
                LOGGER.error("[CharacterTransactionManager] Character {} not found in registry", (Object)characterId);
                boolean bl = false;
                return bl;
            }
            Transaction txn = new Transaction(TransactionType.UPDATE, characterId, playerId, newName);
            if (!CharacterTransactionManager.prepareUpdate(txn, oldName, newName)) {
                boolean bl = false;
                return bl;
            }
            boolean bl = CharacterTransactionManager.commitUpdate(txn, oldName, newName);
            return bl;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to update character", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            lock.unlock();
        }
    }

    private static boolean prepareCreate(Transaction txn) {
        try {
            txn.state = TransactionState.PREPARING;
            CharacterTransactionManager.saveTransaction(txn);
            txn.state = TransactionState.PREPARED;
            CharacterTransactionManager.saveTransaction(txn);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to prepare create transaction", (Throwable)e);
            CharacterTransactionManager.abortTransaction(txn);
            return false;
        }
    }

    private static boolean prepareDelete(Transaction txn) {
        try {
            txn.state = TransactionState.PREPARING;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterProfile character = CharacterFileStorage.loadCharacter(txn.characterId);
            if (character != null) {
                txn.previousData = character;
            }
            txn.state = TransactionState.PREPARED;
            CharacterTransactionManager.saveTransaction(txn);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to prepare delete transaction", (Throwable)e);
            CharacterTransactionManager.abortTransaction(txn);
            return false;
        }
    }

    private static boolean prepareUpdate(Transaction txn, String oldName, String newName) {
        try {
            txn.state = TransactionState.PREPARING;
            CharacterTransactionManager.saveTransaction(txn);
            txn.previousData = CharacterFileStorage.loadCharacter(txn.characterId);
            if (txn.previousData == null) {
                LOGGER.error("[CharacterTransactionManager] Character {} not found for update", (Object)txn.characterId);
                txn.state = TransactionState.ABORTED;
                CharacterTransactionManager.saveTransaction(txn);
                return false;
            }
            txn.characterData = CharacterProfile.deserialize(txn.previousData.serialize());
            txn.characterData.setDisplayName(newName);
            txn.state = TransactionState.PREPARED;
            CharacterTransactionManager.saveTransaction(txn);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to prepare update transaction", (Throwable)e);
            CharacterTransactionManager.abortTransaction(txn);
            return false;
        }
    }

    private static boolean commitCreate(Transaction txn) {
        try {
            txn.state = TransactionState.COMMITTING;
            CharacterTransactionManager.saveTransaction(txn);
            long statecraftId = StatecraftAPI.getInstance().recordDisplayName(txn.playerId, txn.characterName);
            if (statecraftId == -1L) {
                LOGGER.error("[CharacterTransactionManager] Failed to register character with Statecraft");
                CharacterTransactionManager.abortTransaction(txn);
                return false;
            }
            CharacterProfile newProfile = new CharacterProfile(statecraftId, txn.characterName, false);
            if (txn.characterData != null) {
                txn.characterData.getModData().forEach(newProfile::setModData);
            }
            CharacterFileStorage.saveCharacter(newProfile);
            txn.characterData = newProfile;
            txn.characterId = statecraftId;
            if (!GlobalCharacterRegistry.registerCharacter(statecraftId, txn.playerId, txn.characterName)) {
                LOGGER.warn("[CharacterTransactionManager] Failed to register character in local cache, but Statecraft registration succeeded");
            }
            txn.state = TransactionState.COMMITTED;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterTransactionManager.deleteTransaction(txn);
            LOGGER.info("[CharacterTransactionManager] Successfully created character {} with Statecraft ID {} for player {}", new Object[]{txn.characterName, statecraftId, txn.playerId});
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to commit create transaction", (Throwable)e);
            CharacterTransactionManager.rollbackCreate(txn);
            return false;
        }
    }

    private static boolean commitDelete(Transaction txn) {
        try {
            PlayerCharacterData data;
            ServerPlayer player;
            MinecraftServer server;
            txn.state = TransactionState.COMMITTING;
            CharacterTransactionManager.saveTransaction(txn);
            Long statecraftId = null;
            if (txn.previousData != null) {
                statecraftId = txn.previousData.getId();
            }
            if (statecraftId != null && statecraftId != 0L) {
                boolean markedDeceased = StatecraftAPI.getInstance().markCharacterAsDeceased(statecraftId, txn.characterName);
                if (markedDeceased) {
                    LOGGER.info("[CharacterTransactionManager] Successfully marked character {} as deceased in Statecraft", (Object)txn.characterName);
                } else {
                    LOGGER.warn("[CharacterTransactionManager] Failed to mark character {} as deceased in Statecraft, continuing with local deletion", (Object)txn.characterName);
                }
            } else {
                LOGGER.warn("[CharacterTransactionManager] No Statecraft ID for character {} - local deletion only", (Object)txn.characterName);
            }
            if (txn.previousData != null) {
                txn.previousData.setDeceased(true);
                CharacterFileStorage.saveCharacter(txn.previousData);
            }
            if (txn.playerId != null && (server = ServerLifecycleHooks.getCurrentServer()) != null && (player = server.getPlayerList().getPlayer(txn.playerId)) != null && (data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA)) != null && txn.characterId.equals(data.getActiveCharacterId())) {
                LOGGER.info("[CharacterTransactionManager] Deleted character {} was active, switching to another character", (Object)txn.characterName);
                Long newActiveId = null;
                for (Map.Entry<Long, String> entry : data.getCharacterIds().entrySet()) {
                    CharacterProfile profile;
                    if (entry.getKey().equals(txn.characterId) || (profile = data.getCharacter(entry.getKey())) == null || profile.isDeceased()) continue;
                    newActiveId = entry.getKey();
                    break;
                }
                if (newActiveId != null) {
                    data.setActiveCharacterId(newActiveId);
                    LOGGER.info("[CharacterTransactionManager] Switched player {} to character {}", (Object)player.getName().getString(), newActiveId);
                    NeoForge.EVENT_BUS.post((Event)new PersonaEvents.CharacterSwitchEvent((Player)player, txn.characterId, newActiveId));
                    PersonaNetworking.sendToPlayer(data, player);
                } else {
                    data.setActiveCharacterId(null);
                    LOGGER.warn("[CharacterTransactionManager] Player {} has no living characters remaining", (Object)player.getName().getString());
                    NeoForge.EVENT_BUS.post((Event)new PersonaEvents.CharacterSwitchEvent((Player)player, txn.characterId, null));
                    PersonaNetworking.sendToPlayer(data, player);
                }
            }
            GlobalCharacterRegistry.deleteCharacter(txn.characterId, txn.characterName);
            txn.state = TransactionState.COMMITTED;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterTransactionManager.deleteTransaction(txn);
            LOGGER.info("[CharacterTransactionManager] Successfully marked character {} as deceased in Statecraft", (Object)txn.characterName);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to commit delete transaction", (Throwable)e);
            CharacterTransactionManager.rollbackDelete(txn);
            return false;
        }
    }

    private static boolean commitUpdate(Transaction txn, String oldName, String newName) {
        try {
            txn.state = TransactionState.COMMITTING;
            CharacterTransactionManager.saveTransaction(txn);
            if (!GlobalCharacterRegistry.updateCharacterName(txn.characterId, oldName, newName)) {
                LOGGER.error("[CharacterTransactionManager] Failed to update character name in registry");
                CharacterTransactionManager.abortTransaction(txn);
                return false;
            }
            if (!CharacterFileStorage.saveCharacter(txn.characterData)) {
                LOGGER.error("[CharacterTransactionManager] Failed to save updated character");
                GlobalCharacterRegistry.updateCharacterName(txn.characterId, newName, oldName);
                CharacterTransactionManager.abortTransaction(txn);
                return false;
            }
            txn.state = TransactionState.COMMITTED;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterTransactionManager.deleteTransaction(txn);
            LOGGER.info("[CharacterTransactionManager] Successfully updated character name from {} to {}", (Object)oldName, (Object)newName);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to commit update transaction", (Throwable)e);
            CharacterTransactionManager.rollbackUpdate(txn, oldName, newName);
            return false;
        }
    }

    private static void rollbackCreate(Transaction txn) {
        try {
            LOGGER.warn("[CharacterTransactionManager] Rolling back create transaction for character {}", (Object)txn.characterId);
            GlobalCharacterRegistry.unregisterCharacter(txn.characterId, txn.characterName);
            CharacterFileStorage.deleteCharacter(txn.characterId);
            txn.state = TransactionState.ABORTED;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterTransactionManager.deleteTransaction(txn);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to rollback create transaction", (Throwable)e);
        }
    }

    private static void rollbackDelete(Transaction txn) {
        try {
            LOGGER.warn("[CharacterTransactionManager] Rolling back delete transaction for character {}", (Object)txn.characterId);
            if (txn.previousData != null) {
                CharacterFileStorage.saveCharacter(txn.previousData);
                GlobalCharacterRegistry.registerCharacter(txn.characterId, txn.playerId, txn.characterName);
            }
            txn.state = TransactionState.ABORTED;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterTransactionManager.deleteTransaction(txn);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to rollback delete transaction", (Throwable)e);
        }
    }

    private static void rollbackUpdate(Transaction txn, String oldName, String newName) {
        try {
            LOGGER.warn("[CharacterTransactionManager] Rolling back update transaction for character {}", (Object)txn.characterId);
            GlobalCharacterRegistry.updateCharacterName(txn.characterId, newName, oldName);
            if (txn.previousData != null) {
                CharacterFileStorage.saveCharacter(txn.previousData);
            }
            txn.state = TransactionState.ABORTED;
            CharacterTransactionManager.saveTransaction(txn);
            CharacterTransactionManager.deleteTransaction(txn);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterTransactionManager] Failed to rollback update transaction", (Throwable)e);
        }
    }

    private static void saveTransaction(Transaction txn) {
        try {
            Path txnFile = transactionDirectory.resolve(String.valueOf(txn.transactionId) + TRANSACTION_FILE_EXTENSION);
            CompoundTag data = txn.serialize();
            NbtIo.writeCompressed((CompoundTag)data, (Path)txnFile);
        }
        catch (IOException e) {
            LOGGER.error("[CharacterTransactionManager] Failed to save transaction state", (Throwable)e);
        }
    }

    private static void deleteTransaction(Transaction txn) {
        try {
            Path txnFile = transactionDirectory.resolve(String.valueOf(txn.transactionId) + TRANSACTION_FILE_EXTENSION);
            Files.deleteIfExists(txnFile);
        }
        catch (IOException e) {
            LOGGER.error("[CharacterTransactionManager] Failed to delete transaction file", (Throwable)e);
        }
    }

    private static void abortTransaction(Transaction txn) {
        txn.state = TransactionState.ABORTED;
        CharacterTransactionManager.saveTransaction(txn);
        CharacterTransactionManager.deleteTransaction(txn);
    }

    public static void cleanupOldLocks() {
        long currentTime = System.currentTimeMillis();
        long staleThreshold = 3600000L;
        lockLastUsedTime.entrySet().removeIf(entry -> {
            ReentrantLock lock;
            Long characterId = (Long)entry.getKey();
            Long lastUsed = (Long)entry.getValue();
            if (currentTime - lastUsed > staleThreshold && (lock = characterLocks.get(characterId)) != null && !lock.isLocked()) {
                characterLocks.remove(characterId);
                return true;
            }
            return false;
        });
    }

    private static ReentrantLock getCharacterLock(Long characterId) {
        lockLastUsedTime.put(characterId, System.currentTimeMillis());
        return characterLocks.computeIfAbsent(characterId, k -> new ReentrantLock());
    }

    private static void cleanupOldTransactions() {
        try {
            Files.list(transactionDirectory).filter(path -> path.toString().endsWith(TRANSACTION_FILE_EXTENSION)).forEach(path -> {
                try {
                    CompoundTag data = NbtIo.readCompressed((Path)path, (NbtAccounter)NbtAccounter.unlimitedHeap());
                    Transaction txn = Transaction.deserialize(data);
                    long age = System.currentTimeMillis() - txn.timestamp;
                    if (age > 3600000L) {
                        LOGGER.info("[CharacterTransactionManager] Found old transaction {}, attempting recovery", (Object)txn.transactionId);
                        CharacterTransactionManager.recoverTransaction(txn);
                        Files.deleteIfExists(path);
                    }
                }
                catch (Exception e) {
                    LOGGER.error("[CharacterTransactionManager] Failed to process transaction file {}", path, (Object)e);
                    try {
                        Files.deleteIfExists(path);
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
            });
        }
        catch (IOException e) {
            LOGGER.error("[CharacterTransactionManager] Failed to clean up old transactions", (Throwable)e);
        }
    }

    private static void recoverTransaction(Transaction txn) {
        LOGGER.info("[CharacterTransactionManager] Recovering transaction {} in state {}", (Object)txn.transactionId, (Object)txn.state);
        switch (txn.state.ordinal()) {
            case 0: 
            case 1: {
                LOGGER.info("[CharacterTransactionManager] Aborting uncommitted transaction {}", (Object)txn.transactionId);
                break;
            }
            case 2: {
                LOGGER.warn("[CharacterTransactionManager] Rolling back partially committed transaction {}", (Object)txn.transactionId);
                switch (txn.type.ordinal()) {
                    case 0: {
                        CharacterTransactionManager.rollbackCreate(txn);
                        break;
                    }
                    case 1: {
                        CharacterTransactionManager.rollbackDelete(txn);
                        break;
                    }
                    case 2: {
                        LOGGER.error("[CharacterTransactionManager] Cannot fully rollback UPDATE transaction without old name");
                    }
                }
                break;
            }
            case 3: {
                LOGGER.info("[CharacterTransactionManager] Transaction {} was already committed", (Object)txn.transactionId);
                break;
            }
            case 4: 
            case 5: {
                LOGGER.info("[CharacterTransactionManager] Transaction {} was already aborted", (Object)txn.transactionId);
            }
        }
    }

    private static class Transaction {
        final UUID transactionId;
        final TransactionType type;
        Long characterId;
        final UUID playerId;
        final String characterName;
        final long timestamp;
        TransactionState state;
        CharacterProfile characterData;
        CharacterProfile previousData;

        Transaction(TransactionType type, Long characterId, UUID playerId, String characterName) {
            this.transactionId = UUID.randomUUID();
            this.type = type;
            this.characterId = characterId;
            this.playerId = playerId;
            this.characterName = characterName;
            this.timestamp = System.currentTimeMillis();
            this.state = TransactionState.PREPARING;
        }

        private Transaction(UUID transactionId, TransactionType type, Long characterId, UUID playerId, String characterName, long timestamp, TransactionState state) {
            this.transactionId = transactionId;
            this.type = type;
            this.characterId = characterId;
            this.playerId = playerId;
            this.characterName = characterName;
            this.timestamp = timestamp;
            this.state = state;
        }

        CompoundTag serialize() {
            CompoundTag tag = new CompoundTag();
            tag.putUUID("transactionId", this.transactionId);
            tag.putString("type", this.type.name());
            tag.putLong("characterId", this.characterId.longValue());
            tag.putUUID("playerId", this.playerId);
            tag.putString("characterName", this.characterName);
            tag.putLong("timestamp", this.timestamp);
            tag.putString("state", this.state.name());
            if (this.characterData != null) {
                tag.put("characterData", (Tag)this.characterData.serialize());
            }
            if (this.previousData != null) {
                tag.put("previousData", (Tag)this.previousData.serialize());
            }
            return tag;
        }

        static Transaction deserialize(CompoundTag tag) {
            UUID transactionId = tag.getUUID("transactionId");
            TransactionType type = TransactionType.valueOf(tag.getString("type"));
            Long characterId = tag.getLong("characterId");
            UUID playerId = tag.getUUID("playerId");
            String characterName = tag.getString("characterName");
            long timestamp = tag.getLong("timestamp");
            TransactionState state = TransactionState.valueOf(tag.getString("state"));
            Transaction txn = new Transaction(transactionId, type, characterId, playerId, characterName, timestamp, state);
            if (tag.contains("characterData")) {
                txn.characterData = CharacterProfile.deserialize(tag.getCompound("characterData"));
            }
            if (tag.contains("previousData")) {
                txn.previousData = CharacterProfile.deserialize(tag.getCompound("previousData"));
            }
            return txn;
        }
    }

    public static enum TransactionType {
        CREATE,
        DELETE,
        UPDATE;

    }

    private static enum TransactionState {
        PREPARING,
        PREPARED,
        COMMITTING,
        COMMITTED,
        ABORTING,
        ABORTED;

    }
}

