/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.registry;

import java.nio.file.Path;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.level.storage.LevelResource;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.entity.player.PlayerEvent;
import net.neoforged.neoforge.event.server.ServerStartingEvent;
import net.neoforged.neoforge.event.server.ServerStoppingEvent;
import world.landfall.persona.Persona;
import world.landfall.persona.config.Config;
import world.landfall.persona.data.CharacterFileStorage;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.CharacterTransactionManager;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.features.inventory.InventoryHandler;
import world.landfall.persona.features.location.LocationHandler;
import world.landfall.persona.registry.PersonaNetworking;
import world.landfall.persona.registry.RegistryPersistence;
import world.landfall.persona.statecraft.StatecraftAPI;
import world.landfall.persona.statecraft.SyncManager;

@EventBusSubscriber(modid="persona")
public class GlobalCharacterRegistry {
    private static final Map<Long, UUID> characterToPlayerMap = new ConcurrentHashMap<Long, UUID>();
    private static final Map<String, Long> characterNameMap = new ConcurrentHashMap<String, Long>();
    private static final ReentrantReadWriteLock registryLock = new ReentrantReadWriteLock();

    public static void initialize() {
        registryLock.writeLock().lock();
        try {
            characterToPlayerMap.clear();
            characterNameMap.clear();
            Persona.LOGGER.info("[Persona] Global Character Registry initialized.");
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    @SubscribeEvent
    public static void onServerStarting(ServerStartingEvent event) {
        registryLock.writeLock().lock();
        try {
            Path worldPath = event.getServer().getWorldPath(LevelResource.ROOT);
            RegistryPersistence.initialize(worldPath);
            CharacterFileStorage.initialize(worldPath);
            CharacterTransactionManager.initialize(worldPath);
            RegistryPersistence.RegistryData data = RegistryPersistence.loadRegistry();
            characterToPlayerMap.putAll(data.characterToPlayerMap);
            characterNameMap.putAll(data.characterNameMap);
            SyncManager.getInstance().start(event.getServer());
            Persona.LOGGER.info("[Persona] Global Character Registry, File Storage, Transaction Manager, and Statecraft Sync initialized.");
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    @SubscribeEvent
    public static void onServerStopping(ServerStoppingEvent event) {
        registryLock.writeLock().lock();
        try {
            SyncManager.getInstance().stop();
            GlobalCharacterRegistry.saveAllActiveCharacterData(event.getServer());
            RegistryPersistence.saveRegistry(characterToPlayerMap, characterNameMap);
            Persona.LOGGER.info("[Persona] Global Character Registry saved and Statecraft Sync stopped.");
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean registerCharacter(Long characterId, UUID playerId, String characterName) {
        if (characterId == null || playerId == null || characterName == null) {
            throw new IllegalArgumentException("Character registration parameters cannot be null");
        }
        String normalizedName = characterName.toLowerCase();
        registryLock.writeLock().lock();
        try {
            Long existingCharId;
            if (characterNameMap.containsKey(normalizedName) && !characterId.equals(existingCharId = characterNameMap.get(normalizedName))) {
                boolean bl = false;
                return bl;
            }
            characterToPlayerMap.put(characterId, playerId);
            characterNameMap.put(normalizedName, characterId);
            RegistryPersistence.saveRegistry(characterToPlayerMap, characterNameMap);
            boolean bl = true;
            return bl;
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean updateCharacterName(Long characterId, String oldName, String newName) {
        if (characterId == null || oldName == null || newName == null) {
            throw new IllegalArgumentException("Character update parameters cannot be null");
        }
        String normalizedOldName = oldName.toLowerCase();
        String normalizedNewName = newName.toLowerCase();
        registryLock.writeLock().lock();
        try {
            Long existingCharId = characterNameMap.get(normalizedOldName);
            if (!characterId.equals(existingCharId)) {
                boolean bl = false;
                return bl;
            }
            Long existingForNewName = characterNameMap.get(normalizedNewName);
            if (existingForNewName != null && !characterId.equals(existingForNewName)) {
                boolean bl = false;
                return bl;
            }
            characterNameMap.remove(normalizedOldName);
            characterNameMap.put(normalizedNewName, characterId);
            RegistryPersistence.saveRegistry(characterToPlayerMap, characterNameMap);
            boolean bl = true;
            return bl;
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void unregisterCharacter(Long characterId, String characterName) {
        if (characterId == null || characterName == null) {
            throw new IllegalArgumentException("Character unregistration parameters cannot be null");
        }
        String normalizedName = characterName.toLowerCase();
        registryLock.writeLock().lock();
        try {
            Long existingCharId = characterNameMap.get(normalizedName);
            if (characterId.equals(existingCharId)) {
                characterNameMap.remove(normalizedName);
            }
            characterToPlayerMap.remove(characterId);
            RegistryPersistence.saveRegistry(characterToPlayerMap, characterNameMap);
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    public static Optional<UUID> getPlayerForCharacter(Long characterId) {
        if (characterId == null) {
            return Optional.empty();
        }
        registryLock.readLock().lock();
        try {
            Optional<UUID> optional = Optional.ofNullable(characterToPlayerMap.get(characterId));
            return optional;
        }
        finally {
            registryLock.readLock().unlock();
        }
    }

    public static boolean isNameTaken(String name) {
        if (name == null) {
            return false;
        }
        String normalizedName = name.toLowerCase();
        registryLock.readLock().lock();
        try {
            boolean bl = characterNameMap.containsKey(normalizedName);
            return bl;
        }
        finally {
            registryLock.readLock().unlock();
        }
    }

    public static Optional<Long> getCharacterIdByName(String name) {
        if (name == null) {
            return Optional.empty();
        }
        String normalizedName = name.toLowerCase();
        registryLock.readLock().lock();
        try {
            Optional<Long> optional = Optional.ofNullable(characterNameMap.get(normalizedName));
            return optional;
        }
        finally {
            registryLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @SubscribeEvent
    public static void onPlayerLogin(PlayerEvent.PlayerLoggedInEvent event) {
        ServerPlayer player;
        PlayerCharacterData data;
        Player player2 = event.getEntity();
        if (player2 instanceof ServerPlayer && (data = (PlayerCharacterData)(player = (ServerPlayer)player2).getData(PlayerCharacterCapability.CHARACTER_DATA)) != null) {
            registryLock.writeLock().lock();
            try {
                CharacterProfile activeProfile;
                data.loadCharacterIdsFromStorage(player.getUUID());
                data.getCharacterIds().forEach((id, displayName) -> {
                    characterToPlayerMap.put((Long)id, player.getUUID());
                    characterNameMap.put(displayName.toLowerCase(), (Long)id);
                });
                SyncManager.getInstance().syncPlayer(player);
                PersonaNetworking.sendToPlayer(data, player);
                Persona.LOGGER.info("[GlobalCharacterRegistry] Loaded {} characters from Statecraft for player {}", (Object)data.getCharacterIds().size(), (Object)player.getName().getString());
                Long activeCharacterId = data.getActiveCharacterId();
                if (activeCharacterId != null && (activeProfile = data.getCharacter(activeCharacterId)) != null && activeProfile.getDisplayName() != null) {
                    String characterName = activeProfile.getDisplayName();
                    UUID playerUuid = player.getUUID();
                    CompletableFuture.runAsync(() -> {
                        try {
                            long result = StatecraftAPI.getInstance().recordDisplayName(playerUuid, characterName);
                            if (result != -1L) {
                                Persona.LOGGER.debug("[GlobalCharacterRegistry] Updated lastSeen for active character '{}' (ID: {}) on player login", (Object)characterName, (Object)result);
                            }
                        }
                        catch (Exception e) {
                            Persona.LOGGER.warn("[GlobalCharacterRegistry] Failed to update lastSeen for character '{}' on login: {}", (Object)characterName, (Object)e.getMessage());
                        }
                    });
                }
            }
            finally {
                registryLock.writeLock().unlock();
            }
        }
    }

    @SubscribeEvent
    public static void onPlayerLogout(PlayerEvent.PlayerLoggedOutEvent event) {
        Player player = event.getEntity();
        if (player instanceof ServerPlayer) {
            ServerPlayer player2 = (ServerPlayer)player;
            GlobalCharacterRegistry.saveActiveCharacterData(player2);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean deleteCharacter(Long characterId, String characterName) {
        if (characterId == null || characterName == null) {
            throw new IllegalArgumentException("Character deletion parameters cannot be null");
        }
        String normalizedName = characterName.toLowerCase();
        registryLock.writeLock().lock();
        try {
            Long existingCharId = characterNameMap.get(normalizedName);
            if (characterId.equals(existingCharId)) {
                characterNameMap.remove(normalizedName);
                characterToPlayerMap.remove(characterId);
                RegistryPersistence.saveRegistry(characterToPlayerMap, characterNameMap);
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
        finally {
            registryLock.writeLock().unlock();
        }
    }

    public static void syncRegistry(ServerPlayer player) {
        if (player == null) {
            return;
        }
        PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
        if (data != null) {
            registryLock.readLock().lock();
            try {
                PersonaNetworking.sendToPlayer(data, player);
            }
            finally {
                registryLock.readLock().unlock();
            }
        }
    }

    public static Map<Long, UUID> getCharacterToPlayerMap() {
        registryLock.readLock().lock();
        try {
            ConcurrentHashMap<Long, UUID> concurrentHashMap = new ConcurrentHashMap<Long, UUID>(characterToPlayerMap);
            return concurrentHashMap;
        }
        finally {
            registryLock.readLock().unlock();
        }
    }

    public static Map<String, Long> getCharacterNameMap() {
        registryLock.readLock().lock();
        try {
            ConcurrentHashMap<String, Long> concurrentHashMap = new ConcurrentHashMap<String, Long>(characterNameMap);
            return concurrentHashMap;
        }
        finally {
            registryLock.readLock().unlock();
        }
    }

    private static void saveAllActiveCharacterData(MinecraftServer server) {
        try {
            int savedCount = 0;
            for (ServerPlayer player : server.getPlayerList().getPlayers()) {
                if (!GlobalCharacterRegistry.saveActiveCharacterData(player)) continue;
                ++savedCount;
            }
            Persona.LOGGER.info("[Persona] Saved active character data for {} players before server shutdown.", (Object)savedCount);
        }
        catch (Exception e) {
            Persona.LOGGER.error("[Persona] Error saving active character data during server shutdown", (Throwable)e);
        }
    }

    private static boolean saveActiveCharacterData(ServerPlayer player) {
        try {
            PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (data == null) {
                return false;
            }
            Long activeCharacterId = data.getActiveCharacterId();
            if (activeCharacterId == null) {
                return false;
            }
            CharacterProfile activeProfile = data.getCharacter(activeCharacterId);
            if (activeProfile == null) {
                return false;
            }
            if (((Boolean)Config.ENABLE_INVENTORY_SYSTEM.get()).booleanValue()) {
                try {
                    CompoundTag inventoryTag = InventoryHandler.saveInventory(player);
                    activeProfile.setModData(ResourceLocation.fromNamespaceAndPath((String)"persona", (String)"inventory"), inventoryTag);
                }
                catch (Exception e) {
                    Persona.LOGGER.error("[Persona] Failed to save inventory data for player {}", (Object)player.getName().getString(), (Object)e);
                }
            }
            if (((Boolean)Config.ENABLE_LOCATION_SYSTEM.get()).booleanValue()) {
                try {
                    CompoundTag locationTag = LocationHandler.saveLocation(player);
                    activeProfile.setModData(ResourceLocation.fromNamespaceAndPath((String)"persona", (String)"location"), locationTag);
                }
                catch (Exception e) {
                    Persona.LOGGER.error("[Persona] Failed to save location data for player {}", (Object)player.getName().getString(), (Object)e);
                }
            }
            CharacterFileStorage.saveCharacter(activeProfile);
            Persona.LOGGER.debug("[Persona] Saved active character data for player {} (character: {})", (Object)player.getName().getString(), (Object)activeProfile.getDisplayName());
            return true;
        }
        catch (Exception e) {
            Persona.LOGGER.error("[Persona] Error saving active character data for player {}", (Object)player.getName().getString(), (Object)e);
            return false;
        }
    }
}

