/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.statecraft;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.mojang.logging.LogUtils;
import java.io.IOException;
import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import org.slf4j.Logger;
import world.landfall.persona.config.Config;
import world.landfall.persona.statecraft.StatecraftCharacter;

public class StatecraftAPI {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Gson GSON = new Gson();
    private static final int MAX_RETRIES = 3;
    private static final int INITIAL_RETRY_DELAY_MS = 1000;
    private final HttpClient httpClient = HttpClient.newBuilder().connectTimeout(Duration.ofSeconds(5L)).build();
    private final String baseUrl = (String)Config.STATECRAFT_API_URL.get();
    private final String apiKey = (String)Config.STATECRAFT_API_KEY.get();
    private static StatecraftAPI instance;
    private volatile boolean isConnected = false;

    private StatecraftAPI() {
        LOGGER.info("[StatecraftAPI] Initializing with URL: {} and API key: {}", (Object)this.baseUrl, this.apiKey.length() > 4 ? this.apiKey.substring(0, 4) + "..." : "NOT SET");
        if (this.apiKey.equals("CHANGE_ME") || this.apiKey.isEmpty()) {
            LOGGER.warn("[StatecraftAPI] API key is not configured! Please set 'statecraftApiKey' in the config file or STATECRAFT_API_KEY environment variable.");
        }
        this.isConnected = false;
    }

    public static synchronized StatecraftAPI getInstance() {
        if (instance == null) {
            instance = new StatecraftAPI();
        }
        return instance;
    }

    public long recordDisplayName(UUID playerUuid, String displayName) {
        try {
            JsonObject requestBody = new JsonObject();
            requestBody.addProperty("uuid", playerUuid.toString());
            requestBody.addProperty("displayName", displayName);
            String response = this.executeRequest("POST", "/players/record-display-name", requestBody.toString());
            if (response == null) {
                return -1L;
            }
            JsonObject jsonResponse = (JsonObject)GSON.fromJson(response, JsonObject.class);
            if (jsonResponse.has("characterId")) {
                long characterId = jsonResponse.get("characterId").getAsLong();
                boolean isNew = jsonResponse.has("isNew") && jsonResponse.get("isNew").getAsBoolean();
                LOGGER.info("[StatecraftAPI] {} character '{}' with ID {} for player {}", new Object[]{isNew ? "Created" : "Updated", displayName, characterId, playerUuid});
                return characterId;
            }
            LOGGER.error("[StatecraftAPI] Invalid response from record-display-name: {}", (Object)response);
            return -1L;
        }
        catch (Exception e) {
            LOGGER.error("[StatecraftAPI] Failed to record display name '{}' for player {}", new Object[]{displayName, playerUuid, e});
            return -1L;
        }
    }

    public boolean markCharacterAsDeceased(long characterId, String displayName) {
        try {
            JsonObject requestBody = new JsonObject();
            requestBody.addProperty("characterId", (Number)characterId);
            requestBody.addProperty("displayName", displayName);
            String response = this.executeRequest("POST", "/players/mark-deceased", requestBody.toString());
            if (response == null) {
                LOGGER.error("[StatecraftAPI] Failed to mark character {} as deceased - no response", (Object)characterId);
                return false;
            }
            try {
                JsonObject responseJson = (JsonObject)GSON.fromJson(response, JsonObject.class);
                if (responseJson.has("success") && responseJson.get("success").getAsBoolean()) {
                    LOGGER.info("[StatecraftAPI] Successfully marked character {} ({}) as deceased", (Object)characterId, (Object)displayName);
                    return true;
                }
            }
            catch (Exception e) {
                LOGGER.error("[StatecraftAPI] Failed to parse mark-deceased response: {}", (Object)response, (Object)e);
            }
            return false;
        }
        catch (Exception e) {
            LOGGER.error("[StatecraftAPI] Failed to mark character {} as deceased", (Object)characterId, (Object)e);
            return false;
        }
    }

    public StatecraftCharacter getCharacterById(long characterId) {
        try {
            String response = this.executeRequest("GET", "/players/id/" + characterId, null);
            if (response == null) {
                LOGGER.debug("[StatecraftAPI] Could not retrieve character by ID {} - endpoint may not exist", (Object)characterId);
                return null;
            }
            return this.parseCharacterResponse(response);
        }
        catch (Exception e) {
            LOGGER.debug("[StatecraftAPI] Failed to get character {} - endpoint may not exist", (Object)characterId);
            return null;
        }
    }

    public boolean isConnected() {
        return this.isConnected;
    }

    public List<StatecraftCharacter> getCharactersByPlayerUuid(UUID playerUuid) {
        try {
            String response = this.executeRequest("GET", "/players/uuid/" + playerUuid.toString(), null);
            if (response == null) {
                return new ArrayList<StatecraftCharacter>();
            }
            if (!this.isConnected) {
                this.isConnected = true;
                LOGGER.info("[StatecraftAPI] Connection to Statecraft established!");
            }
            ArrayList<StatecraftCharacter> characters = new ArrayList<StatecraftCharacter>();
            JsonObject jsonResponse = (JsonObject)GSON.fromJson(response, JsonObject.class);
            if (jsonResponse.has("displayNames")) {
                JsonArray displayNames = jsonResponse.getAsJsonArray("displayNames");
                for (JsonElement element : displayNames) {
                    long characterId;
                    boolean isDeceased;
                    JsonObject nameObj = element.getAsJsonObject();
                    if (!nameObj.has("name")) continue;
                    String displayName = nameObj.get("name").getAsString();
                    boolean bl = isDeceased = nameObj.has("isDeceased") && nameObj.get("isDeceased").getAsBoolean();
                    if (nameObj.has("characterId")) {
                        characterId = nameObj.get("characterId").getAsLong();
                        LOGGER.debug("[StatecraftAPI] Loaded character '{}' with ID {} from Statecraft", (Object)displayName, (Object)characterId);
                    } else {
                        String composite = displayName.toLowerCase() + playerUuid.toString();
                        characterId = Math.abs((long)composite.hashCode());
                        LOGGER.warn("[StatecraftAPI] Character '{}' missing ID from Statecraft, using deterministic hash: {}", (Object)displayName, (Object)characterId);
                    }
                    StatecraftCharacter character = new StatecraftCharacter(characterId, displayName, playerUuid);
                    character.setDeceased(isDeceased);
                    if (nameObj.has("firstUsed")) {
                        character.setFirstUsed(nameObj.get("firstUsed").getAsLong());
                    }
                    if (nameObj.has("lastSeen")) {
                        character.setLastSeen(nameObj.get("lastSeen").getAsLong());
                    }
                    characters.add(character);
                }
            }
            LOGGER.info("[StatecraftAPI] Found {} characters for player {}", (Object)characters.size(), (Object)playerUuid);
            return characters;
        }
        catch (Exception e) {
            LOGGER.error("[StatecraftAPI] Failed to get characters for player {}", (Object)playerUuid, (Object)e);
            return new ArrayList<StatecraftCharacter>();
        }
    }

    private String executeRequest(String method, String endpoint, String body) {
        String url = this.baseUrl + endpoint;
        for (int attempt = 0; attempt < 3; ++attempt) {
            try {
                HttpRequest.Builder requestBuilder = HttpRequest.newBuilder().uri(URI.create(url)).timeout(Duration.ofSeconds(30L)).header("X-API-Key", this.apiKey).header("Content-Type", "application/json");
                if ("POST".equals(method) && body != null) {
                    requestBuilder.POST(HttpRequest.BodyPublishers.ofString(body));
                } else if ("GET".equals(method)) {
                    requestBuilder.GET();
                }
                HttpRequest request = requestBuilder.build();
                HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() >= 200 && response.statusCode() < 300) {
                    return response.body();
                }
                if (response.statusCode() >= 400 && response.statusCode() < 500) {
                    if (response.statusCode() == 401) {
                        LOGGER.error("[StatecraftAPI] Authentication failed (401) for {}. Please check your API key in the config. Current key starts with: {}", (Object)url, this.apiKey.length() > 4 ? this.apiKey.substring(0, 4) + "..." : "NOT SET");
                    } else {
                        LOGGER.error("[StatecraftAPI] Client error {} for {}: {}", new Object[]{response.statusCode(), url, response.body()});
                    }
                    return null;
                }
                LOGGER.warn("[StatecraftAPI] Server error {} for {} (attempt {}/{})", new Object[]{response.statusCode(), url, attempt + 1, 3});
                if (attempt >= 2) continue;
                int delay = 1000 * (int)Math.pow(2.0, attempt);
                Thread.sleep(delay);
                continue;
            }
            catch (IOException e) {
                String errorMsg;
                String string = errorMsg = e.getMessage() != null ? e.getMessage() : e.toString();
                if (e instanceof ConnectException) {
                    if (attempt == 0) {
                        LOGGER.warn("[StatecraftAPI] Cannot connect to Statecraft at {} - server appears to be offline", (Object)url);
                    }
                    return null;
                }
                LOGGER.warn("[StatecraftAPI] Network error for {} (attempt {}/{}): {} - {}", new Object[]{url, attempt + 1, 3, e.getClass().getSimpleName(), errorMsg});
                if (attempt >= 2) continue;
                try {
                    int delay = 1000 * (int)Math.pow(2.0, attempt);
                    Thread.sleep(delay);
                    continue;
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    return null;
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.error("[StatecraftAPI] Request interrupted for {}", (Object)url);
                return null;
            }
            catch (Exception e) {
                LOGGER.error("[StatecraftAPI] Unexpected error for {}", (Object)url, (Object)e);
                return null;
            }
        }
        return null;
    }

    private StatecraftCharacter parseCharacterResponse(String response) {
        try {
            JsonObject json = (JsonObject)GSON.fromJson(response, JsonObject.class);
            return this.parseCharacterFromJson(json);
        }
        catch (Exception e) {
            LOGGER.error("[StatecraftAPI] Failed to parse character response: {}", (Object)response, (Object)e);
            return null;
        }
    }

    private StatecraftCharacter parseCharacterFromJson(JsonObject json) {
        try {
            long characterId = json.get("characterId").getAsLong();
            String displayName = json.get("displayName").getAsString();
            UUID playerUuid = UUID.fromString(json.get("playerUuid").getAsString());
            StatecraftCharacter character = new StatecraftCharacter(characterId, displayName, playerUuid);
            if (json.has("profilePictureUrl") && !json.get("profilePictureUrl").isJsonNull()) {
                character.setProfilePictureUrl(json.get("profilePictureUrl").getAsString());
            }
            if (json.has("biography") && !json.get("biography").isJsonNull()) {
                character.setBiography(json.get("biography").getAsString());
            }
            if (json.has("isDeceased")) {
                character.setDeceased(json.get("isDeceased").getAsBoolean());
            }
            if (json.has("firstUsed")) {
                character.setFirstUsed(json.get("firstUsed").getAsLong());
            }
            if (json.has("lastSeen")) {
                character.setLastSeen(json.get("lastSeen").getAsLong());
            }
            return character;
        }
        catch (Exception e) {
            LOGGER.error("[StatecraftAPI] Failed to parse character from JSON", (Throwable)e);
            return null;
        }
    }

    public boolean setPlayerPassword(UUID playerUuid, String passwordHash, String salt) {
        try {
            JsonObject requestBody = new JsonObject();
            requestBody.addProperty("uuid", playerUuid.toString());
            requestBody.addProperty("passwordHash", passwordHash);
            requestBody.addProperty("salt", salt);
            String response = this.executeRequest("POST", "/players/set-password", requestBody.toString());
            if (response == null) {
                LOGGER.error("[StatecraftAPI] Failed to set password for player {} - no response", (Object)playerUuid);
                return false;
            }
            try {
                JsonObject responseJson = (JsonObject)GSON.fromJson(response, JsonObject.class);
                if (responseJson.has("success") && responseJson.get("success").getAsBoolean()) {
                    LOGGER.info("[StatecraftAPI] Successfully set password for player {}", (Object)playerUuid);
                    return true;
                }
            }
            catch (Exception e) {
                LOGGER.debug("[StatecraftAPI] Response after setting password: {}", (Object)response);
            }
            LOGGER.info("[StatecraftAPI] Password set for player {}", (Object)playerUuid);
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[StatecraftAPI] Failed to set password for player {}", (Object)playerUuid, (Object)e);
            return false;
        }
    }

    public boolean testConnection() {
        String testUrl = this.baseUrl + "/players/uuid/00000000-0000-0000-0000-000000000000";
        LOGGER.debug("[StatecraftAPI] Testing connection to: {}", (Object)testUrl);
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(testUrl)).timeout(Duration.ofSeconds(5L)).header("X-API-Key", this.apiKey).GET().build();
            HttpResponse<String> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofString());
            LOGGER.debug("[StatecraftAPI] Connection test response: {} - {}", (Object)response.statusCode(), (Object)response.body());
            return response.statusCode() > 0;
        }
        catch (ConnectException e) {
            LOGGER.warn("[StatecraftAPI] Connection test failed - Cannot connect to {}: {}", (Object)testUrl, (Object)e.getMessage());
            return false;
        }
        catch (Exception e) {
            LOGGER.warn("[StatecraftAPI] Connection test failed for {}: {} - {}", new Object[]{testUrl, e.getClass().getSimpleName(), e.getMessage()});
            return false;
        }
    }
}

