/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.features.landfalladdon.decay;

import com.mojang.logging.LogUtils;
import java.util.Objects;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.entity.player.PlayerEvent;
import org.slf4j.Logger;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.features.aging.AgingManager;
import world.landfall.persona.features.landfalladdon.ActionBarManager;
import world.landfall.persona.features.landfalladdon.LandfallAddonData;
import world.landfall.persona.features.landfalladdon.decay.DecayStages;
import world.landfall.persona.features.landfalladdon.shells.Shell;
import world.landfall.persona.registry.PersonaEvents;
import world.landfall.persona.util.CharacterUtils;

@EventBusSubscriber(modid="persona")
public class DecayManager {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final int STABLE_THRESHOLD = 50;
    private static final int MILD_THRESHOLD = 65;
    private static final int MODERATE_THRESHOLD = 80;
    private static final int HIGH_THRESHOLD = 95;
    private static final int CRITICAL_THRESHOLD = 100;
    private static final double MAX_AGE_YEARS = 10000.0;
    private static final int MAX_DEATH_COUNT = 10000;
    private static final double AGE_DIVISOR = 10.0;

    public static int calculateDecayIndex(CharacterProfile profile) {
        Objects.requireNonNull(profile, "CharacterProfile cannot be null");
        try {
            int deathComponent;
            int decayIndex;
            double ageYears = DecayManager.getValidatedAge(profile);
            int ageComponent = (int)Math.floor(ageYears / 10.0);
            if (ageComponent < 0) {
                LOGGER.warn("Negative age component {} for character {}, using 0", (Object)ageComponent, (Object)profile.getDisplayName());
                ageComponent = 0;
            }
            if ((decayIndex = ageComponent + (deathComponent = DecayManager.getValidatedDeathCount(profile))) < 0) {
                LOGGER.error("Decay index overflow for character {} (age: {}, deaths: {}), using 0", new Object[]{profile.getDisplayName(), ageComponent, deathComponent});
                return 0;
            }
            LOGGER.debug("Calculated decay index {} for character {} (age component: {}, death component: {})", new Object[]{decayIndex, profile.getDisplayName(), ageComponent, deathComponent});
            return decayIndex;
        }
        catch (Exception e) {
            LOGGER.error("Error calculating decay index for character {}, returning 0", (Object)profile.getDisplayName(), (Object)e);
            return 0;
        }
    }

    public static DecayStages getStage(int index) {
        try {
            if (index < 0) {
                LOGGER.warn("Negative decay index {}, treating as 0", (Object)index);
                index = 0;
            }
            if (index < 50) {
                return DecayStages.STABLE;
            }
            if (index < 65) {
                return DecayStages.MILD;
            }
            if (index < 80) {
                return DecayStages.MODERATE;
            }
            if (index < 95) {
                return DecayStages.HIGH;
            }
            return DecayStages.SEVERE;
        }
        catch (Exception e) {
            LOGGER.error("Error determining decay stage for index {}, returning STABLE", (Object)index, (Object)e);
            return DecayStages.STABLE;
        }
    }

    public static boolean isCriticalDecay(int index) {
        return index >= 100;
    }

    private static void evaluateAndApply(ServerPlayer player, CharacterProfile profile) {
        Objects.requireNonNull(player, "ServerPlayer cannot be null");
        Objects.requireNonNull(profile, "CharacterProfile cannot be null");
        try {
            if (profile.isDeceased()) {
                LOGGER.debug("Character {} is already deceased, skipping decay evaluation", (Object)profile.getDisplayName());
                return;
            }
            int index = DecayManager.calculateDecayIndex(profile);
            DecayStages stage = DecayManager.getStage(index);
            LOGGER.debug("Evaluated decay for {}: index={} stage={}", new Object[]{profile.getDisplayName(), index, stage});
            DecayManager.updateActionBar(player, stage, profile);
            if (DecayManager.isCriticalDecay(index)) {
                DecayManager.handleCriticalDecay(player, profile, index);
            }
        }
        catch (Exception e) {
            LOGGER.error("Error evaluating decay for character {} (player: {})", new Object[]{profile.getDisplayName(), player.getName().getString(), e});
        }
    }

    private static void handleCriticalDecay(ServerPlayer player, CharacterProfile profile, int index) {
        Objects.requireNonNull(player, "ServerPlayer cannot be null");
        Objects.requireNonNull(profile, "CharacterProfile cannot be null");
        try {
            LOGGER.debug("Character {} reached critical decay ({}). Marking as deceased.", (Object)profile.getDisplayName(), (Object)index);
            CharacterUtils.setCharacterDeceased(player, profile.getId(), true);
        }
        catch (Exception e) {
            LOGGER.error("Error handling critical decay for character {} (index: {})", new Object[]{profile.getDisplayName(), index, e});
        }
    }

    private static void updateActionBar(ServerPlayer player, DecayStages stage, CharacterProfile profile) {
        try {
            Shell currentShell = LandfallAddonData.getCurrentShell(profile);
            ActionBarManager.updatePlayerStatus(player, stage, currentShell);
        }
        catch (Exception e) {
            LOGGER.error("Error updating action bar for player {} (character: {})", new Object[]{player.getName().getString(), profile.getDisplayName(), e});
        }
    }

    @SubscribeEvent
    public static void onPlayerLogin(PlayerEvent.PlayerLoggedInEvent event) {
        if (event == null) {
            LOGGER.error("Received null PlayerLoggedInEvent");
            return;
        }
        try {
            Player player = event.getEntity();
            if (!(player instanceof ServerPlayer)) {
                LOGGER.debug("PlayerLoggedInEvent entity is not a ServerPlayer, ignoring");
                return;
            }
            ServerPlayer player2 = (ServerPlayer)player;
            DecayManager.processPlayerLogin(player2);
        }
        catch (Exception e) {
            LOGGER.error("Unexpected error processing player login event", (Throwable)e);
        }
    }

    @SubscribeEvent
    public static void onCharacterSwitch(PersonaEvents.CharacterSwitchEvent event) {
        if (event == null) {
            LOGGER.error("Received null CharacterSwitchEvent");
            return;
        }
        try {
            Player player = event.getPlayer();
            if (!(player instanceof ServerPlayer)) {
                LOGGER.debug("CharacterSwitchEvent player is not a ServerPlayer, ignoring");
                return;
            }
            ServerPlayer player2 = (ServerPlayer)player;
            Long toId = event.getToCharacterId();
            if (toId == null) {
                LOGGER.debug("Character switch to null character for player {}", (Object)player2.getName().getString());
                return;
            }
            DecayManager.processCharacterSwitch(player2, toId);
        }
        catch (Exception e) {
            LOGGER.error("Unexpected error processing character switch event", (Throwable)e);
        }
    }

    private static void processPlayerLogin(ServerPlayer player) {
        Objects.requireNonNull(player, "Player cannot be null");
        try {
            PlayerCharacterData data = DecayManager.getPlayerCharacterData(player);
            if (data == null) {
                LOGGER.debug("No character data for player {} on login", (Object)player.getName().getString());
                return;
            }
            Long activeId = data.getActiveCharacterId();
            if (activeId == null) {
                LOGGER.debug("No active character for player {} on login", (Object)player.getName().getString());
                return;
            }
            CharacterProfile profile = data.getCharacter(activeId);
            if (profile == null) {
                LOGGER.warn("Active character {} not found for player {} on login", (Object)activeId, (Object)player.getName().getString());
                return;
            }
            DecayManager.evaluateAndApply(player, profile);
        }
        catch (Exception e) {
            LOGGER.error("Error processing login for player {}", (Object)player.getName().getString(), (Object)e);
        }
    }

    private static void processCharacterSwitch(ServerPlayer player, Long toId) {
        Objects.requireNonNull(player, "Player cannot be null");
        Objects.requireNonNull(toId, "Character ID cannot be null");
        try {
            PlayerCharacterData data = DecayManager.getPlayerCharacterData(player);
            if (data == null) {
                LOGGER.debug("No character data for player {} on character switch", (Object)player.getName().getString());
                return;
            }
            CharacterProfile profile = data.getCharacter(toId);
            if (profile == null) {
                LOGGER.warn("Character {} not found for player {} on switch", (Object)toId, (Object)player.getName().getString());
                return;
            }
            DecayManager.evaluateAndApply(player, profile);
        }
        catch (Exception e) {
            LOGGER.error("Error processing character switch for player {} to character {}", new Object[]{player.getName().getString(), toId, e});
        }
    }

    private static PlayerCharacterData getPlayerCharacterData(ServerPlayer player) {
        try {
            return (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
        }
        catch (Exception e) {
            LOGGER.error("Error getting character data for player {}", (Object)player.getName().getString(), (Object)e);
            return null;
        }
    }

    private static double getValidatedAge(CharacterProfile profile) {
        try {
            double ageYears = AgingManager.getCalculatedAge(profile);
            if (Double.isNaN(ageYears) || Double.isInfinite(ageYears)) {
                LOGGER.warn("Invalid age {} for character {}, using 0", (Object)ageYears, (Object)profile.getDisplayName());
                return 0.0;
            }
            if (ageYears < 0.0) {
                LOGGER.warn("Negative age {} for character {}, using 0", (Object)ageYears, (Object)profile.getDisplayName());
                return 0.0;
            }
            if (ageYears > 10000.0) {
                LOGGER.warn("Age {} exceeds maximum {} for character {}, clamping", new Object[]{ageYears, 10000.0, profile.getDisplayName()});
                return 10000.0;
            }
            return ageYears;
        }
        catch (Exception e) {
            LOGGER.error("Error getting age for character {}, using 0", (Object)profile.getDisplayName(), (Object)e);
            return 0.0;
        }
    }

    private static int getValidatedDeathCount(CharacterProfile profile) {
        try {
            int deathCount = LandfallAddonData.getDeathCount(profile);
            if (deathCount < 0) {
                LOGGER.warn("Negative death count {} for character {}, using 0", (Object)deathCount, (Object)profile.getDisplayName());
                return 0;
            }
            if (deathCount > 10000) {
                LOGGER.warn("Death count {} exceeds maximum {} for character {}, clamping", new Object[]{deathCount, 10000, profile.getDisplayName()});
                return 10000;
            }
            return deathCount;
        }
        catch (Exception e) {
            LOGGER.error("Error getting death count for character {}, using 0", (Object)profile.getDisplayName(), (Object)e);
            return 0;
        }
    }

    public static boolean validateConfiguration() {
        try {
            Class.forName("world.landfall.persona.features.aging.AgingManager");
            Class.forName("world.landfall.persona.features.landfalladdon.LandfallAddonData");
            Class.forName("world.landfall.persona.features.landfalladdon.ActionBarManager");
            Class.forName("world.landfall.persona.util.CharacterUtils");
            LOGGER.debug("DecayManager configuration validation passed");
            return true;
        }
        catch (ClassNotFoundException e) {
            LOGGER.error("DecayManager configuration validation failed - missing required class", (Throwable)e);
            return false;
        }
        catch (Exception e) {
            LOGGER.error("DecayManager configuration validation failed", (Throwable)e);
            return false;
        }
    }
}

