/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.statecraft;

import com.mojang.logging.LogUtils;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.bus.api.Event;
import net.neoforged.neoforge.common.NeoForge;
import org.slf4j.Logger;
import world.landfall.persona.config.Config;
import world.landfall.persona.data.CharacterFileStorage;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.registry.PersonaEvents;
import world.landfall.persona.registry.PersonaNetworking;
import world.landfall.persona.statecraft.StatecraftAPI;
import world.landfall.persona.statecraft.StatecraftCharacter;

public class SyncManager {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static SyncManager instance;
    private final ScheduledExecutorService scheduler;
    private final StatecraftAPI api;
    private MinecraftServer server;
    private volatile boolean isRunning = false;
    private final Map<Long, Long> lastSyncTimes = new ConcurrentHashMap<Long, Long>();
    private final ConcurrentHashMap<Long, ReentrantLock> characterLocks = new ConcurrentHashMap();

    private SyncManager() {
        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r);
            thread.setName("Statecraft-Sync-Thread");
            thread.setDaemon(true);
            return thread;
        });
        this.api = StatecraftAPI.getInstance();
    }

    public static synchronized SyncManager getInstance() {
        if (instance == null) {
            instance = new SyncManager();
        }
        return instance;
    }

    public void start(MinecraftServer server) {
        if (this.isRunning) {
            LOGGER.warn("[SyncManager] Already running, ignoring start request");
            return;
        }
        this.server = server;
        int syncInterval = (Integer)Config.STATECRAFT_SYNC_INTERVAL.get();
        if (syncInterval <= 0) {
            LOGGER.info("[SyncManager] Periodic sync disabled (interval = {})", (Object)syncInterval);
            return;
        }
        this.isRunning = true;
        this.scheduler.scheduleAtFixedRate(this::performSync, syncInterval, syncInterval, TimeUnit.SECONDS);
        LOGGER.info("[SyncManager] Started with {}s sync interval", (Object)syncInterval);
    }

    public void stop() {
        if (!this.isRunning) {
            return;
        }
        this.isRunning = false;
        this.performSync();
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
        LOGGER.info("[SyncManager] Stopped");
    }

    private void performSync() {
        if (this.server == null || !this.isRunning) {
            return;
        }
        try {
            ArrayList<Long> allStatecraftIds = new ArrayList<Long>();
            HashMap<Long, ServerPlayer> statecraftIdToPlayer = new HashMap<Long, ServerPlayer>();
            HashMap<Long, Long> statecraftIdToUuid = new HashMap<Long, Long>();
            for (ServerPlayer player : this.server.getPlayerList().getPlayers()) {
                PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
                if (data == null) continue;
                Map<Long, String> characterIds = data.getCharacterIds();
                for (Long characterId : characterIds.keySet()) {
                    if (characterId == null || characterId == 0L) continue;
                    allStatecraftIds.add(characterId);
                    statecraftIdToPlayer.put(characterId, player);
                    statecraftIdToUuid.put(characterId, characterId);
                }
            }
            if (allStatecraftIds.isEmpty()) {
                LOGGER.debug("[SyncManager] No characters with Statecraft IDs to sync");
                return;
            }
            LOGGER.debug("[SyncManager] Syncing {} characters", (Object)allStatecraftIds.size());
            HashSet<UUID> processedPlayers = new HashSet<UUID>();
            for (ServerPlayer player : this.server.getPlayerList().getPlayers()) {
                if (processedPlayers.contains(player.getUUID())) continue;
                processedPlayers.add(player.getUUID());
                PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
                if (data == null) continue;
                List<StatecraftCharacter> currentCharacters = this.api.getCharactersByPlayerUuid(player.getUUID());
                HashMap<Long, StatecraftCharacter> statecraftMap = new HashMap<Long, StatecraftCharacter>();
                for (StatecraftCharacter sc : currentCharacters) {
                    statecraftMap.put(sc.getCharacterId(), sc);
                }
                ConcurrentHashMap<Long, String> localCharacters = new ConcurrentHashMap<Long, String>(data.getCharacterIds());
                HashSet characterIdsCopy = new HashSet(localCharacters.keySet());
                for (Long characterId : characterIdsCopy) {
                    String characterName = (String)localCharacters.get(characterId);
                    if (characterId != null && characterId != 0L && statecraftMap.containsKey(characterId)) {
                        StatecraftCharacter statecraftChar = (StatecraftCharacter)statecraftMap.get(characterId);
                        this.updateCharacterFromStatecraft(player, characterId, statecraftChar);
                        this.lastSyncTimes.put(characterId, System.currentTimeMillis());
                        continue;
                    }
                    if (characterId == null || characterId == 0L) continue;
                    this.handleDeletedCharacter(player, characterId);
                }
            }
            LOGGER.debug("[SyncManager] Sync completed successfully");
            this.cleanupOldLocks();
        }
        catch (Exception e) {
            LOGGER.error("[SyncManager] Error during sync", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateCharacterFromStatecraft(ServerPlayer player, Long characterId, StatecraftCharacter statecraftChar) {
        ReentrantLock lock = this.characterLocks.computeIfAbsent(characterId, k -> new ReentrantLock());
        lock.lock();
        try {
            CharacterProfile profile;
            PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (data == null) {
                return;
            }
            String currentName = data.getCharacterIds().get(characterId);
            if (!statecraftChar.getDisplayName().equals(currentName)) {
                data.updateCharacterDisplayName(characterId, statecraftChar.getDisplayName());
                LOGGER.info("[SyncManager] Updated name for character {} from '{}' to '{}'", new Object[]{characterId, currentName, statecraftChar.getDisplayName()});
            }
            if ((profile = CharacterFileStorage.loadCharacter(characterId)) == null) {
                profile = new CharacterProfile(characterId, statecraftChar.getDisplayName(), false);
            }
            profile.setDisplayNameWithoutValidation(statecraftChar.getDisplayName());
            profile.setDeceased(statecraftChar.isDeceased());
            CharacterFileStorage.saveCharacter(profile);
            if (characterId.equals(data.getActiveCharacterId()) && statecraftChar.isDeceased()) {
                this.handleDeceasedCharacter(player, characterId);
            }
            PersonaNetworking.sendToPlayer(data, player);
        }
        catch (Exception e) {
            LOGGER.error("[SyncManager] Failed to update character {} for player {}", new Object[]{characterId, player.getName().getString(), e});
        }
        finally {
            lock.unlock();
        }
    }

    private void handleDeletedCharacter(ServerPlayer player, Long characterId) {
        try {
            PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (data == null) {
                return;
            }
            String characterName = data.getCharacterIds().get(characterId);
            LOGGER.warn("[SyncManager] Character {} ({}) no longer exists in Statecraft, removing", (Object)characterName, (Object)characterId);
            data.removeCharacter(characterId);
            if (characterId.equals(data.getActiveCharacterId())) {
                Map<Long, String> remainingCharacters = data.getCharacterIds();
                if (!remainingCharacters.isEmpty()) {
                    Long newActiveId = remainingCharacters.keySet().iterator().next();
                    data.setActiveCharacterId(newActiveId);
                    LOGGER.info("[SyncManager] Switched player {} to character {} after deletion", (Object)player.getName().getString(), (Object)newActiveId);
                } else {
                    data.setActiveCharacterId(null);
                    LOGGER.info("[SyncManager] Player {} has no remaining characters", (Object)player.getName().getString());
                }
            }
            PersonaNetworking.sendToPlayer(data, player);
        }
        catch (Exception e) {
            LOGGER.error("[SyncManager] Failed to handle deleted character {} for player {}", new Object[]{characterId, player.getName().getString(), e});
        }
    }

    private void handleDeceasedCharacter(ServerPlayer player, Long characterId) {
        try {
            PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (data == null) {
                return;
            }
            String characterName = data.getCharacterIds().get(characterId);
            LOGGER.info("[SyncManager] Character {} ({}) is deceased, switching player {} to another character", new Object[]{characterName, characterId, player.getName().getString()});
            Map<Long, String> allCharacters = data.getCharacterIds();
            Long newActiveId = null;
            for (Long id : allCharacters.keySet()) {
                CharacterProfile profile;
                if (id.equals(characterId) || (profile = data.getCharacter(id)) == null || profile.isDeceased()) continue;
                newActiveId = id;
                break;
            }
            if (newActiveId != null) {
                oldActiveId = data.getActiveCharacterId();
                data.setActiveCharacterId(newActiveId);
                LOGGER.info("[SyncManager] Switched player {} to character {}", (Object)player.getName().getString(), newActiveId);
                NeoForge.EVENT_BUS.post((Event)new PersonaEvents.CharacterSwitchEvent((Player)player, oldActiveId, newActiveId));
            } else {
                oldActiveId = data.getActiveCharacterId();
                data.setActiveCharacterId(null);
                LOGGER.warn("[SyncManager] Player {} has no living characters remaining", (Object)player.getName().getString());
                NeoForge.EVENT_BUS.post((Event)new PersonaEvents.CharacterSwitchEvent((Player)player, oldActiveId, null));
            }
            PersonaNetworking.sendToPlayer(data, player);
        }
        catch (Exception e) {
            LOGGER.error("[SyncManager] Failed to handle deceased character {} for player {}", new Object[]{characterId, player.getName().getString(), e});
        }
    }

    public void syncPlayer(ServerPlayer player) {
        if (!this.isRunning || player == null) {
            return;
        }
        this.scheduler.execute(() -> {
            try {
                PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
                if (data == null) {
                    return;
                }
                ArrayList<Long> statecraftIds = new ArrayList<Long>();
                HashMap<Long, Long> statecraftIdToUuid = new HashMap<Long, Long>();
                for (Long characterId : data.getCharacterIds().keySet()) {
                    if (characterId == null || characterId == 0L) continue;
                    statecraftIds.add(characterId);
                    statecraftIdToUuid.put(characterId, characterId);
                }
                if (statecraftIds.isEmpty()) {
                    return;
                }
                LOGGER.debug("[SyncManager] Immediate sync for player {}", (Object)player.getName().getString());
                List<StatecraftCharacter> currentCharacters = this.api.getCharactersByPlayerUuid(player.getUUID());
                HashMap<Long, StatecraftCharacter> statecraftMap = new HashMap<Long, StatecraftCharacter>();
                for (StatecraftCharacter statecraftCharacter : currentCharacters) {
                    statecraftMap.put(statecraftCharacter.getCharacterId(), statecraftCharacter);
                }
                for (Map.Entry entry : statecraftIdToUuid.entrySet()) {
                    Long statecraftId = (Long)entry.getKey();
                    Long characterId = (Long)entry.getValue();
                    if (!statecraftMap.containsKey(statecraftId)) continue;
                    StatecraftCharacter statecraftChar = (StatecraftCharacter)statecraftMap.get(statecraftId);
                    this.updateCharacterFromStatecraft(player, characterId, statecraftChar);
                    this.lastSyncTimes.put(statecraftId, System.currentTimeMillis());
                }
            }
            catch (Exception e) {
                LOGGER.error("[SyncManager] Failed to sync player {}", (Object)player.getName().getString(), (Object)e);
            }
        });
    }

    public void syncCharacter(Long characterId, Long statecraftId) {
        if (!this.isRunning || characterId == null || statecraftId == null) {
            return;
        }
        this.scheduler.execute(() -> {
            block4: {
                try {
                    StatecraftCharacter character = this.api.getCharacterById(statecraftId);
                    if (character == null) {
                        LOGGER.warn("[SyncManager] Character {} not found in Statecraft", (Object)statecraftId);
                        return;
                    }
                    if (this.server == null) break block4;
                    for (ServerPlayer player : this.server.getPlayerList().getPlayers()) {
                        PlayerCharacterData data = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
                        if (data == null || !data.hasCharacter(characterId)) continue;
                        this.updateCharacterFromStatecraft(player, characterId, character);
                        this.lastSyncTimes.put(statecraftId, System.currentTimeMillis());
                        break;
                    }
                }
                catch (Exception e) {
                    LOGGER.error("[SyncManager] Failed to sync character {} (statecraft: {})", new Object[]{characterId, statecraftId, e});
                }
            }
        });
    }

    public long getLastSyncTime(Long characterId) {
        return this.lastSyncTimes.getOrDefault(characterId, 0L);
    }

    private void cleanupOldLocks() {
        long currentTime = System.currentTimeMillis();
        long staleThreshold = 3600000L;
        this.lastSyncTimes.entrySet().removeIf(entry -> {
            ReentrantLock lock;
            Long characterId = (Long)entry.getKey();
            Long lastSync = (Long)entry.getValue();
            if (currentTime - lastSync > staleThreshold && (lock = this.characterLocks.get(characterId)) != null && !lock.isLocked()) {
                this.characterLocks.remove(characterId);
                return true;
            }
            return false;
        });
    }
}

