/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.features.landfalladdon.death;

import com.mojang.logging.LogUtils;
import java.util.Objects;
import javax.annotation.Nonnull;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.bus.api.SubscribeEvent;
import net.neoforged.fml.common.EventBusSubscriber;
import net.neoforged.neoforge.event.entity.player.PlayerEvent;
import org.slf4j.Logger;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.features.landfalladdon.ActionBarManager;
import world.landfall.persona.features.landfalladdon.LandfallAddonData;
import world.landfall.persona.features.landfalladdon.decay.DecayManager;
import world.landfall.persona.features.landfalladdon.decay.DecayStages;
import world.landfall.persona.features.landfalladdon.shells.Shell;
import world.landfall.persona.features.landfalladdon.shells.ShellManager;
import world.landfall.persona.util.CharacterUtils;

@EventBusSubscriber(modid="persona")
public final class DeathHandler {
    private static final Logger LOGGER = LogUtils.getLogger();

    private DeathHandler() {
        throw new UnsupportedOperationException("Utility class cannot be instantiated");
    }

    @SubscribeEvent
    public static void onPlayerDeath(@Nonnull PlayerEvent.PlayerRespawnEvent event) {
        Objects.requireNonNull(event, "PlayerRespawnEvent cannot be null");
        try {
            Player player = event.getEntity();
            if (!(player instanceof ServerPlayer)) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[DeathHandler] PlayerRespawnEvent entity is not a ServerPlayer, ignoring");
                }
                return;
            }
            ServerPlayer player2 = (ServerPlayer)player;
            DeathHandler.processPlayerDeath(player2);
            CharacterUtils.processPendingAutoSwitch(player2);
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Unexpected error processing player death event: {}", (Object)e.getMessage(), (Object)e);
        }
    }

    private static void processPlayerDeath(@Nonnull ServerPlayer player) {
        Objects.requireNonNull(player, "Player cannot be null");
        try {
            PlayerCharacterData data = DeathHandler.getPlayerCharacterData(player);
            if (data == null) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[DeathHandler] No character data found for player {}, skipping death processing", (Object)player.getName().getString());
                }
                return;
            }
            Long activeId = data.getActiveCharacterId();
            if (activeId == null) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[DeathHandler] No active character for player {}, skipping death processing", (Object)player.getName().getString());
                }
                return;
            }
            CharacterProfile currentProfile = data.getCharacter(activeId);
            if (currentProfile == null) {
                LOGGER.warn("[DeathHandler] Active character {} not found for player {}", (Object)activeId, (Object)player.getName().getString());
                return;
            }
            if (currentProfile.isDeceased()) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("[DeathHandler] Character '{}' is already deceased, skipping death processing", (Object)currentProfile.getDisplayName());
                }
                return;
            }
            DeathHandler.processCharacterDeath(player, currentProfile);
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error processing death for player {}: {}", new Object[]{player.getName().getString(), e.getMessage(), e});
        }
    }

    private static void processCharacterDeath(@Nonnull ServerPlayer player, @Nonnull CharacterProfile profile) {
        Objects.requireNonNull(player, "Player cannot be null");
        Objects.requireNonNull(profile, "Character profile cannot be null");
        try {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[DeathHandler] Processing death for character '{}' (player: {})", (Object)profile.getDisplayName(), (Object)player.getName().getString());
            }
            DeathHandler.incrementDeathCount(profile);
            String origin = DeathHandler.getOrigin(profile);
            Shell newShell = DeathHandler.assignNewShell(profile, origin);
            int deathCount = LandfallAddonData.getDeathCount(profile);
            LOGGER.info("[DeathHandler] Character '{}' died (Origin: {}, New Shell: {}, Total Deaths: {}) for player {}", new Object[]{profile.getDisplayName(), origin, newShell.name(), deathCount, player.getName().getString()});
            DeathHandler.updatePlayerActionBar(player, profile, newShell);
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error processing death for character '{}' (player: {}): {}", new Object[]{profile.getDisplayName(), player.getName().getString(), e.getMessage(), e});
        }
    }

    private static PlayerCharacterData getPlayerCharacterData(@Nonnull ServerPlayer player) {
        Objects.requireNonNull(player, "Player cannot be null");
        try {
            return (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error getting character data for player {}: {}", new Object[]{player.getName().getString(), e.getMessage(), e});
            return null;
        }
    }

    private static void incrementDeathCount(@Nonnull CharacterProfile profile) {
        Objects.requireNonNull(profile, "Character profile cannot be null");
        try {
            LandfallAddonData.incrementDeathCount(profile);
            if (LOGGER.isDebugEnabled()) {
                int newCount = LandfallAddonData.getDeathCount(profile);
                LOGGER.debug("[DeathHandler] Incremented death count for character '{}' to {}", (Object)profile.getDisplayName(), (Object)newCount);
            }
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Failed to increment death count for character '{}': {}", new Object[]{profile.getDisplayName(), e.getMessage(), e});
        }
    }

    @Nonnull
    private static String getOrigin(@Nonnull CharacterProfile profile) {
        Objects.requireNonNull(profile, "Character profile cannot be null");
        try {
            String origin = LandfallAddonData.getOrigin(profile);
            if (origin == null || origin.isBlank()) {
                LOGGER.warn("[DeathHandler] Character '{}' has null/blank origin, using UNKNOWN_ORIGIN", (Object)profile.getDisplayName());
                return "UNKNOWN_ORIGIN";
            }
            return origin;
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error getting origin for character '{}', using UNKNOWN_ORIGIN: {}", new Object[]{profile.getDisplayName(), e.getMessage(), e});
            return "UNKNOWN_ORIGIN";
        }
    }

    @Nonnull
    private static Shell assignNewShell(@Nonnull CharacterProfile profile, @Nonnull String origin) {
        Objects.requireNonNull(profile, "Profile cannot be null");
        Objects.requireNonNull(origin, "Origin cannot be null");
        try {
            Shell newShell = ShellManager.getRandomShell(origin);
            if (newShell == null) {
                LOGGER.error("[DeathHandler] ShellManager returned null shell for origin '{}', using NEUTRAL", (Object)origin);
                newShell = Shell.NEUTRAL;
            }
            LandfallAddonData.setCurrentShell(profile, newShell);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[DeathHandler] Assigned shell {} to character '{}' (origin: {})", new Object[]{newShell.name(), profile.getDisplayName(), origin});
            }
            return newShell;
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error assigning new shell for character '{}' with origin '{}', using NEUTRAL: {}", new Object[]{profile.getDisplayName(), origin, e.getMessage(), e});
            try {
                LandfallAddonData.setCurrentShell(profile, Shell.NEUTRAL);
                return Shell.NEUTRAL;
            }
            catch (Exception fallbackError) {
                LOGGER.error("[DeathHandler] Failed to set fallback NEUTRAL shell for character '{}': {}", new Object[]{profile.getDisplayName(), fallbackError.getMessage(), fallbackError});
                return Shell.NEUTRAL;
            }
        }
    }

    private static void updatePlayerActionBar(@Nonnull ServerPlayer player, @Nonnull CharacterProfile profile, @Nonnull Shell shell) {
        Objects.requireNonNull(player, "Player cannot be null");
        Objects.requireNonNull(profile, "Profile cannot be null");
        Objects.requireNonNull(shell, "Shell cannot be null");
        try {
            DecayStages decayStage = DecayManager.getStage(DecayManager.calculateDecayIndex(profile));
            ActionBarManager.updatePlayerStatus(player, decayStage, shell);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[DeathHandler] Updated action bar for player {} (character: '{}', shell: {}, decay: {})", new Object[]{player.getName().getString(), profile.getDisplayName(), shell.name(), decayStage.name()});
            }
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error updating action bar for player {} (character: '{}'): {}", new Object[]{player.getName().getString(), profile.getDisplayName(), e.getMessage(), e});
        }
    }

    public static boolean validateConfiguration() {
        try {
            Class.forName("world.landfall.persona.features.landfalladdon.LandfallAddonData");
            Class.forName("world.landfall.persona.features.landfalladdon.shells.ShellManager");
            Class.forName("world.landfall.persona.features.landfalladdon.decay.DecayManager");
            Class.forName("world.landfall.persona.features.landfalladdon.ActionBarManager");
            Class.forName("world.landfall.persona.util.CharacterUtils");
            LOGGER.info("[DeathHandler] Configuration validation passed");
            return true;
        }
        catch (ClassNotFoundException e) {
            LOGGER.error("[DeathHandler] Configuration validation failed - missing required class: {}", (Object)e.getMessage(), (Object)e);
            return false;
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Configuration validation failed: {}", (Object)e.getMessage(), (Object)e);
            return false;
        }
    }

    @Nonnull
    public static String getStatistics() {
        try {
            return String.format("[DeathHandler] Status: Active, Configuration: %s", DeathHandler.validateConfiguration() ? "Valid" : "Invalid");
        }
        catch (Exception e) {
            LOGGER.error("[DeathHandler] Error getting statistics: {}", (Object)e.getMessage(), (Object)e);
            return "[DeathHandler] Status: Error getting statistics";
        }
    }
}

