/*
 * Decompiled with CFR 0.152.
 */
package world.landfall.persona.util;

import com.mojang.logging.LogUtils;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nonnull;
import net.minecraft.network.chat.Component;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.player.Player;
import net.neoforged.bus.api.Event;
import net.neoforged.neoforge.common.NeoForge;
import org.slf4j.Logger;
import world.landfall.persona.data.CharacterProfile;
import world.landfall.persona.data.PlayerCharacterCapability;
import world.landfall.persona.data.PlayerCharacterData;
import world.landfall.persona.registry.PersonaEvents;
import world.landfall.persona.registry.PersonaNetworking;
import world.landfall.persona.util.CharacterSwitchCooldownManager;

public final class CharacterUtils {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final Map<UUID, Long> pendingAutoSwitches = new ConcurrentHashMap<UUID, Long>();

    private CharacterUtils() {
        throw new UnsupportedOperationException("Utility class cannot be instantiated");
    }

    public static boolean setCharacterDeceased(@Nonnull ServerPlayer player, @Nonnull Long characterId, boolean deceased) {
        return CharacterUtils.setCharacterDeceased(player, characterId, deceased, false);
    }

    public static boolean setCharacterDeceased(@Nonnull ServerPlayer player, @Nonnull Long characterId, boolean deceased, boolean delayAutoSwitch) {
        Objects.requireNonNull(player, "Player cannot be null");
        Objects.requireNonNull(characterId, "Character ID cannot be null");
        try {
            PlayerCharacterData characterData = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (characterData == null) {
                LOGGER.warn("[CharacterUtils] No character data found for player {}", (Object)player.getName().getString());
                return false;
            }
            CharacterProfile profile = characterData.getCharacter(characterId);
            if (profile == null) {
                LOGGER.warn("[CharacterUtils] Character {} not found for player {}", (Object)characterId, (Object)player.getName().getString());
                return false;
            }
            profile.setDeceased(deceased);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[CharacterUtils] Set deceased status for character {} ({}) to {} for player {}", new Object[]{profile.getDisplayName(), characterId, deceased, player.getName().getString()});
            }
            if (deceased && characterId.equals(characterData.getActiveCharacterId())) {
                if (delayAutoSwitch) {
                    Long previousPending = pendingAutoSwitches.put(player.getUUID(), characterId);
                    if (previousPending != null && LOGGER.isDebugEnabled()) {
                        LOGGER.debug("[CharacterUtils] Replaced pending auto-switch for player {} from {} to {}", new Object[]{player.getName().getString(), previousPending, characterId});
                    }
                    LOGGER.info("[CharacterUtils] Scheduled delayed auto-switch for player {} after character {} death", (Object)player.getName().getString(), (Object)profile.getDisplayName());
                } else {
                    CharacterUtils.handleDeceasedActiveCharacter(player, characterData, profile);
                }
            }
            return true;
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error setting deceased status for character {} (player: {}): {}", new Object[]{characterId, player.getName().getString(), e.getMessage(), e});
            return false;
        }
    }

    public static boolean setCharacterDeceasedByName(@Nonnull ServerPlayer player, @Nonnull String characterName, boolean deceased) {
        return CharacterUtils.setCharacterDeceasedByName(player, characterName, deceased, false);
    }

    public static boolean setCharacterDeceasedByName(@Nonnull ServerPlayer player, @Nonnull String characterName, boolean deceased, boolean delayAutoSwitch) {
        Objects.requireNonNull(player, "Player cannot be null");
        Objects.requireNonNull(characterName, "Character name cannot be null");
        if (characterName.trim().isEmpty()) {
            LOGGER.warn("[CharacterUtils] Character name is empty for player {}", (Object)player.getName().getString());
            return false;
        }
        try {
            PlayerCharacterData characterData = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (characterData == null) {
                LOGGER.warn("[CharacterUtils] No character data found for player {}", (Object)player.getName().getString());
                return false;
            }
            Optional<Long> characterIdOpt = characterData.getCharacterIds().entrySet().stream().filter(entry -> ((String)entry.getValue()).equalsIgnoreCase(characterName.trim())).map(Map.Entry::getKey).findFirst();
            if (characterIdOpt.isEmpty()) {
                LOGGER.warn("[CharacterUtils] Character '{}' not found for player {}", (Object)characterName, (Object)player.getName().getString());
                return false;
            }
            return CharacterUtils.setCharacterDeceased(player, characterIdOpt.get(), deceased, delayAutoSwitch);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error setting deceased status by name for character '{}' (player: {}): {}", new Object[]{characterName, player.getName().getString(), e.getMessage(), e});
            return false;
        }
    }

    public static void processPendingAutoSwitch(@Nonnull ServerPlayer player) {
        Objects.requireNonNull(player, "Player cannot be null");
        try {
            UUID playerId = player.getUUID();
            Long deceasedCharacterId = pendingAutoSwitches.remove(playerId);
            if (deceasedCharacterId == null) {
                return;
            }
            LOGGER.info("[CharacterUtils] Processing pending auto-switch for player {} after respawn", (Object)player.getName().getString());
            PlayerCharacterData characterData = (PlayerCharacterData)player.getData(PlayerCharacterCapability.CHARACTER_DATA);
            if (characterData == null) {
                LOGGER.error("[CharacterUtils] No character data found for player {} during pending auto-switch", (Object)player.getName().getString());
                return;
            }
            CharacterProfile deceasedProfile = characterData.getCharacter(deceasedCharacterId);
            if (deceasedProfile == null) {
                LOGGER.error("[CharacterUtils] Deceased character {} not found for player {} during pending auto-switch", (Object)deceasedCharacterId, (Object)player.getName().getString());
                return;
            }
            CharacterUtils.handleDeceasedActiveCharacter(player, characterData, deceasedProfile);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error processing pending auto-switch for player {}: {}", new Object[]{player.getName().getString(), e.getMessage(), e});
        }
    }

    public static void clearPendingAutoSwitch(@Nonnull UUID playerId) {
        Objects.requireNonNull(playerId, "Player ID cannot be null");
        try {
            Long removedCharacterId = pendingAutoSwitches.remove(playerId);
            if (removedCharacterId != null && LOGGER.isDebugEnabled()) {
                LOGGER.debug("[CharacterUtils] Cleared pending auto-switch for player {} (character: {})", (Object)playerId, (Object)removedCharacterId);
            }
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error clearing pending auto-switch for player {}: {}", new Object[]{playerId, e.getMessage(), e});
        }
    }

    public static boolean hasPendingAutoSwitch(@Nonnull UUID playerId) {
        Objects.requireNonNull(playerId, "Player ID cannot be null");
        try {
            return pendingAutoSwitches.containsKey(playerId);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error checking pending auto-switch for player {}: {}", new Object[]{playerId, e.getMessage(), e});
            return false;
        }
    }

    public static int getPendingAutoSwitchCount() {
        try {
            return pendingAutoSwitches.size();
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error getting pending auto-switch count: {}", (Object)e.getMessage(), (Object)e);
            return 0;
        }
    }

    public static void clearAllPendingAutoSwitches() {
        try {
            int clearedCount = pendingAutoSwitches.size();
            pendingAutoSwitches.clear();
            LOGGER.info("[CharacterUtils] Cleared all pending auto-switches ({} entries)", (Object)clearedCount);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error clearing all pending auto-switches: {}", (Object)e.getMessage(), (Object)e);
        }
    }

    private static void handleDeceasedActiveCharacter(@Nonnull ServerPlayer player, @Nonnull PlayerCharacterData characterData, @Nonnull CharacterProfile deceasedProfile) {
        Objects.requireNonNull(player, "Player cannot be null");
        Objects.requireNonNull(characterData, "Character data cannot be null");
        Objects.requireNonNull(deceasedProfile, "Deceased profile cannot be null");
        try {
            Long oldActiveCharacterId = characterData.getActiveCharacterId();
            Optional<Long> newActiveProfileId = characterData.getCharacterIds().keySet().stream().filter(Objects::nonNull).filter(id -> !id.equals(deceasedProfile.getId())).filter(id -> {
                CharacterProfile profile = characterData.getCharacter((Long)id);
                return profile != null && !profile.isDeceased();
            }).findFirst();
            if (newActiveProfileId.isPresent()) {
                CharacterProfile newProfile = characterData.getCharacter(newActiveProfileId.get());
                characterData.setActiveCharacterId(newProfile.getId());
                player.sendSystemMessage((Component)Component.translatable((String)"command.persona.info.auto_switched_deceased", (Object[])new Object[]{deceasedProfile.getDisplayName(), newProfile.getDisplayName()}));
                LOGGER.info("[CharacterUtils] Auto-switched player {} from deceased character '{}' to '{}'", new Object[]{player.getName().getString(), deceasedProfile.getDisplayName(), newProfile.getDisplayName()});
                NeoForge.EVENT_BUS.post((Event)new PersonaEvents.CharacterSwitchEvent((Player)player, oldActiveCharacterId, newProfile.getId()));
                CharacterSwitchCooldownManager.recordCharacterSwitch(player);
            } else {
                characterData.setActiveCharacterId(null);
                player.sendSystemMessage((Component)Component.translatable((String)"command.persona.info.auto_switched_deceased_no_available", (Object[])new Object[]{deceasedProfile.getDisplayName()}));
                LOGGER.info("[CharacterUtils] No available characters for auto-switch after '{}' died for player {}", (Object)deceasedProfile.getDisplayName(), (Object)player.getName().getString());
                NeoForge.EVENT_BUS.post((Event)new PersonaEvents.CharacterSwitchEvent((Player)player, oldActiveCharacterId, null));
            }
            PersonaNetworking.sendToPlayer(characterData, player);
        }
        catch (Exception e) {
            LOGGER.error("[CharacterUtils] Error handling deceased active character '{}' for player {}: {}", new Object[]{deceasedProfile.getDisplayName(), player.getName().getString(), e.getMessage(), e});
        }
    }
}

