#version 120

// ===== VERTEX SHADER FOR TERRAIN =====
// Smooth vegetation waving + correct block detection

// Input attributes
attribute vec4 mc_Entity;
attribute vec4 at_tangent;

// Uniforms from Minecraft
uniform mat4 gbufferModelView;
uniform mat4 gbufferProjection;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowProjection;
uniform mat4 shadowModelView;
uniform vec3 cameraPosition;
uniform float frameTimeCounter;

// Varying outputs to fragment shader
varying vec2 texcoord;
varying vec2 lmcoord;
varying vec4 glcolor;
varying vec3 normal;
varying vec3 worldPos;
varying float blockId;
varying vec4 shadowPos;

// ===== ANIMATION SETTINGS =====
#ifndef WAVING_SPEED_MULTIPLIER
    #define WAVING_SPEED_MULTIPLIER 2.0 // Default 2x speed multiplier
#endif

#ifndef WAVING_PLANTS
    #define WAVING_PLANTS 1 // Default: enabled
#endif

const float BASE_WAVE_SPEED = 3.0;      // Base speed before multiplier
const float WAVE_SPEED = BASE_WAVE_SPEED * WAVING_SPEED_MULTIPLIER;
const float GRASS_WAVE_STRENGTH = 0.12; // Visible but not excessive
const float LEAF_WAVE_STRENGTH = 0.08;  // Gentle leaf movement
const float CROP_WAVE_STRENGTH = 0.10;  // Visible crop movement

bool isVegetationBlock(float id) {
    // Simple vegetation detection - include more plants
    return (
        // Grass and small plants
        id == 31.0  || // Tall Grass
        id == 32.0  || // Dead Bush
        id == 175.0 || // Double Plants
        
        // Flowers
        id == 37.0  || id == 38.0  || // Poppy, Dandelion
        id == 39.0  || id == 40.0  || // Mushrooms
        
        // Leaves
        id == 18.0  || id == 161.0 || // Oak, Acacia Leaves
        (id >= 18.0 && id <= 21.0) || // All leaf types
        
        // Vines and plants
        id == 106.0 || // Vines
        id == 83.0  || // Sugar Cane
        
        // Crops
        id == 59.0  || id == 141.0 || id == 142.0 || // Wheat, Carrots, Potatoes
        id == 207.0 || // Beetroot
        
        // Saplings
        id == 6.0   || // Oak Sapling
        (id >= 6.0 && id <= 12.0) // All sapling types
    ) && 
    // EXCLUDE solid blocks
    id != 2.0 && id != 3.0 && id != 1.0 && id != 4.0 && id != 17.0;
}

void main() {
    vec4 position = gl_Vertex;
    texcoord = (gl_TextureMatrix[0] * gl_MultiTexCoord0).xy;
    lmcoord = (gl_TextureMatrix[1] * gl_MultiTexCoord1).xy;
    glcolor = gl_Color;
    blockId = mc_Entity.x;

    // Calculate world position
    vec4 worldPosition = gbufferModelViewInverse * (gl_ModelViewMatrix * position);
    worldPos = worldPosition.xyz;

    vec3 originalNormal = gl_Normal;

    // ===== Waving logic =====
    // Only specific vegetation blocks should wave if enabled
    bool shouldWave = WAVING_PLANTS == 1 && isVegetationBlock(blockId);
    
    if (shouldWave) {
        float time = frameTimeCounter * WAVE_SPEED;

        // Enhanced segment phase for smoother group movement
        float segmentPhase = (worldPos.x + worldPos.z) * 0.12;
        float finePhase = (worldPos.x * 0.3 + worldPos.z * 0.7) * 0.08;

        // BSL-style natural wind motion (much more subtle)
        float windX = sin(time + segmentPhase) * 0.6 
                    + sin(time * 0.8 + finePhase) * 0.3
                    + sin(time * 1.2 + segmentPhase * 0.7) * 0.1;
        
        float windZ = cos(time * 0.7 + segmentPhase) * 0.5 
                    + sin(time * 0.9 + finePhase) * 0.3
                    + cos(time * 1.1 + segmentPhase * 0.6) * 0.2;

        // Improved height-based movement - fixes close distance issue
        float heightFactor = clamp((position.y + 0.5), 0.1, 1.0); // Prevent zero factor
        heightFactor = smoothstep(0.1, 1.0, heightFactor); // Smoother transition
        
        // Distance-based factor to maintain waving at all distances
        vec3 viewPos = (gbufferModelView * position).xyz;
        float distanceFactor = 1.0 - clamp(length(viewPos) / 32.0, 0.0, 0.8);
        distanceFactor = max(distanceFactor, 0.3); // Always keep some movement

        // Choose wave strength based on plant type
        float waveStrength = GRASS_WAVE_STRENGTH;
        if (blockId == 18.0 || blockId == 161.0) waveStrength = LEAF_WAVE_STRENGTH;
        if (blockId == 59.0 || blockId == 141.0 || blockId == 142.0 || blockId == 207.0) waveStrength = CROP_WAVE_STRENGTH;

        // Apply enhanced waving with smoother motion
        float finalStrength = waveStrength * heightFactor * distanceFactor;
        
        position.x += windX * finalStrength;
        position.z += windZ * finalStrength;
        
        // Smoother vertical movement
        float verticalWave = sin(time * 1.2 + segmentPhase) * 0.3 
                           + sin(time * 0.8 + finePhase) * 0.2;
        position.y += verticalWave * finalStrength * 0.4;

        // Enhanced normal correction for better lighting
        vec3 displacementVector = vec3(windX, verticalWave * 0.4, windZ) * finalStrength;
        vec3 correctedNormal = normalize(originalNormal + displacementVector * 0.25);
        normal = normalize(gl_NormalMatrix * correctedNormal);
    } else {
        // Keep solid blocks perfectly still
        normal = normalize(gl_NormalMatrix * originalNormal);
    }

    // Calculate shadow position with improved stability
    vec4 shadowViewPos = shadowModelView * position;
    shadowPos = shadowProjection * shadowViewPos;
    
    // Prevent extreme values that can cause shadow glitches
    if (abs(shadowPos.w) < 0.001) {
        shadowPos.w = 0.001;
    }
    
    gl_Position = gbufferProjection * gbufferModelView * position;
}
