#version 120

// ===== FRAGMENT SHADER FOR TERRAIN =====
// Handles texturing with enhanced lighting and shadow effects

// Uniforms
uniform sampler2D texture;
uniform sampler2D lightmap;
uniform sampler2D shadowtex0;
uniform vec4 entityColor;
uniform float frameTimeCounter;
uniform int worldTime;
uniform vec3 sunPosition;
uniform mat4 shadowProjection;
uniform mat4 shadowModelView;

// Varying inputs from vertex shader
varying vec2 texcoord;
varying vec2 lmcoord;
varying vec4 glcolor;
varying vec3 normal;
varying vec3 worldPos;
varying float blockId;
varying vec4 shadowPos;

// ===== ENVIRONMENTAL LIGHTING SETTINGS =====
const float AMBIENT_STRENGTH = 0.25;     // Natural ambient lighting
const float SHADOW_SOFTNESS = 0.8;       // Soft, natural shadows
const float VEGETATION_BRIGHTNESS = 1.08; // Enhanced vegetation brightness
const float SUNLIGHT_BOOST = 1.06;       // Natural sunlight enhancement

// ===== ORE GLOW SETTINGS =====
const float ORE_GLOW_STRENGTH = 0.8;     // How bright ore glow is
const float ORE_PULSE_SPEED = 2.0;       // Speed of glow pulsing

// Function to detect ore blocks
bool isOreBlock(float id) {
    return (
        id == 56.0  || // Diamond Ore
        id == 14.0  || // Gold Ore
        id == 15.0  || // Iron Ore
        id == 129.0 || // Emerald Ore
        id == 21.0  || // Lapis Ore
        id == 73.0  || // Redstone Ore
        id == 74.0  || // Redstone Ore (lit)
        id == 153.0 || // Nether Quartz Ore
        id == 16.0  || // Coal Ore
        // Modern ore IDs (1.17+)
        id == 119.0 || // Copper Ore
        id == 120.0 || // Deepslate Copper Ore
        id == 121.0 || // Deepslate Diamond Ore
        id == 122.0 || // Deepslate Gold Ore
        id == 123.0 || // Deepslate Iron Ore
        id == 124.0 || // Deepslate Emerald Ore
        id == 125.0 || // Deepslate Lapis Ore
        id == 126.0 || // Deepslate Redstone Ore
        id == 127.0    // Deepslate Coal Ore
    );
}

// Get ore glow color based on block ID
vec3 getOreGlowColor(float id) {
    if (id == 56.0 || id == 121.0) return vec3(0.4, 0.8, 1.0);  // Diamond - cyan
    if (id == 14.0 || id == 122.0) return vec3(1.0, 0.8, 0.2);  // Gold - yellow
    if (id == 15.0 || id == 123.0) return vec3(0.8, 0.6, 0.4);  // Iron - orange
    if (id == 129.0 || id == 124.0) return vec3(0.2, 1.0, 0.3); // Emerald - green
    if (id == 119.0 || id == 120.0) return vec3(0.8, 0.4, 0.2); // Copper - orange-red
    if (id == 21.0 || id == 125.0) return vec3(0.2, 0.4, 1.0);  // Lapis - blue
    if (id == 73.0 || id == 74.0 || id == 126.0) return vec3(1.0, 0.2, 0.2); // Redstone - red
    if (id == 153.0) return vec3(1.0, 1.0, 0.8);                // Quartz - white
    if (id == 16.0 || id == 127.0) return vec3(0.3, 0.3, 0.3);  // Coal - dark gray
    return vec3(0.5, 0.5, 0.5); // Default glow
}

// Improved shadow sampling with reduced glitches
float getShadow() {
    if (shadowPos.w <= 0.0) return 1.0;
    
    vec3 projCoords = shadowPos.xyz / shadowPos.w;
    projCoords = projCoords * 0.5 + 0.5;
    
    // Check bounds more carefully
    if (projCoords.x < 0.0 || projCoords.x > 1.0 || 
        projCoords.y < 0.0 || projCoords.y > 1.0 ||
        projCoords.z < 0.0 || projCoords.z > 1.0) {
        return 1.0;
    }
    
    // Dynamic bias based on slope to reduce shadow acne
    float bias = 0.005 + 0.001 * abs(dot(normal, normalize(sunPosition)));
    
    // Sample shadow map with PCF for smoother shadows
    float shadowDepth = texture2D(shadowtex0, projCoords.xy).r;
    float currentDepth = projCoords.z - bias;
    
    // Simple shadow calculation with softer edges
    float shadow = (currentDepth <= shadowDepth) ? 1.0 : 0.85;
    
    // Add slight smoothing to reduce flickering
    shadow = mix(shadow, 1.0, 0.1);
    
    return shadow;
}



void main() {
    // Sample the base texture
    vec4 color = texture2D(texture, texcoord);
    
    // Fix for pink world border - better handling of edge cases
    // Check if color is near magenta (world border indicator)
    if (color.r > 0.8 && color.g < 0.2 && color.b > 0.8 && color.a > 0.9) {
        // Blend with sky color instead of making invisible
        color = vec4(0.5, 0.7, 1.0, 1.0) * 0.8; // Sky blue tint
    }
    
    // Apply vertex color (includes basic lighting)
    color *= glcolor;
    
    // ===== NATURAL ENVIRONMENTAL LIGHTING =====
    // Sample lightmap for basic lighting
    vec4 lightmapColor = texture2D(lightmap, lmcoord);
    
    // Calculate shadow factor
    float shadowFactor = getShadow();
    
    // Simple shadow application
    float sunlight = lightmapColor.g;
    if (sunlight > 0.4) {
        lightmapColor.g *= shadowFactor * SUNLIGHT_BOOST;
    }
    
    // Apply lightmap
    color *= lightmapColor;
    
    // Natural torch light (minimal to avoid conflicts)
    float torchLight = lightmapColor.r;
    if (torchLight > 0.3) {
        // Very subtle torch enhancement
        float flicker = 1.0 + sin(frameTimeCounter * 1.2) * 0.005;
        vec3 warmTint = vec3(0.02, 0.01, 0.005) * torchLight * flicker;
        color.rgb += warmTint;
    }
    
    // Enhanced ambient lighting for natural look
    float totalLight = lightmapColor.r + lightmapColor.g;
    if (totalLight < 0.4) {
        color.rgb += vec3(0.02, 0.024, 0.026) * AMBIENT_STRENGTH;
    }
    
    // ===== BALANCED LIGHTING FOR VEGETATION =====
    // Detect vegetation blocks
    bool isVegetation = (blockId == 31.0 || blockId == 175.0 || // Grass
                        (blockId >= 18.0 && blockId <= 21.0) || // Leaves
                        blockId == 59.0 || blockId == 141.0 || blockId == 142.0 || // Crops
                        (blockId >= 37.0 && blockId <= 40.0) || // Flowers
                        blockId == 6.0); // Saplings
    
    if (isVegetation) {
        // Moderate brightness boost to vegetation
        color.rgb *= VEGETATION_BRIGHTNESS;
        
        // Natural subsurface scattering effect
        float subsurface = max(0.0, dot(normal, normalize(sunPosition))) * 0.2;
        color.rgb += vec3(0.04, 0.06, 0.02) * subsurface;
        
        // Subtle green enhancement for grass/leaves
        if (blockId == 31.0 || blockId == 175.0 || (blockId >= 18.0 && blockId <= 21.0)) {
            color.g *= 1.03; // Subtle green boost
            color.r *= 1.01; // Very slight warm tint
        }
    }
    
    // ===== ORE GLOW EFFECTS (DISABLED FOR NOW) =====
    // Temporarily disabled to fix pink glow issue
    /*
    if (isOreBlock(blockId)) {
        vec3 oreGlowColor = getOreGlowColor(blockId);
        
        // Create pulsing effect
        float pulse = sin(frameTimeCounter * ORE_PULSE_SPEED) * 0.3 + 0.7;
        
        // Apply glow based on darkness (more glow in darker areas)
        float glowIntensity = (1.0 - totalLight) * ORE_GLOW_STRENGTH * pulse;
        glowIntensity = clamp(glowIntensity, 0.0, 0.3); // Reduced intensity
        
        // Add the glow effect
        color.rgb += oreGlowColor * glowIntensity;
        
        // Slight brightness boost for ores
        color.rgb *= 1.05;
    }
    */
    
    // ===== GENTLE LIGHTING ENHANCEMENTS =====
    // Very subtle shadow softening for very dark areas
    if (totalLight < 0.2) {
        color.rgb *= 0.95; // Slightly darken very dark areas
    }
    
    // Apply entity color if present (for colored blocks)
    color.rgb = mix(color.rgb, entityColor.rgb, entityColor.a);
    
    // Ensure alpha is preserved
    if (color.a < 0.1) discard;
    
    // Output color and normal data for composite pass
/* DRAWBUFFERS:0 */
    gl_FragData[0] = color;                           // Color to colortex0
}