#version 120

// ===== ENTITIES FRAGMENT SHADER =====
// Enhanced lighting for entities

uniform sampler2D lightmap;
uniform sampler2D texture;

varying vec2 lmcoord;
varying vec2 texcoord;
varying vec4 glcolor;
varying vec3 worldPos;
varying vec4 shadowPos;

uniform sampler2D shadowtex0;
uniform float shadowBias = 0.0001;

void main() {
    vec4 color = texture2D(texture, texcoord) * glcolor;
    
    // Fix for pink world border - better handling with sky blend
    if (color.r > 0.8 && color.g < 0.2 && color.b > 0.8 && color.a > 0.9) {
        color = vec4(0.5, 0.7, 1.0, 1.0) * 0.8; // Sky blue tint
    }
    
    color *= texture2D(lightmap, lmcoord);
    
    // Shadow quality settings
    #ifndef SHADOW_QUALITY
        #define SHADOW_QUALITY 1  // Default: Medium quality
    #endif
    
    // Improved shadow sampling with PCF (quality based)
    float shadow = 0.0;
    vec3 projCoords = shadowPos.xyz / shadowPos.w * 0.5 + 0.5;
    float currentDepth = projCoords.z;
    
    // Sample points based on quality setting
    int samples = 1;
    float radius = 0.0005;
    
    if (SHADOW_QUALITY == 0) { // Low quality - 4 samples
        samples = 1;
        radius = 0.001;
    } else if (SHADOW_QUALITY == 1) { // Medium quality - 16 samples
        samples = 2;
        radius = 0.0008;
    } else if (SHADOW_QUALITY == 2) { // High quality - 36 samples
        samples = 3;
        radius = 0.0006;
    }
    
    int totalSamples = 0;
    for (int x = -samples; x <= samples; x++) {
        for (int y = -samples; y <= samples; y++) {
            float pcfDepth = texture2D(shadowtex0, projCoords.xy + vec2(x, y) * radius).r;
            shadow += currentDepth - shadowBias > pcfDepth ? 0.0 : 1.0;
            totalSamples++;
        }
    }
    shadow /= float(totalSamples);
    
    // Apply shadow to entity lighting (reduce brightness in shadow)
    // Only apply shadows if ENTITY_SHADOWS is enabled
    #ifndef ENTITY_SHADOWS
        #define ENTITY_SHADOWS 0
    #endif
    
    if (ENTITY_SHADOWS == 1) {
        color.rgb *= mix(0.3, 1.0, shadow); // Darker in shadow, full in light
    } else {
        // When entity shadows are disabled, keep full brightness
        color.rgb *= 1.0;
    }
    
    // Slight brightness boost for entities to make them pop
    color.rgb *= 1.05;

/* DRAWBUFFERS:0 */
    gl_FragData[0] = color;
}