package org.dfood.util;

import net.minecraft.class_1747;
import net.minecraft.class_1792;
import net.minecraft.class_1802;
import net.minecraft.class_1814;
import net.minecraft.class_2246;
import net.minecraft.class_2248;
import net.minecraft.class_3612;
import net.minecraft.class_4174;
import net.minecraft.class_4176;
import net.minecraft.item.*;
import org.dfood.block.FoodBlocks;
import org.dfood.item.*;
import org.dfood.mixin.FoodToBlockMixin;

import java.util.HashMap;
import java.util.Map;

/**
 * 一个映射类，使开发者可以高度自定义新的Item实例
 *
 * <p>该类通过映射表将原版食物物品替换为对应的方块物品，实现食物到方块的转换功能。</p>
 *
 * <h2>关于延时注册</h2>
 * <p>延时注册机制是为了解决物品注册时的依赖关系问题而设计的。在原版Minecraft中，
 * 某些物品的注册存在依赖顺序，例如蜂蜜瓶({@code honey_bottle})的注册依赖于玻璃瓶({@code glass_bottle})，
 * 因为蜂蜜瓶的配方剩余物设置为玻璃瓶。</p>
 *
 * <p>在食物到方块的替换过程中，如果直接在静态初始化阶段创建所有物品实例，可能会遇到以下问题：</p>
 * <ul>
 *   <li>当替换蜂蜜瓶时，需要引用{@linkplain  class_1802#field_8469}作为配方剩余物</li>
 *   <li>但此时玻璃瓶可能尚未完成注册，导致{@linkplain  class_1802#field_8469}为null</li>
 *   <li>这会导致替换的蜂蜜瓶错误地绑定空值，引发游戏崩溃</li>
 * </ul>
 *
 * <p>延时注册通过延迟实例化的方式，仅在真正需要时才创建这些有依赖关系的物品，
 * 确保所有依赖项都已正确初始化，从而避免了空引用和崩溃问题。</p>
 *
 * <p>对于有类似依赖关系的物品，都应使用延时注册机制进行处理。</p>
 *
 * @see FoodToBlockMixin
 * @see #getItem(String)
 */
public class FoodToBlocks {
    /**
     * @apiNote 需要延时注册的物品请在此Map中预留位置，值设为null。
     */
    public static final Map<String, class_1792> FOOD_MAP = new HashMap<>();

    static {
        // 零食类
        FOOD_MAP.put("cookie", createItem(FoodBlocks.COOKIE, class_4176.field_18655));
        FOOD_MAP.put("apple", createItem(FoodBlocks.APPLE, class_4176.field_18638));
        FOOD_MAP.put("melon_slice", createItem(FoodBlocks.MELON_SLICE, class_4176.field_18660));
        FOOD_MAP.put("bread", createItem(FoodBlocks.BREAD, class_4176.field_18643));

        // 蔬菜类
        FOOD_MAP.put("beetroot", createItem(FoodBlocks.BEETROOT, class_4176.field_18641));
        FOOD_MAP.put("potato", new DoubleBlockItem(class_2246.field_10247, new class_1792.class_1793().method_19265(class_4176.field_18627), FoodBlocks.POTATO));
        FOOD_MAP.put("baked_potato", createItem(FoodBlocks.BAKED_POTATO, class_4176.field_18639));
        FOOD_MAP.put("carrot", new DoubleBlockItem(class_2246.field_10609, new class_1792.class_1793().method_19265(class_4176.field_18644), FoodBlocks.CARROT));
        FOOD_MAP.put("sweet_berries", new DoubleBlockItem(class_2246.field_16999, new class_1792.class_1793().method_19265(class_4176.field_18636), FoodBlocks.SWEET_BERRIES));
        FOOD_MAP.put("glow_berries", new DoubleBlockItem(class_2246.field_28675, new class_1792.class_1793().method_19265(class_4176.field_28647), FoodBlocks.GLOW_BERRIES));

        // 金制食物
        FOOD_MAP.put("golden_apple", createItem(FoodBlocks.GOLDEN_APPLE, class_4176.field_18658));
        FOOD_MAP.put("golden_carrot", createItem(FoodBlocks.GOLDEN_CARROT, class_4176.field_18659));
        FOOD_MAP.put("glistering_melon_slice", new class_1747(FoodBlocks.GLISTERING_MELON_SLICE, new class_1792.class_1793()));

        // 生熟肉类
        FOOD_MAP.put("chicken", createItem(FoodBlocks.CHICKEN, class_4176.field_18645));
        FOOD_MAP.put("cooked_chicken", createItem(FoodBlocks.COOKED_CHICKEN, class_4176.field_18649));
        FOOD_MAP.put("beef", createItem(FoodBlocks.BEEF, class_4176.field_18640));
        FOOD_MAP.put("cooked_beef", createItem(FoodBlocks.COOKED_BEEF, class_4176.field_18648));
        FOOD_MAP.put("mutton", createItem(FoodBlocks.MUTTON, class_4176.field_18662));
        FOOD_MAP.put("cooked_mutton", createItem(FoodBlocks.COOKED_MUTTON, class_4176.field_18651));
        FOOD_MAP.put("porkchop", createItem(FoodBlocks.PORKCHOP, class_4176.field_18626));
        FOOD_MAP.put("cooked_porkchop", createItem(FoodBlocks.COOKED_PORKCHOP, class_4176.field_18652));
        FOOD_MAP.put("rabbit", createItem(FoodBlocks.RABBIT, class_4176.field_18630));
        FOOD_MAP.put("cooked_rabbit", createItem(FoodBlocks.COOKED_RABBIT, class_4176.field_18653));

        // 鱼类
        FOOD_MAP.put("cod", createItem(FoodBlocks.COD, class_4176.field_18647));
        FOOD_MAP.put("cooked_cod", createItem(FoodBlocks.COOKED_COD, class_4176.field_18650));
        FOOD_MAP.put("salmon", createItem(FoodBlocks.SALMON, class_4176.field_18633));
        FOOD_MAP.put("cooked_salmon", createItem(FoodBlocks.COOKED_SALMON, class_4176.field_18654));
        FOOD_MAP.put("pufferfish", createItem(FoodBlocks.PUFFERFISH, class_4176.field_18628));

        // 炖菜类
        FOOD_MAP.put("rabbit_stew", new ModStewItem(FoodBlocks.RABBIT_STEW, new class_1792.class_1793().method_7889(1).method_19265(class_4176.field_18631)));
        FOOD_MAP.put("mushroom_stew", new ModStewItem(FoodBlocks.MUSHROOM_STEW, new class_1792.class_1793().method_7889(1).method_19265(class_4176.field_18661)));
        FOOD_MAP.put("beetroot_soup", new ModStewItem(FoodBlocks.BEETROOT_SOUP, new class_1792.class_1793().method_7889(1).method_19265(class_4176.field_18642)));
        FOOD_MAP.put("suspicious_stew", new ModSuspiciousStewItem(FoodBlocks.SUSPICIOUS_STEW, new class_1792.class_1793().method_7889(1).method_19265(class_4176.field_18635)));
        FOOD_MAP.put("bowl", new class_1747(FoodBlocks.BOWL, new class_1792.class_1793()));

        // 桶
        FOOD_MAP.put("bucket", new ModBucketItem(class_3612.field_15906, new class_1792.class_1793().method_7889(16), FoodBlocks.BUCKET));
        FOOD_MAP.put("water_bucket", null);
        FOOD_MAP.put("milk_bucket", null);
        FOOD_MAP.put("lava_bucket", null);

        // 其他
        FOOD_MAP.put("pumpkin_pie", createItem(FoodBlocks.PUMPKIN_PIE, class_4176.field_18629));
        FOOD_MAP.put("chorus_fruit", new ModChorusFruitItem(FoodBlocks.CHORUS_FRUIT,new class_1792.class_1793().method_19265(class_4176.field_18646)));
        FOOD_MAP.put("egg", new ModEggItem(FoodBlocks.EGG, new class_1792.class_1793()));
        FOOD_MAP.put("totem_of_undying", new class_1747(FoodBlocks.TOTEM_OF_UNDYING, new class_1792.class_1793().method_7889(1).method_7894(class_1814.field_8907)));

        // 药水类
        FOOD_MAP.put("potion", new ModPotionItem(FoodBlocks.POTION, new class_1792.class_1793().method_7889(1)));
        FOOD_MAP.put("glass_bottle", new ModGlassBottleItem(FoodBlocks.GLASS_BOTTLE, new class_1792.class_1793()));
        FOOD_MAP.put("honey_bottle", null);
    }

    public static class_1747 createItem(class_2248 foodBlock, class_4174 foodComponent) {
        return new class_1747(foodBlock, new class_1792.class_1793().method_19265(foodComponent));
    }

    /**
     * 通过id获取对应的物品。
     * @param id 要获取的物品
     * @return 对应的物品
     */
    public static class_1792 getItem(String id){
        class_1792 result = FOOD_MAP.get(id);
        if (result == null){
            return createDelayedItem(id);
        }
        return result;
    }

    /**
     * 创建需要延时注册的物品。
     * @param id 需要延时注册的物品id
     * @return 成功创建的物品，如果创建失败则返回null
     */
    private static class_1792 createDelayedItem(String id) {
        synchronized (FOOD_MAP) {
            class_1792 item = FOOD_MAP.get(id);
            if (item != null) {
                return item;
            }

            switch (id) {
                //桶
                case "milk_bucket":
                    item = new ModMilkBucketItem(FoodBlocks.MILK_BUCKET,
                            new class_1792.class_1793().method_7896(class_1802.field_8550).method_7889(1));
                    break;
                case "water_bucket":
                    item = new ModBucketItem(class_3612.field_15910,
                            new class_1792.class_1793().method_7896(class_1802.field_8550).method_7889(1), FoodBlocks.WATER_BUCKET);
                    break;
                case "lava_bucket":
                    item = new ModBucketItem(class_3612.field_15908,
                            new class_1792.class_1793().method_7896(class_1802.field_8550).method_7889(1), FoodBlocks.LAVA_BUCKET);
                    break;

                // 药水类
                case "honey_bottle":
                    item = new ModHoneyBottleItem(FoodBlocks.HONEY_BOTTLE,
                            new class_1792.class_1793().method_7896(class_1802.field_8469)
                                    .method_19265(class_4176.field_20381).method_7889(16));
                    break;

                default:
                    return null;
            }

            FOOD_MAP.put(id, item);
            return item;
        }
    }
}