/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book.page;

import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.book.BookTextHolder;
import com.klikli_dev.modonomicon.book.RenderedBookTextHolder;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.conditions.BookNoneCondition;
import com.klikli_dev.modonomicon.book.entries.BookContentEntry;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.util.BookGsonHelper;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import net.minecraft.class_10297;
import net.minecraft.class_10363;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1860;
import net.minecraft.class_1937;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3518;
import net.minecraft.class_5250;
import net.minecraft.class_5321;
import net.minecraft.class_7225;
import net.minecraft.class_7924;
import net.minecraft.class_9129;
import net.minecraft.class_9334;

public abstract class BookRecipePage<T extends class_1860<?>> extends BookPage {

    protected BookTextHolder title1;
    protected class_5321<class_1860<?>> recipeKey1;

    /**
     * Can be null during serverside construction. On both server and client it can be null if the recipe does not exist.
     */
    @Nullable
    protected class_10297 recipeDisplayEntry1;

    protected BookTextHolder title2;
    protected class_5321<class_1860<?>> recipeKey2;

    /**
     * Can be null during serverside construction. On both server and client it can be null if the recipe does not exist.
     */
    @Nullable
    protected class_10297 recipeDisplayEntry2;

    protected BookTextHolder text;

    public BookRecipePage(JsonDataHolder common) {
        this(common.title1(), common.recipeId1(), common.title2(), common.recipeId2(), common.text(), common.anchor(), common.condition());
    }

    public BookRecipePage(NetworkDataHolder common) {
        this(common.title1(), common.recipeKey1(), common.recipeDisplayEntry1(), common.title2(), common.recipeKey2(), common.recipeDisplayEntry2(), common.text(), common.anchor(), common.condition());
    }

    private BookRecipePage(BookTextHolder title1, class_5321<class_1860<?>> recipeKey1, BookTextHolder title2, class_5321<class_1860<?>> recipeKey2, BookTextHolder text, String anchor, BookCondition condition) {
        super(anchor, condition);
        this.title1 = title1;
        this.recipeKey1 = recipeKey1;
        this.title2 = title2;
        this.recipeKey2 = recipeKey2;
        this.text = text;
    }

    private BookRecipePage(BookTextHolder title1, class_5321<class_1860<?>> recipeKey1, @Nullable class_10297 recipeDisplayEntry1, BookTextHolder title2, class_5321<class_1860<?>> recipeKey2, @Nullable class_10297 recipeDisplayEntry2, BookTextHolder text, String anchor, BookCondition condition) {
        super(anchor, condition);
        this.title1 = title1;
        this.recipeKey1 = recipeKey1;
        this.recipeDisplayEntry1 = recipeDisplayEntry1;
        this.title2 = title2;
        this.recipeKey2 = recipeKey2;
        this.recipeDisplayEntry2 = recipeDisplayEntry2;
        this.text = text;

    }

    public static JsonDataHolder commonFromJson(class_2960 entryId, JsonObject json, class_7225.class_7874 provider) {
        var title1 = BookGsonHelper.getAsBookTextHolder(json, "title1", BookTextHolder.EMPTY, provider);
        class_2960 recipeId1 = json.has("recipe_id_1") ? class_2960.method_12829(class_3518.method_15265(json, "recipe_id_1")) : null;
        var recipeKey1 = recipeId1 != null ? class_5321.method_29179(class_7924.field_52178, recipeId1) : null;

        var title2 = BookGsonHelper.getAsBookTextHolder(json, "title2", BookTextHolder.EMPTY, provider);
        class_2960 recipeId2 = json.has("recipe_id_2") ? class_2960.method_12829(class_3518.method_15265(json, "recipe_id_2")) : null;
        var recipeKey2 = recipeId2 != null ? class_5321.method_29179(class_7924.field_52178, recipeId2) : null;

        var text = BookGsonHelper.getAsBookTextHolder(json, "text", BookTextHolder.EMPTY, provider);

        var anchor = class_3518.method_15253(json, "anchor", "");
        var condition = json.has("condition")
                ? BookCondition.fromJson(entryId, json.getAsJsonObject("condition"), provider)
                : new BookNoneCondition();

        return new JsonDataHolder(title1, recipeKey1, title2, recipeKey2, text, anchor, condition);
    }

    public static NetworkDataHolder commonFromNetwork(class_9129 buffer) {
        var title1 = BookTextHolder.fromNetwork(buffer);
        var recipeKey1 = buffer.readBoolean() ? buffer.method_44112(class_7924.field_52178) : null;
        var recipeDisplayEntry1 = buffer.readBoolean() ? class_10297.field_54663.decode(buffer) : null;

        var title2 = BookTextHolder.fromNetwork(buffer);
        var recipeKey2 = buffer.readBoolean() ? buffer.method_44112(class_7924.field_52178) : null;
        var recipeDisplayEntry2 = buffer.readBoolean() ? class_10297.field_54663.decode(buffer) : null;

        var text = BookTextHolder.fromNetwork(buffer);

        var anchor = buffer.method_19772();
        var condition = BookCondition.fromNetwork(buffer);

        return new NetworkDataHolder(title1, recipeKey1, recipeDisplayEntry1, title2, recipeKey2, recipeDisplayEntry2, text, anchor, condition);
    }

    public BookTextHolder getTitle1() {
        return this.title1;
    }

    public class_5321<class_1860<?>> getRecipeKey1() {
        return this.recipeKey1;
    }

    @Nullable
    public class_10297 getRecipeDisplayEntry1() {
        return this.recipeDisplayEntry1;
    }

    public BookTextHolder getTitle2() {
        return this.title2;
    }

    public class_5321<class_1860<?>> getRecipeKey2() {
        return this.recipeKey2;
    }

    @Nullable
    public class_10297 getRecipeDisplayEntry2() {
        return this.recipeDisplayEntry2;
    }

    public BookTextHolder getText() {
        return this.text;
    }

    protected class_1799 getRecipeOutput(class_1937 level, class_10297 recipeDisplayEntry) {
        if (recipeDisplayEntry == null) {
            var item = new class_1799(class_1802.field_8077);
            item.method_57379(class_9334.field_49631, class_2561.method_43470("Recipe not found, please check the logs."));
            return item;
        }

        var results = recipeDisplayEntry.method_64730(class_10363.method_65008(level));
        return results.stream().findFirst().orElse(class_1799.field_8037);
    }

    private class_10297 getRecipeDisplayEntry(class_3218 serverLevel, class_5321<class_1860<?>> key) {
        if (key == null) {
            return null;
        }

        var list = new ArrayList<class_10297>();
        serverLevel.method_64577().method_64679(key, list::add);
        var entry = list.stream().findFirst().orElse(null);

        if (entry == null) {
            Modonomicon.LOG.warn("Recipe {} not found.", key);
        }

        return entry;
    }

    @Override
    public void build(class_1937 level, BookContentEntry parentEntry, int pageNum) {
        super.build(level, parentEntry, pageNum);

        //TODO: handle multiple displays per recipe?

        //if we are on the server we have to load the recipe display info.
        //on the client we already get it in the constructor, sent from the server.
        if (level instanceof class_3218 serverLevel) {
            this.recipeDisplayEntry1 = this.getRecipeDisplayEntry(serverLevel, this.recipeKey1);
            this.recipeDisplayEntry2 = this.getRecipeDisplayEntry(serverLevel, this.recipeKey2);
        }

        if (this.recipeDisplayEntry1 == null && this.recipeDisplayEntry2 != null) {
            this.recipeDisplayEntry1 = this.recipeDisplayEntry2;
            this.recipeDisplayEntry2 = null;
        }

        if (this.title1.isEmpty()) {
            //use recipe title if we don't have a custom one
            this.title1 = new BookTextHolder(((class_5250) this.getRecipeOutput(level, this.recipeDisplayEntry1).method_7964())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getBook().getDefaultTitleColor())
                    ));
        }

        if (this.recipeDisplayEntry2 != null && this.title2.isEmpty()) {
            //use recipe title if we don't have a custom one
            this.title2 = new BookTextHolder(((class_5250) this.getRecipeOutput(level, this.recipeDisplayEntry2).method_7964())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getBook().getDefaultTitleColor())
                    ));
        }

        if (this.title1.equals(this.title2)) {
            this.title2 = BookTextHolder.EMPTY;
        }
    }

    @Override
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
        super.prerenderMarkdown(textRenderer);

        if (!this.title1.hasComponent()) {
            this.title1 = new BookTextHolder(class_2561.method_43471(this.title1.getKey())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getCategory().getBook().getDefaultTitleColor())));
        }
        if (!this.title2.hasComponent()) {
            this.title2 = new BookTextHolder(class_2561.method_43471(this.title2.getKey())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getCategory().getBook().getDefaultTitleColor())));
        }

        if (!this.text.hasComponent()) {
            this.text = new RenderedBookTextHolder(this.text, textRenderer.render(this.text.getString()));
        }
    }

    @Override
    public void toNetwork(class_9129 buffer) {
        this.title1.toNetwork(buffer);
        buffer.method_52964(this.recipeKey1 != null);
        if (this.recipeKey1 != null) {
            buffer.method_44116(this.recipeKey1);
        }
        buffer.method_52964(this.recipeDisplayEntry1 != null);
        if (this.recipeDisplayEntry1 != null) {
            class_10297.field_54663.encode(buffer, this.recipeDisplayEntry1);
        }

        this.title2.toNetwork(buffer);
        buffer.method_52964(this.recipeKey2 != null);
        if (this.recipeKey2 != null) {
            buffer.method_44116(this.recipeKey2);
        }
        buffer.method_52964(this.recipeDisplayEntry2 != null);
        if (this.recipeDisplayEntry2 != null) {
            class_10297.field_54663.encode(buffer, this.recipeDisplayEntry2);
        }

        this.text.toNetwork(buffer);

        super.toNetwork(buffer);
    }

    @Override
    public boolean matchesQuery(String query, class_1937 level) {
        return this.title1.getString().toLowerCase().contains(query)
                || this.title2.getString().toLowerCase().contains(query)
                || this.text.getString().toLowerCase().contains(query);
    }

    public record JsonDataHolder(BookTextHolder title1, class_5321<class_1860<?>> recipeId1, BookTextHolder title2,
                                 class_5321<class_1860<?>> recipeId2, BookTextHolder text, String anchor,
                                 BookCondition condition) {
    }

    public record NetworkDataHolder(BookTextHolder title1, class_5321<class_1860<?>> recipeKey1,
                                    class_10297 recipeDisplayEntry1, BookTextHolder title2,
                                    class_5321<class_1860<?>> recipeKey2, class_10297 recipeDisplayEntry2,
                                    BookTextHolder text, String anchor, BookCondition condition) {
    }
}
