/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 * SPDX-FileCopyrightText: 2021 Authors of Arcana
 *
 * SPDX-License-Identifier: MIT
 */
package com.klikli_dev.modonomicon.client.gui.book.node;

import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.book.Book;
import com.klikli_dev.modonomicon.book.BookCategory;
import com.klikli_dev.modonomicon.book.BookFrameOverlay;
import com.klikli_dev.modonomicon.bookstate.BookUnlockStateManager;
import com.klikli_dev.modonomicon.bookstate.visual.BookVisualState;
import com.klikli_dev.modonomicon.client.gui.BookGuiManager;
import com.klikli_dev.modonomicon.client.gui.book.BookAddress;
import com.klikli_dev.modonomicon.client.gui.book.BookParentScreen;
import com.klikli_dev.modonomicon.client.gui.book.BookScreenWithButtons;
import com.klikli_dev.modonomicon.client.gui.book.bookmarks.BookBookmarksScreen;
import com.klikli_dev.modonomicon.client.gui.book.button.*;
import com.klikli_dev.modonomicon.client.gui.book.search.BookSearchScreen;
import com.klikli_dev.modonomicon.item.ModonomiconItem;
import com.klikli_dev.modonomicon.networking.ClickReadAllButtonMessage;
import com.klikli_dev.modonomicon.networking.SaveBookStateMessage;
import com.klikli_dev.modonomicon.networking.SyncBookUnlockStatesMessage;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.klikli_dev.modonomicon.platform.Services;
import com.klikli_dev.modonomicon.util.GuiGraphicsExt;
import com.mojang.blaze3d.systems.RenderSystem;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.glfw.GLFW;

import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_10799;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_437;
import net.minecraft.class_7919;

public class BookParentNodeScreen extends class_437 implements BookParentScreen, BookScreenWithButtons {

    public static final int MAX_CATEGORY_BUTTONS = 13;

    private final Book book;
    private final List<BookCategory> categories;
    //TODO: make the frame thickness configurable in the book?
    private final int frameThicknessW = 14;
    private final int frameThicknessH = 14;
    //This allows the BookCategoryIndexOnNodeScreen to give us mouseX and Y coordinates when this screen is rendered on a lower layer and does not get  x/y coordinates.
    public int renderMouseXOverride = -1;
    public int renderMouseYOverride = -1;
    private BookCategoryNodeScreen currentCategoryNodeScreen;
    private boolean hasUnreadEntries;
    private boolean hasUnreadUnlockedEntries;

    private int categoryButtonRenderOffset = 0;

    public BookParentNodeScreen(Book book) {
        super(class_2561.method_43470(""));

        //somehow there are render calls before init(), leaving minecraft null
        this.field_22787 = class_310.method_1551();

        this.book = book;

        this.categories = book.getCategoriesSorted(); //we no longer handle category locking here, is done on init to be able to refresh on unlock
    }

    public class_310 getMinecraft() {
        return this.field_22787;
    }

    @Override
    public void onDisplay() {
        this.updateUnreadEntriesState();
    }

    protected void updateUnreadEntriesState() {
        //check if ANY entry is unread
        this.hasUnreadEntries = this.book.getEntries().values().stream().anyMatch(e -> !BookUnlockStateManager.get().isReadFor(this.field_22787.field_1724, e));

        //check if any currently unlocked entry is unread
        this.hasUnreadUnlockedEntries = this.book.getEntries().values().stream().anyMatch(e -> BookUnlockStateManager.get().isUnlockedFor(this.field_22787.field_1724, e) && !BookUnlockStateManager.get().isReadFor(this.field_22787.field_1724, e));
    }

    public BookCategoryNodeScreen getCurrentCategoryScreen() {
        return this.currentCategoryNodeScreen;
    }

    public void setCurrentCategoryScreen(BookCategoryNodeScreen currentCategoryNodeScreen) {
        this.currentCategoryNodeScreen = currentCategoryNodeScreen;
    }

    @Override
    public Book getBook() {
        return this.book;
    }

    @Override
    public void setTooltip(List<class_2561> tooltip) {
        //Just implemented for category scroll button, which will set its own tooltip
        //So we do nothing
    }

    public class_2960 getBookOverviewTexture() {
        return this.book.getBookOverviewTexture();
    }

    /**
     * gets the x coordinate of the inner area of the book frame
     */
    public int getInnerX() {
        return (this.field_22789 - this.getFrameWidth()) / 2 + this.frameThicknessW / 2;
    }

    /**
     * gets the y coordinate of the inner area of the book frame
     */
    public int getInnerY() {
        return (this.field_22790 - this.getFrameHeight()) / 2 + this.frameThicknessH / 2;
    }

    /**
     * gets the width of the inner area of the book frame
     */
    public int getInnerWidth() {
        return this.getFrameWidth() - this.frameThicknessW;
    }

    /**
     * gets the height of the inner area of the book frame
     */
    public int getInnerHeight() {
        return this.getFrameHeight() - this.frameThicknessH;
    }

    public int getFrameThicknessW() {
        return this.frameThicknessW;
    }

    public int getFrameThicknessH() {
        return this.frameThicknessH;
    }


    /**
     * Gets the outer width of the book frame
     */
    public int getFrameWidth() {
        //TODO: enable config frame width
        return this.field_22789 - 60;
    }

    /**
     * Gets the outer height of the book frame
     */
    public int getFrameHeight() {
        //TODO: enable config frame height
        return this.field_22790 - 20;
    }

    protected void renderFrame(class_332 guiGraphics) {
        int width = this.getFrameWidth();
        int height = this.getFrameHeight();
        int x = (this.field_22789 - width) / 2;
        int y = (this.field_22790 - height) / 2;

        //draw a resizeable border. Center parts of each side will be stretched
        //the exact border size mostly does not matter because the center is empty anyway, but 50 gives a lot of flexiblity
        GuiGraphicsExt.blitWithBorder(guiGraphics, class_10799.field_56883, this.book.getFrameTexture(), x, y, 0, 0, width, height, 140, 140, 50, 50, 50, 50);

        //now render overlays on top of that border to cover repeating elements
        this.renderFrameOverlay(guiGraphics, this.book.getTopFrameOverlay(), (x + (width / 2)), y);
        this.renderFrameOverlay(guiGraphics, this.book.getBottomFrameOverlay(), (x + (width / 2)), (y + height));
        this.renderFrameOverlay(guiGraphics, this.book.getLeftFrameOverlay(), x, y + (height / 2));
        this.renderFrameOverlay(guiGraphics, this.book.getRightFrameOverlay(), x + width, y + (height / 2));
    }

    protected void renderFrameOverlay(class_332 guiGraphics, BookFrameOverlay overlay, int x, int y) {
        if (overlay.getFrameWidth() > 0 && overlay.getFrameHeight() > 0) {
            guiGraphics.method_25290(class_10799.field_56883, overlay.getTexture(), overlay.getFrameX(x), overlay.getFrameY(y), overlay.getFrameU(), overlay.getFrameV(), overlay.getFrameWidth(), overlay.getFrameHeight(), 256, 256);
        }
    }

    protected void onBookCategoryButtonClick(CategoryButton button) {
        BookGuiManager.get().openCategory(button.getCategory(), BookAddress.defaultFor(button.getCategory().getBook()));
    }

    protected void onReadAllButtonClick(ReadAllButton button) {
        if (this.hasUnreadUnlockedEntries && !class_437.method_25442()) {
            Services.NETWORK.sendToServer(new ClickReadAllButtonMessage(this.book.getId(), false));
            this.hasUnreadUnlockedEntries = false;
        } else if (this.hasUnreadEntries && class_437.method_25442()) {
            Services.NETWORK.sendToServer(new ClickReadAllButtonMessage(this.book.getId(), true));
            this.hasUnreadEntries = false;
        }
    }

    protected boolean canSeeReadAllButton() {
        return this.hasUnreadEntries || this.hasUnreadUnlockedEntries;
    }

    @Override
    public boolean method_25402(double pMouseX, double pMouseY, int pButton) {
        //ignore return value, because we need our base class to handle dragging and such
        this.getCurrentCategoryScreen().mouseClicked(pMouseX, pMouseY, pButton);
        return super.method_25402(pMouseX, pMouseY, pButton);
    }

    @Override
    public boolean method_25403(double pMouseX, double pMouseY, int pButton, double pDragX, double pDragY) {
        return this.getCurrentCategoryScreen().mouseDragged(pMouseX, pMouseY, pButton, pDragX, pDragY);
    }


    @Override
    public boolean method_25401(double mouseX, double mouseY, double scrollX, double scrollY) {
        this.getCurrentCategoryScreen().zoom(scrollY);
        return super.method_25401(mouseX, mouseY, scrollX, scrollY);
    }


    @Override
    public void method_25394(class_332 guiGraphics, int pMouseX, int pMouseY, float pPartialTick) {
        if (this.renderMouseXOverride != -1 && this.renderMouseYOverride != -1) {
            pMouseX = this.renderMouseXOverride;
            pMouseY = this.renderMouseYOverride;
        }

        //not needed any more in 1.21.8+ because screen now calls it before render()
//        this.renderBackground(guiGraphics, pMouseX, pMouseY, pPartialTick);

        this.getCurrentCategoryScreen().renderBackground(guiGraphics);

        this.getCurrentCategoryScreen().render(guiGraphics, pMouseX, pMouseY, pPartialTick);

        this.renderFrame(guiGraphics);

        this.getCurrentCategoryScreen().renderEntryTooltips(guiGraphics, pMouseX, pMouseY, pPartialTick);

        //manually call the renderables like super does -> otherwise super renders the background again on top of our stuff
        for (var renderable : this.field_33816) {
            renderable.method_25394(guiGraphics, pMouseX, pMouseY, pPartialTick);
        }

        this.renderMouseXOverride = -1;
        this.renderMouseYOverride = -1;
    }

    @Override
    public boolean method_25404(int key, int scanCode, int modifiers) {
        //delegate key handling to the open category
        //this ensures the open category is saved by calling the right overload of onEsc
        if (this.getCurrentCategoryScreen().keyPressed(key, scanCode, modifiers)) {
            return true;
        }

        //This is unlikely to be reached as the category screen will already handle esc
        if (key == GLFW.GLFW_KEY_ESCAPE) {
            BookGuiManager.get().closeScreenStack(this);
            return true;
        }
        return super.method_25404(key, scanCode, modifiers);
    }

    @Override
    public void method_25419() {
        //do not call super - gui manager should handle gui removal
    }

    @Override
    public void loadState(BookVisualState state) {
        //currently nothing to save - open category is handled by gui manager
    }

    @Override
    public void saveState(BookVisualState state) {
        //currently nothing to save - open category is handled by gui manager
    }

    private void sendBookClosedPacket(net.minecraft.class_1268 hand) {
        Services.NETWORK.sendToServer(new com.klikli_dev.modonomicon.networking.BookClosedMessage(hand));
    }

    @Override
    public boolean method_25430(@Nullable class_2583 pStyle) {
        return super.method_25430(pStyle);
    }

    @Override
    public void onSyncBookUnlockStatesMessage(SyncBookUnlockStatesMessage message) {
        //this leads to re-init of the category buttons after a potential unlock
        this.method_41843();
        this.updateUnreadEntriesState();
    }

    @Override
    protected void method_25426() {
        super.method_25426();

        int buttonXOffset = -11;
        int buttonYOffset = 30 + this.getBook().getCategoryButtonYOffset();
        int buttonX = (this.field_22789 - this.getFrameWidth()) / 2 - this.getFrameThicknessW() + buttonXOffset;
        int buttonY = (this.field_22790 - this.getFrameHeight()) / 2 - this.getFrameThicknessH() + buttonYOffset;
        //calculate button width so it aligns with the outer edge of the frame
        int buttonWidth = (this.field_22789 - this.getFrameWidth()) / 2 + buttonXOffset + 6;
        int buttonHeight = 20;
        int buttonSpacing = 2;

        this.updateCategoryButtons(buttonX, buttonY, buttonWidth, buttonHeight, buttonSpacing);

        int readAllButtonX = this.getFrameWidth() + this.getFrameThicknessW() + ReadAllButton.WIDTH / 2 - 3; //(this.width - this.getFrameWidth()); // / 2 - this.getFrameThicknessW() + buttonXOffset;
        int readAllButtonYOffset = 30 + this.getBook().getReadAllButtonYOffset();

        int readAllButtonY = (this.field_22790 - this.getFrameHeight()) / 2 + ReadAllButton.HEIGHT / 2 + readAllButtonYOffset;

        var readAllButton = new ReadAllButton(this, readAllButtonX, readAllButtonY, () -> this.hasUnreadUnlockedEntries, //if we have unlocked entries that are not read -> blue
                this::canSeeReadAllButton, //display condition -> if we have any unlocked entries -> grey
                (b) -> this.onReadAllButtonClick((ReadAllButton) b));

        this.method_37063(readAllButton);


        int searchButtonXOffset = 7;
        int searchButtonYOffset = -30 + this.getBook().getSearchButtonYOffset();
        int searchButtonX = this.getFrameWidth() + this.getFrameThicknessW() + ReadAllButton.WIDTH / 2 + searchButtonXOffset;
        int searchButtonY = this.getFrameHeight() + this.getFrameThicknessH() - ReadAllButton.HEIGHT / 2 + searchButtonYOffset;
        int searchButtonWidth = 44; //width in png
        int scissorX = this.getFrameWidth() + this.getFrameThicknessW() * 2 + 2; //this is the render location of our frame so our search button never overlaps

        var searchButton = new SearchButton(this, searchButtonX, searchButtonY,
                scissorX,
                searchButtonWidth, buttonHeight,
                (b) -> this.onSearchButtonClick((SearchButton) b),
                class_7919.method_47407(class_2561.method_43471(ModonomiconConstants.I18n.Gui.OPEN_SEARCH)));
        this.method_37063(searchButton);

        searchButtonY -= buttonHeight + 2;

        var showBookmarksButton = new ShowBookmarksButton(this, searchButtonX, searchButtonY,
                scissorX,
                searchButtonWidth, buttonHeight,
                (b) -> this.onShowBookmarksButtonClick((ShowBookmarksButton) b),
                class_7919.method_47407(class_2561.method_43471(ModonomiconConstants.I18n.Gui.OPEN_BOOKMARKS)));
        this.method_37063(showBookmarksButton);
    }

    protected void updateCategoryButtons(int buttonX, int buttonY, int buttonWidth, int buttonHeight, int buttonSpacing) {
        int buttonCount = 0;
        for (int i = this.categoryButtonRenderOffset, size = this.categories.size(); i < size; i++) {
            if (buttonCount >= MAX_CATEGORY_BUTTONS) {
                break; // Stop adding buttons once we reach the maximum
            }
            if (this.categories.get(i).showCategoryButton() && BookUnlockStateManager.get().isUnlockedFor(this.field_22787.field_1724, this.categories.get(i))) {
                var button = new CategoryButton(this, this.categories.get(i),
                        buttonX, buttonY + (buttonHeight + buttonSpacing) * buttonCount, buttonWidth, buttonHeight,
                        (b) -> this.onBookCategoryButtonClick((CategoryButton) b),
                        class_7919.method_47407(class_2561.method_43471(this.categories.get(i).getName())));

                this.method_37063(button);
                buttonCount++;
            }
        }

        buttonX += 8;
        // Add the top scroll button if there are categories before the current offset
        if (this.categoryButtonRenderOffset > 0) {
            int topScrollButtonY = (this.field_22790 - this.getFrameHeight()) / 2 - 5;

            var topScrollButton = new CategoryScrollButton(this, buttonX, topScrollButtonY, false,
                    () -> this.categoryButtonRenderOffset > 0,
                    (b) -> {
                        this.categoryButtonRenderOffset--;
                        this.method_41843();
                    });
            this.method_37063(topScrollButton);
        }


        // Add the bottom scroll button if there are more categories than can be displayed
        if (this.categories.size() > this.categoryButtonRenderOffset + MAX_CATEGORY_BUTTONS) {
            int bottomScrollButtonY= this.getFrameHeight() + this.getFrameThicknessH() - ReadAllButton.HEIGHT / 2 - 3;

            var bottomScrollButton = new CategoryScrollButton(this, buttonX, bottomScrollButtonY, true,
                    () -> this.categories.size() > this.categoryButtonRenderOffset + MAX_CATEGORY_BUTTONS,
                    (b) -> {
                        this.categoryButtonRenderOffset++;
                        this.method_41843();
                    });
            this.method_37063(bottomScrollButton);
        }
    }

    protected void onSearchButtonClick(SearchButton button) {
        ClientServices.GUI.pushGuiLayer(new BookSearchScreen(this));
    }

    protected void onShowBookmarksButtonClick(ShowBookmarksButton button) {
        ClientServices.GUI.pushGuiLayer(new BookBookmarksScreen(this));
    }

    @Override
    public boolean method_25421() {
        return false;
    }
}
