/*
 * SPDX-FileCopyrightText: 2022 Authors of Patchouli
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.multiblock;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.multiblock.StateMatcher;
import com.klikli_dev.modonomicon.api.multiblock.TriPredicate;
import com.klikli_dev.modonomicon.data.LoaderRegistry;
import com.klikli_dev.modonomicon.multiblock.matcher.Matchers;
import com.mojang.datafixers.util.Pair;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1922;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2382;
import net.minecraft.class_2470;
import net.minecraft.class_2540;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7225;
import net.minecraft.class_9129;

public class DenseMultiblock extends AbstractMultiblock {

    public static final class_2960 TYPE = Modonomicon.loc("dense");

    private static final Gson GSON = new GsonBuilder().create();

    private final String[][] pattern;
    private final class_2382 size;
    /**
     * Keep only for serialization
     */
    private final Map<Character, StateMatcher> targets;
    private StateMatcher[][][] stateMatchers;

    public DenseMultiblock(String[][] pattern, Map<Character, StateMatcher> targets) {
        this.pattern = pattern;
        this.targets = targets;

        if (!targets.containsKey('_')) {
            targets.put('_', Matchers.ANY);
        }
        if (!targets.containsKey(' ')) {
            targets.put(' ', Matchers.AIR);
        }
        if (!targets.containsKey('0')) {
            targets.put('0', Matchers.AIR);
        }

        this.size = this.build(targets, getPatternDimensions(pattern));
    }

    public static DenseMultiblock fromNetwork(class_9129 buffer) {
        var symmetrical = buffer.readBoolean();
        var offX = buffer.method_10816();
        var offY = buffer.method_10816();
        var offZ = buffer.method_10816();
        var viewOffX = buffer.method_10816();
        var viewOffY = buffer.method_10816();
        var viewOffZ = buffer.method_10816();

        var sizeX = buffer.method_10816();
        var sizeY = buffer.method_10816();
        var pattern = new String[sizeY][sizeX];
        for (int y = 0; y < sizeY; y++) {
            for (int x = 0; x < sizeX; x++) {
                pattern[y][x] = buffer.method_19772();
            }
        }

        var targets = new Object2ObjectOpenHashMap<Character, StateMatcher>();
        var targetCount = buffer.method_10816();
        for (int i = 0; i < targetCount; i++) {
            var key = buffer.readChar();
            var type = buffer.method_10810();
            var stateMatcher = LoaderRegistry.getStateMatcherNetworkLoader(type).fromNetwork(buffer);
            targets.put(key, stateMatcher);
        }

        var multiblock = new DenseMultiblock(pattern, targets);
        multiblock.setSymmetrical(symmetrical);
        multiblock.setOffset(offX, offY, offZ);
        multiblock.setViewOffset(viewOffX, viewOffY, viewOffZ);
        return multiblock;
    }

    public static DenseMultiblock fromJson(JsonObject json, class_7225.class_7874 provider) {
        var pattern = GSON.fromJson(json.get("pattern"), String[][].class);

        var jsonMapping = class_3518.method_15296(json, "mapping");
        var mapping = mappingFromJson(jsonMapping, provider);

        var multiblock = new DenseMultiblock(pattern, mapping);
        return additionalPropertiesFromJson(multiblock, json);
    }

    private static class_2382 getPatternDimensions(String[][] pattern) {
        int expectedLenX = -1;
        int expectedLenZ = -1;
        for (String[] arr : pattern) {
            if (expectedLenX == -1) {
                expectedLenX = arr.length;
            }
            if (arr.length != expectedLenX) {
                throw new IllegalArgumentException("Inconsistent array length. Expected" + expectedLenX + ", got " + arr.length);
            }

            for (String s : arr) {
                if (expectedLenZ == -1) {
                    expectedLenZ = s.length();
                }
                if (s.length() != expectedLenZ) {
                    throw new IllegalArgumentException("Inconsistent array length. Expected" + expectedLenX + ", got " + s.length());
                }
            }
        }

        return new class_2382(expectedLenX, pattern.length, expectedLenZ);
    }

    private class_2382 build(Map<Character, StateMatcher> stateMap, class_2382 dimensions) {
        boolean foundCenter = false;

        this.stateMatchers = new StateMatcher[dimensions.method_10263()][dimensions.method_10264()][dimensions.method_10260()];
        for (int y = 0; y < dimensions.method_10264(); y++) {
            for (int x = 0; x < dimensions.method_10263(); x++) {
                for (int z = 0; z < dimensions.method_10260(); z++) {
                    char c = this.pattern[y][x].charAt(z);
                    if (!stateMap.containsKey(c)) {
                        throw new IllegalArgumentException("Character " + c + " isn't mapped");
                    }

                    StateMatcher matcher = stateMap.get(c);
                    if (c == '0') {
                        if (foundCenter) {
                            throw new IllegalArgumentException("A structure can't have two centers");
                        }
                        foundCenter = true;
                        this.offX = x;
                        this.offY = dimensions.method_10264() - y - 1;
                        this.offZ = z;
                        this.setViewOffset();
                    }

                    this.stateMatchers[x][dimensions.method_10264() - y - 1][z] = matcher;
                }
            }
        }

        if (!foundCenter) {
            throw new IllegalArgumentException("A structure can't have no center");
        }
        return dimensions;
    }

    @Override
    public class_2960 getType() {
        return TYPE;
    }

    @Override
    public Pair<class_2338, Collection<SimulateResult>> simulate(class_1937 level, class_2338 anchor, class_2470 rotation, boolean forView, boolean disableOffset) {
        class_2338 disp = forView
                ? new class_2338(-this.viewOffX, -this.viewOffY + 1, -this.viewOffZ).method_10070(rotation)
                : new class_2338(-this.offX, -this.offY, -this.offZ).method_10070(rotation);
        if (disableOffset)
            disp = class_2338.field_10980;

        // the local origin of this multiblock, in world coordinates
        class_2338 origin = anchor.method_10081(disp);
        List<SimulateResult> ret = new ArrayList<>();
        for (int x = 0; x < this.size.method_10263(); x++) {
            for (int y = 0; y < this.size.method_10264(); y++) {
                for (int z = 0; z < this.size.method_10260(); z++) {
                    class_2338 currDisp = new class_2338(x, y, z).method_10070(rotation);
                    class_2338 actionPos = origin.method_10081(currDisp);
                    char currC = this.pattern[y][x].charAt(z);
                    ret.add(new SimulateResultImpl(actionPos, this.stateMatchers[x][y][z], currC));
                }
            }
        }
        return Pair.of(origin, ret);
    }

    @Override
    public boolean test(class_1937 level, class_2338 start, int x, int y, int z, class_2470 rotation) {
        this.setLevel(level);
        if (x < 0 || y < 0 || z < 0 || x >= this.size.method_10263() || y >= this.size.method_10264() || z >= this.size.method_10260()) {
            return false;
        }
        class_2338 checkPos = start.method_10081(new class_2338(x, y, z).method_10070(AbstractMultiblock.fixHorizontal(rotation)));
        TriPredicate<class_1922, class_2338, class_2680> pred = this.stateMatchers[x][y][z].getStatePredicate();
        class_2680 state = level.method_8320(checkPos).method_26186(rotation);

        return pred.test(level, checkPos, state);
    }

    @Override
    public void toNetwork(class_2540 buffer) {
        buffer.method_52964(this.symmetrical);
        buffer.method_10804(this.offX);
        buffer.method_10804(this.offY);
        buffer.method_10804(this.offZ);
        buffer.method_10804(this.viewOffX);
        buffer.method_10804(this.viewOffY);
        buffer.method_10804(this.viewOffZ);

        buffer.method_10804(this.size.method_10263());
        buffer.method_10804(this.size.method_10264());
        for (int y = 0; y < this.size.method_10264(); y++) {
            for (int x = 0; x < this.size.method_10263(); x++) {
                buffer.method_10814(this.pattern[y][x]);
            }
        }

        buffer.method_10804(this.targets.size());
        for (var entry : this.targets.entrySet()) {
            buffer.method_53004(entry.getKey());
            buffer.method_10812(entry.getValue().getType());
            entry.getValue().toNetwork(buffer);
        }
    }

    @Override
    public class_2382 getSize() {
        return this.size;
    }

    @Override
    public class_2680 method_8320(class_2338 pos) {
        int x = pos.method_10263();
        int y = pos.method_10264();
        int z = pos.method_10260();
        if (x < 0 || y < 0 || z < 0 || x >= size.method_10263() || y >= size.method_10264() || z >= size.method_10260()) {
            return class_2246.field_10124.method_9564();
        }
        long ticks = level != null ? level.method_8510() : 0L;
        return stateMatchers[x][y][z].getDisplayedState(ticks);
    }
}
