/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 * SPDX-FileCopyrightText: 2021 Authors of Arcana
 *
 * SPDX-License-Identifier: MIT
 */
package com.klikli_dev.modonomicon.client.gui.book;

import I;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.book.Book;
import com.klikli_dev.modonomicon.book.BookCategory;
import com.klikli_dev.modonomicon.book.BookFrameOverlay;
import com.klikli_dev.modonomicon.bookstate.BookUnlockStateManager;
import com.klikli_dev.modonomicon.bookstate.BookVisualStateManager;
import com.klikli_dev.modonomicon.bookstate.visual.BookVisualState;
import com.klikli_dev.modonomicon.client.gui.BookGuiManager;
import com.klikli_dev.modonomicon.client.gui.book.button.CategoryButton;
import com.klikli_dev.modonomicon.client.gui.book.button.ReadAllButton;
import com.klikli_dev.modonomicon.client.gui.book.button.SearchButton;
import com.klikli_dev.modonomicon.item.ModonomiconItem;
import com.klikli_dev.modonomicon.networking.ClickReadAllButtonMessage;
import com.klikli_dev.modonomicon.networking.SaveBookStateMessage;
import com.klikli_dev.modonomicon.networking.SyncBookUnlockStatesMessage;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.klikli_dev.modonomicon.platform.Services;
import com.klikli_dev.modonomicon.util.GuiGraphicsExt;
import com.mojang.blaze3d.systems.RenderSystem;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_437;
import net.minecraft.class_746;
import net.minecraft.class_7919;

public class BookOverviewScreen extends class_437 {

    private final Book book;
    private final List<BookCategory> categories;
    private final List<BookCategoryScreen> categoryScreens;

    //TODO: make the frame thickness configurable in the book?
    private final int frameThicknessW = 14;
    private final int frameThicknessH = 14;

    private int currentCategory = 0;
    private boolean hasUnreadEntries;
    private boolean hasUnreadUnlockedEntries;

    public BookOverviewScreen(Book book) {
        super(class_2561.method_43470(""));

        //somehow there are render calls before init(), leaving minecraft null
        this.field_22787 = class_310.method_1551();

        this.book = book;

        this.categories = book.getCategoriesSorted(); //we no longer handle category locking here, is done on init to be able to refresh on unlock
        this.categoryScreens = this.categories.stream().map(c -> new BookCategoryScreen(this, c)).toList();
    }

    public class_310 getMinecraft() {
        return this.field_22787;
    }

    public void onDisplay() {
        this.loadBookState();

        this.updateUnreadEntriesState();

        var currentScreen = this.categoryScreens.get(this.currentCategory);
        currentScreen.onDisplay();
    }

    protected void updateUnreadEntriesState() {
        //check if ANY entry is unread
        this.hasUnreadEntries = this.book.getEntries().values().stream().anyMatch(e -> !BookUnlockStateManager.get().isReadFor(this.field_22787.field_1724, e));

        //check if any currently unlocked entry is unread
        this.hasUnreadUnlockedEntries = this.book.getEntries().values().stream().anyMatch(e -> BookUnlockStateManager.get().isUnlockedFor(this.field_22787.field_1724, e) && !BookUnlockStateManager.get().isReadFor(this.field_22787.field_1724, e));
    }

    public BookCategoryScreen getCurrentCategoryScreen() {
        return this.categoryScreens.get(this.currentCategory);
    }

    public int getCurrentCategory() {
        return this.currentCategory;
    }

    public Book getBook() {
        return this.book;
    }

    public class_2960 getBookOverviewTexture() {
        return this.book.getBookOverviewTexture();
    }

    /**
     * gets the x coordinate of the inner area of the book frame
     */
    public int getInnerX() {
        return (this.field_22789 - this.getFrameWidth()) / 2 + this.frameThicknessW / 2;
    }

    /**
     * gets the y coordinate of the inner area of the book frame
     */
    public int getInnerY() {
        return (this.field_22790 - this.getFrameHeight()) / 2 + this.frameThicknessH / 2;
    }

    /**
     * gets the width of the inner area of the book frame
     */
    public int getInnerWidth() {
        return this.getFrameWidth() - this.frameThicknessW;
    }

    /**
     * gets the height of the inner area of the book frame
     */
    public int getInnerHeight() {
        return this.getFrameHeight() - this.frameThicknessH;
    }

    public int getFrameThicknessW() {
        return this.frameThicknessW;
    }

    public int getFrameThicknessH() {
        return this.frameThicknessH;
    }

    public void changeCategory(BookCategory category) {
        if (category == null) {
            Modonomicon.LOG.warn("Tried to change to a null category in this book ({}).", this.book.getId());
        }

        int index = this.categories.indexOf(category);
        if (index != -1) {
            this.changeCategory(index);
        } else {
            Modonomicon.LOG.warn("Tried to change to a category ({}) that does not exist in this book ({}).", this.book.getId(), category.getId());
        }
    }

    public void changeCategory(int categoryIndex) {
        if (this.currentCategory == categoryIndex) {
            return; //this is an easy fix for #179, otherwise we have to rethink state tracking
        }

        var oldIndex = this.currentCategory;
        this.currentCategory = categoryIndex;
        this.onCategoryChanged(oldIndex, this.currentCategory);
    }

    public void onCategoryChanged(int oldIndex, int newIndex) {
        var oldScreen = this.categoryScreens.get(oldIndex);
        oldScreen.onClose();

        var newScreen = this.categoryScreens.get(newIndex);
        newScreen.onDisplay();

        //TODO: SFX for category change?
    }

    /**
     * Gets the outer width of the book frame
     */
    protected int getFrameWidth() {
        //TODO: enable config frame width
        return this.field_22789 - 60;
    }

    /**
     * Gets the outer height of the book frame
     */
    protected int getFrameHeight() {
        //TODO: enable config frame height
        return this.field_22790 - 20;
    }

    protected void renderFrame(class_332 guiGraphics) {
        int width = this.getFrameWidth();
        int height = this.getFrameHeight();
        int x = (this.field_22789 - width) / 2;
        int y = (this.field_22790 - height) / 2;

        //draw a resizeable border. Center parts of each side will be stretched
        //the exact border size mostly does not matter because the center is empty anyway, but 50 gives a lot of flexiblity
        RenderSystem.setShaderColor(1.0F, 1.0F, 1.0F, 1.0F);
        RenderSystem.enableBlend();
        GuiGraphicsExt.blitWithBorder(guiGraphics, this.book.getFrameTexture(), x, y, 0, 0, width, height, 140, 140, 50, 50, 50, 50);

        //now render overlays on top of that border to cover repeating elements
        this.renderFrameOverlay(guiGraphics, this.book.getTopFrameOverlay(), (x + (width / 2)), y);
        this.renderFrameOverlay(guiGraphics, this.book.getBottomFrameOverlay(), (x + (width / 2)), (y + height));
        this.renderFrameOverlay(guiGraphics, this.book.getLeftFrameOverlay(), x, y + (height / 2));
        this.renderFrameOverlay(guiGraphics, this.book.getRightFrameOverlay(), x + width, y + (height / 2));
    }

    protected void renderFrameOverlay(class_332 guiGraphics, BookFrameOverlay overlay, int x, int y) {
        if (overlay.getFrameWidth() > 0 && overlay.getFrameHeight() > 0) {
            guiGraphics.method_25302(overlay.getTexture(), overlay.getFrameX(x), overlay.getFrameY(y), overlay.getFrameU(), overlay.getFrameV(), overlay.getFrameWidth(), overlay.getFrameHeight());
        }
    }

    protected void onBookCategoryButtonClick(CategoryButton button) {
        this.changeCategory(button.getCategoryIndex());
    }

    protected void onReadAllButtonClick(ReadAllButton button) {
        if (this.hasUnreadUnlockedEntries && !class_437.method_25442()) {
            Services.NETWORK.sendToServer(new ClickReadAllButtonMessage(this.book.getId(), false));
            this.hasUnreadUnlockedEntries = false;
        } else if (this.hasUnreadEntries && class_437.method_25442()) {
            Services.NETWORK.sendToServer(new ClickReadAllButtonMessage(this.book.getId(), true));
            this.hasUnreadEntries = false;
        }
    }

    protected boolean canSeeReadAllButton() {
        return this.hasUnreadEntries || this.hasUnreadUnlockedEntries;
    }

    private void loadBookState() {
        var state = BookVisualStateManager.get().getBookStateFor(this.field_22787.field_1724, this.book);
        if (state != null) {
            if (state.openCategory != null) {
                var openCategory = this.book.getCategory(state.openCategory);
                if (openCategory != null) {
                    this.currentCategory = this.categories.indexOf(openCategory);
                }
            }
        }
    }

    @Override
    public boolean method_25402(double pMouseX, double pMouseY, int pButton) {
        //ignore return value, because we need our base class to handle dragging and such
        this.getCurrentCategoryScreen().mouseClicked(pMouseX, pMouseY, pButton);
        return super.method_25402(pMouseX, pMouseY, pButton);
    }

    @Override
    public boolean method_25403(double pMouseX, double pMouseY, int pButton, double pDragX, double pDragY) {
        return this.getCurrentCategoryScreen().mouseDragged(pMouseX, pMouseY, pButton, pDragX, pDragY);
    }

    @Override
    public boolean method_25401(double pMouseX, double pMouseY, double pDelta) {
        this.getCurrentCategoryScreen().zoom(pDelta);
        return super.method_25401(pMouseX, pMouseY, pDelta);
    }

    @Override
    public void method_25394(class_332 guiGraphics, int pMouseX, int pMouseY, float pPartialTick) {
        RenderSystem.disableDepthTest(); //guard against depth test being enabled by other rendering code, that would cause ui elements to vanish

        this.method_25420(guiGraphics);

        this.getCurrentCategoryScreen().renderBackground(guiGraphics);

        this.getCurrentCategoryScreen().render(guiGraphics, pMouseX, pMouseY, pPartialTick);

        this.renderFrame(guiGraphics);

        this.getCurrentCategoryScreen().renderEntryTooltips(guiGraphics, pMouseX, pMouseY, pPartialTick);

        //do super render last -> it does the widgets including especially the tooltips and we want those to go over the frame
        super.method_25394(guiGraphics, pMouseX, pMouseY, pPartialTick);
    }

    @Override
    public boolean method_25422() {
        return true;
    }

    @Override
    public void method_25419() {
        this.getCurrentCategoryScreen().onClose();
        Services.NETWORK.sendToServer(new SaveBookStateMessage(this.book, this.getCurrentCategoryScreen().getCategory().getId()));

        BookGuiManager.get().resetHistory();

        BookGuiManager.get().openOverviewScreen = null;

        // Find the ModonomiconItem in the player's hand and send a packet to the server
        var player = class_310.method_1551().field_1724;
        if (player != null) {

            //If the book in the main hand is this book, we send MAIN_HAND, otherwise OFF_HAND
            //That means if the book is in neither hand, we send OFF_HAND
            //that is fine, the server will then just not update the closed state nbt on any item.
            //this is for the case of a custom button opening the book gui while the book is not in hand
            if (ModonomiconItem.getBook(player.method_6047()).getId().equals(this.book.getId()))
                this.sendBookClosedPacket(net.minecraft.class_1268.field_5808);
            else
                this.sendBookClosedPacket(net.minecraft.class_1268.field_5810);
        }

        super.method_25419();
    }

    private void sendBookClosedPacket(net.minecraft.class_1268 hand) {
        Services.NETWORK.sendToServer(new com.klikli_dev.modonomicon.networking.BookClosedMessage(hand));
    }

    @Override
    public boolean method_25430(@Nullable class_2583 pStyle) {
        return super.method_25430(pStyle);
    }

    public void onSyncBookUnlockCapabilityMessage(SyncBookUnlockStatesMessage message) {
        //this leads to re-init of the category buttons after a potential unlock
        this.method_41843();
        this.updateUnreadEntriesState();
    }


    @Override
    protected void method_25426() {
        super.method_25426();

        BookGuiManager.get().openOverviewScreen = this;

        int buttonXOffset = -11;


        int buttonYOffset = 30 + this.getBook().getCategoryButtonYOffset();

        int buttonX = (this.field_22789 - this.getFrameWidth()) / 2 - this.getFrameThicknessW() + buttonXOffset;
        int buttonY = (this.field_22790 - this.getFrameHeight()) / 2 - this.getFrameThicknessH() + buttonYOffset;
        //calculate button width so it aligns with the outer edge of the frame
        int buttonWidth = (this.field_22789 - this.getFrameWidth()) / 2 + buttonXOffset + 6;
        int buttonHeight = 20;
        int buttonSpacing = 2;

        int buttonCount = 0;
        for (int i = 0, size = this.categories.size(); i < size; i++) {
            if (this.categories.get(i).showCategoryButton() && BookUnlockStateManager.get().isUnlockedFor(this.field_22787.field_1724, this.categories.get(i))) {
                var button = new CategoryButton(this, this.categories.get(i), i, buttonX, buttonY + (buttonHeight + buttonSpacing) * buttonCount, buttonWidth, buttonHeight, (b) -> this.onBookCategoryButtonClick((CategoryButton) b), class_7919.method_47407(class_2561.method_43471(this.categories.get(i).getName())));

                this.method_37063(button);
                buttonCount++;
            }
        }

        int readAllButtonX = this.getFrameWidth() + this.getFrameThicknessW() + ReadAllButton.WIDTH / 2 - 3; //(this.width - this.getFrameWidth()); // / 2 - this.getFrameThicknessW() + buttonXOffset;
        int readAllButtonYOffset = 30 + this.getBook().getReadAllButtonYOffset();

        int readAllButtonY = (this.field_22790 - this.getFrameHeight()) / 2 + ReadAllButton.HEIGHT / 2 + readAllButtonYOffset;

        var readAllButton = new ReadAllButton(this, readAllButtonX, readAllButtonY, () -> this.hasUnreadUnlockedEntries, //if we have unlocked entries that are not read -> blue
                this::canSeeReadAllButton, //display condition -> if we have any unlocked entries -> grey
                (b) -> this.onReadAllButtonClick((ReadAllButton) b));

        this.method_37063(readAllButton);


        int searchButtonXOffset = 7;
        int searchButtonYOffset = -30 + this.getBook().getSearchButtonYOffset();
        int searchButtonX = this.getFrameWidth() + this.getFrameThicknessW() + ReadAllButton.WIDTH / 2 + searchButtonXOffset;
        int searchButtonY = this.getFrameHeight() + this.getFrameThicknessH() - ReadAllButton.HEIGHT / 2 + searchButtonYOffset;
        int searchButtonWidth = 44; //width in png
        int scissorX = this.getFrameWidth() + this.getFrameThicknessW() * 2 + 2; //this is the render location of our frame so our search button never overlaps

        var searchButton = new SearchButton(this, searchButtonX, searchButtonY, scissorX, searchButtonWidth, buttonHeight, (b) -> this.onSearchButtonClick((SearchButton) b), class_7919.method_47407(class_2561.method_43471(ModonomiconConstants.I18n.Gui.OPEN_SEARCH)));

        this.method_37063(searchButton);
    }

    protected void onSearchButtonClick(SearchButton button) {
        ClientServices.GUI.pushGuiLayer(new BookSearchScreen(this));
    }

    @Override
    public boolean method_25421() {
        return false;
    }
}
