/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */


package com.klikli_dev.modonomicon.api.datagen;

import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.api.datagen.book.BookCategoryModel;
import com.klikli_dev.modonomicon.api.datagen.book.BookCommandModel;
import com.klikli_dev.modonomicon.api.datagen.book.BookEntryModel;
import com.klikli_dev.modonomicon.api.datagen.book.BookModel;
import com.mojang.logging.LogUtils;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.slf4j.Logger;

import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collector;
import java.util.stream.Collector.Characteristics;
import java.util.stream.Stream;
import net.minecraft.class_2405;
import net.minecraft.class_2960;
import net.minecraft.class_7403;
import net.minecraft.class_7784;

public abstract class BookProvider implements class_2405 {

    protected static final Logger LOGGER = LogUtils.getLogger();
    
    public static final Collector<ModonomiconLanguageProvider, ?, Object2ObjectOpenHashMap<String, ModonomiconLanguageProvider>> mapMaker
            = Collector.<ModonomiconLanguageProvider, Object2ObjectOpenHashMap<String, ModonomiconLanguageProvider>, Object2ObjectOpenHashMap<String, ModonomiconLanguageProvider>>of(
                    Object2ObjectOpenHashMap::new,
                    (map, l) -> map.put(l.locale(), l),
                    (m1, m2) -> { m1.putAll(m2); return m1; },
                    (map) -> { map.trim(); return map; },
                    Characteristics.UNORDERED);

    protected final class_7784 packOutput;
    protected final ModonomiconLanguageProvider lang;
    protected final Map<String, ModonomiconLanguageProvider> translations;
    protected final Map<class_2960, BookModel> bookModels;
    protected final String modid;
    protected String bookId;
    protected BookContextHelper context;

    protected Map<String, String> defaultMacros;

    protected ConditionHelper conditionHelper;

    /**
     * @param defaultLang The LanguageProvider to fill with this book provider. IMPORTANT: the Languag Provider needs to be added to the DataGenerator AFTER the BookProvider.
     */
    public BookProvider(String bookId, class_7784 packOutput, String modid, ModonomiconLanguageProvider defaultLang, ModonomiconLanguageProvider... translations) {
        this.modid = modid;
        this.packOutput = packOutput;
        this.lang = defaultLang;
        this.bookModels = new Object2ObjectOpenHashMap<>();
        this.translations = Stream.concat(Arrays.stream(translations), Stream.of(defaultLang))
                                  .collect(mapMaker);

        this.bookId = bookId;
        this.context = new BookContextHelper(this.modid);
        this.defaultMacros = new Object2ObjectOpenHashMap<>();
        this.conditionHelper = new ConditionHelper();
    }

    protected ModonomiconLanguageProvider lang() {
        return this.lang;
    }

    protected ModonomiconLanguageProvider lang(String locale) {
        return this.translations.get(locale);
    }

    public String bookId() {
        return this.bookId;
    }

    protected BookContextHelper context() {
        return this.context;
    }

    protected ConditionHelper condition() {
        return this.conditionHelper;
    }

    /**
     * Call registerMacro() here to make macros (= simple string.replace() of macro -> value) available to all category providers of this book.
     */
    protected abstract void registerDefaultMacros();

    /**
     * Override this to generate your book.
     * Each BookProvider should generate only one book.
     * Context already is set to the book id provided in the constructor.
     */
    protected abstract BookModel generateBook();

    /**
     * Register a macro (= simple string.replace() of macro -> value) to be used in all category providers of this book.
     */
    protected void registerDefaultMacro(String macro, String value) {
        this.defaultMacros.put(macro, value);
    }

    /**
     * Get the default macros (= simple string.replace() of macro -> value) to be used in all category providers of this book.
     */
    protected Map<String, String> defaultMacros() {
        return this.defaultMacros;
    }

    /**
     * Only override if you know what you are doing.
     * Generally you should not.
     */
    protected void generate() {
        this.context.book(this.bookId);
        this.add(this.generateBook());
    }

    protected class_2960 modLoc(String name) {
        return new class_2960(this.modid, name);
    }

    protected BookModel add(BookModel bookModel) {
        if (this.bookModels.containsKey(bookModel.getId()))
            throw new IllegalStateException("Duplicate book " + bookModel.getId());
        this.bookModels.put(bookModel.getId(), bookModel);
        return bookModel;
    }

    protected Path getPath(Path dataFolder, BookModel bookModel) {
        class_2960 id = bookModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(id.method_12832() + "/book.json");
    }

    protected Path getPath(Path dataFolder, BookCategoryModel bookCategoryModel) {
        class_2960 id = bookCategoryModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(bookCategoryModel.getBook().getId().method_12832())
                .resolve("categories")
                .resolve(id.method_12832() + ".json");
    }

    protected Path getPath(Path dataFolder, BookCommandModel bookCommandModel) {
        class_2960 id = bookCommandModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(bookCommandModel.getBook().getId().method_12832())
                .resolve("commands")
                .resolve(id.method_12832() + ".json");
    }

    protected Path getPath(Path dataFolder, BookEntryModel bookEntryModel) {
        class_2960 id = bookEntryModel.getId();
        return dataFolder
                .resolve(id.method_12836())
                .resolve(ModonomiconConstants.Data.MODONOMICON_DATA_PATH)
                .resolve(bookEntryModel.getCategory().getBook().getId().method_12832())
                .resolve("entries")
                .resolve(id.method_12832() + ".json");
    }

    @Override
    public CompletableFuture<?> method_10319(class_7403 cache) {

        List<CompletableFuture<?>> futures = new ArrayList<>();

        Path dataFolder = this.packOutput.method_45972(class_7784.class_7490.field_39367);

        this.registerDefaultMacros();
        this.generate();

        for (var bookModel : this.bookModels.values()) {
            Path bookPath = this.getPath(dataFolder, bookModel);
            futures.add(class_2405.method_10320(cache, bookModel.toJson(), bookPath));

            for (var bookCategoryModel : bookModel.getCategories()) {
                Path bookCategoryPath = this.getPath(dataFolder, bookCategoryModel);
                futures.add(class_2405.method_10320(cache, bookCategoryModel.toJson(), bookCategoryPath));

                for (var bookEntryModel : bookCategoryModel.getEntries()) {
                    Path bookEntryPath = this.getPath(dataFolder, bookEntryModel);
                    futures.add(class_2405.method_10320(cache, bookEntryModel.toJson(), bookEntryPath));
                }
            }

            for (var bookCommandModel : bookModel.getCommands()) {
                Path bookCommandPath = this.getPath(dataFolder, bookCommandModel);
                futures.add(class_2405.method_10320(cache, bookCommandModel.toJson(), bookCommandPath));
            }
        }

        return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new));
    }

    @Override
    public String method_10321() {
        return "Books: " + this.modid;
    }

}
