/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book.page;

import F;
import Z;
import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.api.ModonomiconConstants.Data.Page;
import com.klikli_dev.modonomicon.book.entries.ContentBookEntry;
import com.klikli_dev.modonomicon.book.BookTextHolder;
import com.klikli_dev.modonomicon.book.RenderedBookTextHolder;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.conditions.BookNoneCondition;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.util.BookGsonHelper;
import com.klikli_dev.modonomicon.util.EntityUtil;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public class BookEntityPage extends BookPage {

    protected BookTextHolder entityName;
    protected BookTextHolder text;

    //is string, because we allow appending nbt 
    protected String entityId;
    protected float scale = 1.0f;
    protected float offset = 0f;
    protected boolean rotate = true;
    protected float defaultRotation = -45f;


    public BookEntityPage(BookTextHolder entityName, BookTextHolder text, String entityId, float scale, float offset, boolean rotate, float defaultRotation, String anchor, BookCondition condition) {
        super(anchor, condition);
        this.entityName = entityName;
        this.text = text;
        this.entityId = entityId;
        this.scale = scale;
        this.offset = offset;
        this.rotate = rotate;
        this.defaultRotation = defaultRotation;
    }

    public static BookEntityPage fromJson(JsonObject json) {
        var entityName = BookGsonHelper.getAsBookTextHolder(json, "name", BookTextHolder.EMPTY);
        var text = BookGsonHelper.getAsBookTextHolder(json, "text", BookTextHolder.EMPTY);
        var entityId = class_3518.method_15265(json, "entity_id");
        var scale = class_3518.method_15277(json, "scale", 1.0f);
        var offset = class_3518.method_15277(json, "offset", 0.0f);
        var rotate = class_3518.method_15258(json, "rotate", true);
        var defaultRotation = class_3518.method_15277(json, "default_rotation", -45.0f);

        var anchor = class_3518.method_15253(json, "anchor", "");
        var condition = json.has("condition")
                ? BookCondition.fromJson(json.getAsJsonObject("condition"))
                : new BookNoneCondition();
        return new BookEntityPage(entityName, text, entityId, scale, offset, rotate, defaultRotation, anchor, condition);
    }

    public static BookEntityPage fromNetwork(class_2540 buffer) {
        var entityName = BookTextHolder.fromNetwork(buffer);
        var text = BookTextHolder.fromNetwork(buffer);
        var entityId = buffer.method_19772();
        var scale = buffer.readFloat();
        var offset = buffer.readFloat();
        var rotate = buffer.readBoolean();
        var defaultRotation = buffer.readFloat();
        var anchor = buffer.method_19772();
        var condition = BookCondition.fromNetwork(buffer);
        return new BookEntityPage(entityName, text, entityId, scale, offset, rotate, defaultRotation, anchor, condition);
    }

    public String getEntityId() {
        return this.entityId;
    }

    public float getScale() {
        return this.scale;
    }

    public float getOffset() {
        return this.offset;
    }

    public boolean doesRotate() {
        return this.rotate;
    }

    public float getDefaultRotation() {
        return this.defaultRotation;
    }

    public BookTextHolder getEntityName() {
        return this.entityName;
    }

    public BookTextHolder getText() {
        return this.text;
    }

    @Override
    public class_2960 getType() {
        return Page.ENTITY;
    }

    @Override
    public void build(class_1937 level, ContentBookEntry parentEntry, int pageNum) {
        super.build(level, parentEntry, pageNum);

        if (this.entityName.isEmpty()) {
            //use entity name if we don't have a custom title
            this.entityName = new BookTextHolder(class_2561.method_43471(EntityUtil.getEntityName(this.entityId))
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getBook().getDefaultTitleColor())
                    ));
        }
    }

    @Override
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
        super.prerenderMarkdown(textRenderer);

        if (!this.entityName.hasComponent()) {
            this.entityName = new BookTextHolder(class_2561.method_43471(this.entityName.getKey())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getBook().getDefaultTitleColor())));
        }
        if (!this.text.hasComponent()) {
            this.text = new RenderedBookTextHolder(this.text, textRenderer.render(this.text.getString()));
        }
    }

    @Override
    public void toNetwork(class_2540 buffer) {
        this.entityName.toNetwork(buffer);
        this.text.toNetwork(buffer);
        buffer.method_10814(this.entityId);
        buffer.writeFloat(this.scale);
        buffer.writeFloat(this.offset);
        buffer.writeBoolean(this.rotate);
        buffer.writeFloat(this.defaultRotation);
        super.toNetwork(buffer);
    }

    @Override
    public boolean matchesQuery(String query) {
        return this.entityName.getString().toLowerCase().contains(query)
                || this.text.getString().toLowerCase().contains(query);
    }
}
