/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 * SPDX-FileCopyrightText: 2024 DaFuqs
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book.entries;

import I;
import Z;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.book.*;
import com.klikli_dev.modonomicon.book.conditions.BookAndCondition;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.conditions.BookEntryReadCondition;
import com.klikli_dev.modonomicon.book.conditions.BookNoneCondition;
import com.klikli_dev.modonomicon.book.error.BookErrorManager;
import com.klikli_dev.modonomicon.book.page.BookPage;
import com.klikli_dev.modonomicon.bookstate.BookUnlockStateManager;
import com.klikli_dev.modonomicon.client.gui.book.BookCategoryScreen;
import com.klikli_dev.modonomicon.client.gui.book.BookContentScreen;
import com.klikli_dev.modonomicon.client.gui.book.EntryDisplayState;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.data.JsonLoader;
import com.klikli_dev.modonomicon.data.LoaderRegistry;
import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public abstract class BookEntry {

    protected final BookEntryData data;
    protected class_2960 id;
    protected Book book;
    protected BookCategory category;
    protected List<ResolvedBookEntryParent> parents;

    /**
     * if this is not null, the command will be run when the entry is first read.
     */
    protected class_2960 commandToRunOnFirstReadId;
    protected BookCommand commandToRunOnFirstRead;

    public BookEntry(class_2960 id, BookEntryData data, class_2960 commandToRunOnFirstReadId) {
        this.id = id;
        this.data = data;

        this.commandToRunOnFirstReadId = commandToRunOnFirstReadId;
    }

    public int getX() {
        return this.data.x;
    }

    public int getY() {
        return this.data.y;
    }

    public abstract class_2960 getType();

    public abstract BookContentScreen openEntry(BookCategoryScreen categoryScreen);

    /**
     * Called after build() (after loading the book jsons) to render markdown and store any errors
     */
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
    }

    /**
     * call after loading the book jsons to finalize.
     */
    public void build(class_1937 level, BookCategory category) {
        this.book = category.getBook();
        this.category = category;

        //resolve parents
        var newParents = new ArrayList<ResolvedBookEntryParent>();
        for (var parent : this.data.parents) {
            var parentEntry = this.getBook().getEntry(parent.getEntryId());
            if (parentEntry == null) {
                BookErrorManager.get().error("Entry \"" + this.getId() + "\" has a parent that does not exist in this book: \"" + parent.getEntryId() + "\". This parent will be ignored");
            } else {
                newParents.add(new ResolvedBookEntryParent(parent, parentEntry));
            }
        }
        this.parents = newParents;

        if (this.commandToRunOnFirstReadId != null) {
            this.commandToRunOnFirstRead = this.getBook().getCommand(this.commandToRunOnFirstReadId);

            if (this.commandToRunOnFirstRead == null) {
                BookErrorManager.get().error("Command to run on first read \"" + this.commandToRunOnFirstReadId + "\" does not exist in this book. Set to null.");
                this.commandToRunOnFirstReadId = null;
            }
        }
    }

    public EntryDisplayState getEntryDisplayState(class_1657 player) {
        var isEntryUnlocked = BookUnlockStateManager.get().isUnlockedFor(player, this);

        // if an entry has its unlock condition met, it will always shop up
        if (isEntryUnlocked) {
            return EntryDisplayState.UNLOCKED;
        }

        // if an entry does have parents, it might be hidden
        if (!this.getParents().isEmpty()) {
            var anyParentsUnlocked = false;
            var allParentsUnlocked = true;
            for (var parent : this.getParents()) {
                if (!BookUnlockStateManager.get().isUnlockedFor(player, parent.getEntry())) {
                    allParentsUnlocked = false;
                } else {
                    anyParentsUnlocked = true;
                }
            }

            if (this.showWhenAnyParentUnlocked() && !anyParentsUnlocked)
                return EntryDisplayState.HIDDEN;

            if (!this.showWhenAnyParentUnlocked() && !allParentsUnlocked)
                return EntryDisplayState.HIDDEN;
        }

        // either the entry does not have any parents or any/all parents are unlocked
        return this.hideWhileLocked() ? EntryDisplayState.HIDDEN : EntryDisplayState.LOCKED;
    }

    public class_2960 getId() {
        return this.id;
    }

    /**
     * Returns true if this entry should show up in search for the given query.
     */
    public boolean matchesQuery(String query) {
        return this.data.name().toLowerCase().contains(query);
    }

    public int getPageNumberForAnchor(String anchor) {
        return -1;
    }

    public List<BookPage> getPages() {
        return List.of();
    }

    public List<BookPage> getUnlockedPagesFor(class_1657 player) {
        return List.of();
    }

    public BookCommand getCommandToRunOnFirstRead() {
        return this.commandToRunOnFirstRead;
    }

    public BookCondition getCondition() {
        return this.data.condition;
    }

    public String getName() {
        return this.data.name;
    }

    public BookCategory getCategory() {
        return this.category;
    }

    public Book getBook() {
        return this.book;
    }

    public String getDescription() {
        return this.data.description;
    }

    public List<? extends BookEntryParent> getParents() {
        return this.parents == null ? this.data.parents : this.parents;
    }

    public int getEntryBackgroundUIndex() {
        return this.data.entryBackgroundUIndex;
    }

    public int getEntryBackgroundVIndex() {
        return this.data.entryBackgroundVIndex;
    }

    public boolean showWhenAnyParentUnlocked() {
        return this.data.showWhenAnyParentUnlocked;
    }

    public boolean hideWhileLocked() {
        return this.data.hideWhileLocked;
    }

    public BookIcon getIcon() {
        return this.data.icon;
    }

    public class_2960 getCategoryId() {
        return this.data.categoryId;
    }

    public abstract void toNetwork(class_2540 buf);

    /**
     * The first two rows in "entry_textures.png" are reserved for the entry icons.
     * the entry background is selected by querying the texture at entryBackgroundUIndex * 26 (= Y Axis / Up-Down), entryBackgroundUIndex * 26 (= X Axis / Left-Right)
     * U index = Y Axis / Up-Down
     * V index = X Axis / Left-Right
     */
    public record BookEntryData(class_2960 categoryId, List<BookEntryParent> parents, int x, int y, String name,
                                String description, BookIcon icon, int entryBackgroundUIndex, int entryBackgroundVIndex,
                                BookCondition condition, boolean hideWhileLocked, boolean showWhenAnyParentUnlocked) {

        public static BookEntryData fromJson(JsonObject json, boolean autoAddReadConditions) {
            var categoryId = new class_2960(class_3518.method_15265(json, "category"));
            var x = class_3518.method_15260(json, "x");
            var y = class_3518.method_15260(json, "y");

            var parents = new ArrayList<BookEntryParent>();
            if (json.has("parents")) {
                for (var parent : class_3518.method_15261(json, "parents")) {
                    parents.add(BookEntryParent.fromJson(parent.getAsJsonObject()));
                }
            }

            var pages = new ArrayList<BookPage>();
            if (json.has("pages")) {
                for (var pageElem : class_3518.method_15261(json, "pages")) {
                    BookErrorManager.get().setContext("Page Index: {}", pages.size());
                    var pageJson = class_3518.method_15295(pageElem, "page");
                    var type = new class_2960(class_3518.method_15265(pageJson, "type"));
                    var loader = LoaderRegistry.getPageJsonLoader(type);
                    var page = loader.fromJson(pageJson);
                    pages.add(page);
                }
            }

            var name = class_3518.method_15265(json, "name");
            var description = class_3518.method_15253(json, "description", "");
            var icon = BookIcon.fromJson(json.get("icon"));
            var entryBackgroundUIndex = class_3518.method_15282(json, "background_u_index", 0);
            var entryBackgroundVIndex = class_3518.method_15282(json, "background_v_index", 0);

            BookCondition condition = new BookNoneCondition(); //default to unlocked
            if (json.has("condition")) {
                condition = BookCondition.fromJson(json.getAsJsonObject("condition"));
            } else if (autoAddReadConditions) {
                if (parents.size() == 1) {
                    condition = new BookEntryReadCondition(null, parents.get(0).getEntryId());
                } else if (parents.size() > 1) {
                    var conditions = parents.stream().map(parent -> new BookEntryReadCondition(null, parent.getEntryId())).toList();
                    condition = new BookAndCondition(null, conditions.toArray(new BookEntryReadCondition[0]));
                }
            }
            var hideWhileLocked = class_3518.method_15258(json, "hide_while_locked", false);

            /**
             * If true, the entry will show (locked) as soon as any parent is unlocked.
             * If false, the entry will only show (locked) as soon as all parents are unlocked.
             */
            var showWhenAnyParentUnlocked = class_3518.method_15258(json, "show_when_any_parent_unlocked", false);

            return new BookEntryData(categoryId, parents, x, y, name, description, icon, entryBackgroundUIndex, entryBackgroundVIndex, condition, hideWhileLocked, showWhenAnyParentUnlocked);
        }

        public static BookEntryData fromNetwork(class_2540 buffer) {
            var categoryId = buffer.method_10810();
            var name = buffer.method_19772();
            var description = buffer.method_19772();
            var icon = BookIcon.fromNetwork(buffer);
            var x = buffer.method_10816();
            var y = buffer.method_10816();
            var entryBackgroundUIndex = buffer.method_10816();
            var entryBackgroundVIndex = buffer.method_10816();
            var hideWhileLocked = buffer.readBoolean();
            var showWhenAnyParentUnlocked = buffer.readBoolean();
            var condition = BookCondition.fromNetwork(buffer);

            var parentEntries = new ArrayList<BookEntryParent>();
            var parentCount = buffer.method_10816();
            for (var i = 0; i < parentCount; i++) {
                parentEntries.add(BookEntryParent.fromNetwork(buffer));
            }

            return new BookEntryData(categoryId, parentEntries, x, y, name, description, icon, entryBackgroundUIndex, entryBackgroundVIndex, condition, hideWhileLocked, showWhenAnyParentUnlocked);
        }

        public void toNetwork(class_2540 buffer) {
            buffer.method_10812(this.categoryId);
            buffer.method_10814(this.name);
            buffer.method_10814(this.description);
            this.icon.toNetwork(buffer);
            buffer.method_10804(this.x);
            buffer.method_10804(this.y);
            buffer.method_10804(this.entryBackgroundUIndex);
            buffer.method_10804(this.entryBackgroundVIndex);
            buffer.writeBoolean(this.hideWhileLocked);
            buffer.writeBoolean(this.showWhenAnyParentUnlocked);

            buffer.method_10812(this.condition.getType());
            this.condition.toNetwork(buffer);

            buffer.method_10804(this.parents.size());
            for (var parent : this.parents) {
                parent.toNetwork(buffer);
            }
        }

    }

}
