/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book.entries;

import I;
import com.google.gson.*;
import com.klikli_dev.modonomicon.api.*;
import com.klikli_dev.modonomicon.book.*;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.error.BookErrorManager;
import com.klikli_dev.modonomicon.book.page.BookPage;
import com.klikli_dev.modonomicon.bookstate.BookUnlockStateManager;
import com.klikli_dev.modonomicon.client.gui.book.*;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.data.JsonLoader;
import com.klikli_dev.modonomicon.data.LoaderRegistry;
import com.klikli_dev.modonomicon.data.NetworkLoader;
import java.util.*;
import net.minecraft.class_1657;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public class ContentBookEntry extends BookEntry {

    protected List<BookPage> pages;

    public ContentBookEntry(class_2960 id, BookEntryData data, class_2960 commandToRunOnFirstReadId, List<BookPage> pages) {
        super(id, data, commandToRunOnFirstReadId);
        this.pages = pages;
    }
    
    @Override
    public class_2960 getType() {
        return ModonomiconConstants.Data.EntryType.CONTENT;
    }

    public static ContentBookEntry fromJson(class_2960 id, JsonObject json, boolean autoAddReadConditions) {
        BookEntryData data = BookEntryData.fromJson(json, autoAddReadConditions);

        class_2960 commandToRunOnFirstReadId = null;
        if (json.has("command_to_run_on_first_read")) {
            commandToRunOnFirstReadId = new class_2960(class_3518.method_15265(json, "command_to_run_on_first_read"));
        }

        var pages = new ArrayList<BookPage>();
        if (json.has("pages")) {
            var jsonPages = class_3518.method_15261(json, "pages");
            for (var pageElem : jsonPages) {
                BookErrorManager.get().setContext("Page Index: {}", pages.size());
                var pageJson = class_3518.method_15295(pageElem, "page");
                var type = new class_2960(class_3518.method_15265(pageJson, "type"));
                var loader = LoaderRegistry.getPageJsonLoader(type);
                var page = loader.fromJson(pageJson);
                pages.add(page);
            }
        }
        
        return new ContentBookEntry(id, data, commandToRunOnFirstReadId, pages);
    }

    @Override
    public void toNetwork(class_2540 buffer) {
        buffer.method_10812(this.id);
        this.data.toNetwork(buffer);
        buffer.method_43826(this.commandToRunOnFirstReadId, class_2540::method_10812);
        
        buffer.method_10804(this.pages.size());
        for (var page : this.pages) {
            buffer.method_10812(page.getType());
            page.toNetwork(buffer);
        }
    }
    
    public static ContentBookEntry fromNetwork(class_2540 buffer) {
        var id = buffer.method_10810();
        BookEntryData data = BookEntryData.fromNetwork(buffer);
        class_2960 commandToRunOnFirstReadId = buffer.method_43827(class_2540::method_10810);

        var pages = new ArrayList<BookPage>();
        var pageCount = buffer.method_10816();
        for (var i = 0; i < pageCount; i++) {
            var type = buffer.method_10810();
            var loader = LoaderRegistry.getPageNetworkLoader(type);
            var page = loader.fromNetwork(buffer);
            pages.add(page);
        }
        
        return new ContentBookEntry(id, data, commandToRunOnFirstReadId, pages);
    }
    
    /**
     * call after loading the book jsons to finalize.
     */
    @Override
    public void build(class_1937 level, BookCategory category) {
        super.build(level, category);

        //build pages
        int pageNum = 0;
        for (var page : this.pages) {
            BookErrorManager.get().getContextHelper().pageNumber = pageNum;
            page.build(level, this, pageNum);
            BookErrorManager.get().getContextHelper().pageNumber = -1;
            pageNum++;
        }
    }
    
    @Override
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
        for (var page : this.pages) {
            BookErrorManager.get().getContextHelper().pageNumber = page.getPageNumber();
            page.prerenderMarkdown(textRenderer);
            BookErrorManager.get().getContextHelper().pageNumber = -1;
        }
    }
    
    @Override
    public List<BookPage> getPages() {
        return this.pages;
    }
    
    @Override
    public List<BookPage> getUnlockedPagesFor(class_1657 player) {
        BookUnlockStateManager unlockManager = BookUnlockStateManager.get();
        return unlockManager.getUnlockedPagesFor(player, this);
    }
    
    @Override
    public int getPageNumberForAnchor(String anchor) {
        var pages = this.getPages();
        for (int i = 0; i < pages.size(); i++) {
            var page = pages.get(i);
            if (anchor.equals(page.getAnchor())) {
                return i;
            }
        }

        return -1;
    }
    
    @Override
    public boolean matchesQuery(String query) {
        if(super.matchesQuery(query)) {
            return true;
        }

        for (var page : this.getPages()) {
            if (page.matchesQuery(query)) {
                return true;
            }
        }

        return false;
    }
    
    public BookContentScreen openEntry(BookCategoryScreen categoryScreen) {
        return categoryScreen.openContentEntry(this);
    }
}
