/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.client.render.page;

import F;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.book.BookTextHolder;
import com.klikli_dev.modonomicon.book.RenderedBookTextHolder;
import com.klikli_dev.modonomicon.book.error.BookErrorManager;
import com.klikli_dev.modonomicon.book.page.BookPage;
import com.klikli_dev.modonomicon.client.gui.book.BookContentScreen;
import com.klikli_dev.modonomicon.client.gui.book.markdown.MarkdownComponentRenderUtils;
import com.klikli_dev.modonomicon.data.BookDataManager;
import com.klikli_dev.modonomicon.util.GuiGraphicsExt;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_4185;
import net.minecraft.class_5250;
import net.minecraft.class_5481;

public abstract class BookPageRenderer<T extends BookPage> {
    public int left;
    public int top;
    protected T page;
    protected BookContentScreen parentScreen;
    protected class_310 mc;
    protected class_327 font;

    private List<class_4185> buttons = new ArrayList<>();


    public BookPageRenderer(T page) {
        this.page = page;
    }

    /**
     * Will render the given BookTextHolder as (left-aligned) content text. Will automatically handle markdown.
     */
    public static void renderBookTextHolder(class_332 guiGraphics, BookTextHolder text, class_327 font, int x, int y, int width) {
        if (text.hasComponent()) {
            //if it is a component, we draw it directly
            for (class_5481 formattedcharsequence : font.method_1728(text.getComponent(), width)) {
                guiGraphics.method_51430(font, formattedcharsequence, x, y, 0, false);
                y += font.field_2000;
            }
        } else if (text instanceof RenderedBookTextHolder renderedText) {
            //if it is not a component it was sent through the markdown renderer
            var components = renderedText.getRenderedText();

            for (var component : components) {
                var wrapped = MarkdownComponentRenderUtils.wrapComponents(component, width, width - 10, font);
                for (class_5481 formattedcharsequence : wrapped) {
                    guiGraphics.method_51430(font, formattedcharsequence, x, y, 0, false);
                    y += font.field_2000;
                }
            }
        } else {
            Modonomicon.LOG.warn("BookTextHolder with String {} has no component, but is not rendered to markdown either.", text.getString());
        }
    }

    /**
     * Call when the page is being set up to be displayed (when book content screen opens, or pages are changed)
     */
    public void onBeginDisplayPage(BookContentScreen parentScreen, int left, int top) {
        this.parentScreen = parentScreen;

        this.mc = parentScreen.getMinecraft();
        this.font = this.mc.field_1772;
        this.left = left;
        this.top = top;

        this.buttons = new ArrayList<>();
    }

    public T getPage() {
        return this.page;
    }

    /**
     * Call when the page is will no longer be displayed (when book content screen opens, or pages are changed)
     */
    public void onEndDisplayPage(BookContentScreen parentScreen) {
        parentScreen.removeRenderableWidgets(this.buttons);
    }

    /**
     * @param pMouseX localized to page x (mouseX - bookLeft - page.left)
     * @param pMouseY localized to page y (mouseY - bookTop - page.top)
     */
    public boolean mouseClicked(double pMouseX, double pMouseY, int pButton) {
        return false;
    }

    /**
     * Will render the given BookTextHolder as (left-aligned) content text. Will automatically handle markdown.
     */
    public void renderBookTextHolder(class_332 guiGraphics, BookTextHolder text, int x, int y, int width) {
        x += this.parentScreen.getBook().getBookTextOffsetX();
        y += this.parentScreen.getBook().getBookTextOffsetY();
        width += this.parentScreen.getBook().getBookTextOffsetWidth();
        width -= this.parentScreen.getBook().getBookTextOffsetX(); //always remove the offset x from the width to avoid overflow

        renderBookTextHolder(guiGraphics, text, this.font, x, y, width);
    }

    /**
     * Will render the given BookTextHolder as (centered) title.
     */
    public void renderTitle(class_332 guiGraphics, BookTextHolder title, boolean showTitleSeparator, int x, int y) {

        guiGraphics.method_51448().method_22903();

        if (title instanceof RenderedBookTextHolder renderedTitle) {
            //if user decided to use markdown title, we need to use the  rendered version
            var formattedCharSequence = class_5481.method_30755(
                    renderedTitle.getRenderedText().stream().map(class_2561::method_30937).toList());

            //if title is larger than allowed, scaled to fit
            var scale = Math.min(1.0f, (float) BookContentScreen.MAX_TITLE_WIDTH / (float) this.font.method_30880(formattedCharSequence));
            if (scale < 1) {
                guiGraphics.method_51448().method_46416(0, y - y * scale, 0);
                guiGraphics.method_51448().method_22905(scale, scale, scale);
            }

            this.drawCenteredStringNoShadow(guiGraphics, formattedCharSequence, x, y, 0, scale);
        } else if(title.hasComponent()) {
            //non-markdown title we just render as usual

            var font = BookDataManager.Client.get().safeFont(this.page.getBook().getFont());

            var titleComponent = class_2561.method_43473().method_10852(title.getComponent()).method_27694(s -> s.method_27704(font));
            //if title is larger than allowed, scaled to fit
            var scale = Math.min(1.0f, (float) BookContentScreen.MAX_TITLE_WIDTH / (float) this.font.method_30880(titleComponent.method_30937()));
            if (scale < 1) {
                guiGraphics.method_51448().method_46416(0, y - y * scale, 0);
                guiGraphics.method_51448().method_22905(scale, scale, scale);
            }

            //otherwise we use the component - that is either provided by the user, or created from the default title style.
            this.drawCenteredStringNoShadow(guiGraphics, titleComponent.method_30937(), x, y, 0, scale);
        } else {
            //this means a non-markdown title has no component -> this should not be possible, it indicates that either:
            // - a page did not set up its (non markdown) book text holder correctly in preprender markdown
            // - or a markdown title failed to render and remained a non-rendered book text holder
            BookErrorManager.get().setTo(this.page);
            BookErrorManager.get().error("Non-markdown title has no component.");
            BookErrorManager.get().getContextHelper().reset();
            BookErrorManager.get().setCurrentBookId(null);
        }

        guiGraphics.method_51448().method_22909();

        if (showTitleSeparator)
            BookContentScreen.drawTitleSeparator(guiGraphics, this.page.getBook(), x, y + 12);
    }

    public abstract void render(class_332 guiGraphics, int mouseX, int mouseY, float ticks);

    public void drawCenteredStringNoShadow(class_332 guiGraphics, class_5481 s, int x, int y, int color, float scale) {
        GuiGraphicsExt.drawString(guiGraphics, this.font, s, x - this.font.method_30880(s) * scale / 2.0F, y + (this.font.field_2000 * (1 - scale)), color, false);
    }

    public void drawCenteredStringNoShadow(class_332 guiGraphics, String s, int x, int y, int color, float scale) {
        GuiGraphicsExt.drawString(guiGraphics, this.font, s, x - this.font.method_1727(s) * scale / 2.0F, y + (this.font.field_2000 * (1 - scale)), color, false);
    }

    public void drawWrappedStringNoShadow(class_332 guiGraphics, class_2561 s, int x, int y, int color, int width) {
        for (class_5481 formattedcharsequence : this.font.method_1728(s, width)) {
            guiGraphics.method_51430(this.font, formattedcharsequence, x, y + (this.font.field_2000), color, false);
            y += this.font.field_2000;
        }
    }

    /**
     * @param pMouseX localized to page x (mouseX - bookLeft - page.left)
     * @param pMouseY localized to page y (mouseY - bookTop - page.top)
     */
    @Nullable
    public class_2583 getClickedComponentStyleAt(double pMouseX, double pMouseY) {
        return null;
    }

    protected void addButton(class_4185 button) {
        button.method_46421(button.method_46426() + this.parentScreen.getBookLeft() + this.left);
        button.method_46419(button.method_46427() + this.parentScreen.getBookTop() + this.top);
        this.buttons.add(button);
        this.parentScreen.method_37063(button);
    }

    @Nullable
    protected class_2583 getClickedComponentStyleAtForTitle(BookTextHolder title, int x, int y, double pMouseX, double pMouseY) {
        //they say good code comments itself. Well, this is not good code.
        if (title instanceof RenderedBookTextHolder renderedTitle) {
            //markdown title
            var formattedCharSequence = class_5481.method_30755(
                    renderedTitle.getRenderedText().stream().map(class_2561::method_30937).toList());
            if (pMouseY > y && pMouseY < y + this.font.field_2000) {
                //check if we are vertically over the title line

                x = x - this.font.method_30880(formattedCharSequence) / 2;
                if (pMouseX < x)
                    return null;
                //if we are horizontally left of the title, exit

                //horizontally over and right of the title is handled by font splitter
                return this.font.method_27527().method_30876(formattedCharSequence, (int) pMouseX - x);
            }
        } else {
            if (pMouseY > y && pMouseY < y + this.font.field_2000) {
                //check if we are vertically over the title line

                var formattedCharSequence = title.getComponent().method_30937();
                x = x - this.font.method_30880(formattedCharSequence) / 2;
                if (pMouseX < x)
                    return null;
                //if we are horizontally left of the title, exit

                //horizontally over and right of the title is handled by font splitter
                return this.font.method_27527().method_30876(formattedCharSequence, (int) pMouseX - x);
            }
        }
        return null;
    }

    @Nullable
    protected class_2583 getClickedComponentStyleAtForTextHolder(BookTextHolder text, int x, int y, int width, double pMouseX, double pMouseY) {
        if (text.hasComponent()) {
            //we don't do math to get the current line, we just split and iterate.
            //why? Because performance should not matter (significantly enough to bother)
            for (class_5481 formattedcharsequence : this.font.method_1728(text.getComponent(), width)) {
                if (pMouseY > y && pMouseY < y + this.font.field_2000) {
                    //check if we are vertically over the title line
                    //horizontally over and right of the title is handled by font splitter
                    return this.font.method_27527().method_30876(formattedcharsequence, (int) pMouseX - x);
                }
                y += this.font.field_2000;
            }
        } else if (text instanceof RenderedBookTextHolder renderedText) {
            var components = renderedText.getRenderedText();
            for (var component : components) {
                var wrapped = MarkdownComponentRenderUtils.wrapComponents(component, width, width - 10, this.font);
                for (class_5481 formattedcharsequence : wrapped) {
                    if (pMouseY > y && pMouseY < y + this.font.field_2000) {
                        //check if we are vertically over the title line
                        //horizontally over and right of the title is handled by font splitter
                        return this.font.method_27527().method_30876(formattedcharsequence, (int) pMouseX - x);
                    }
                    y += this.font.field_2000;
                }
            }
        }

        return null;
    }
}
