/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.client.render.page;

import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.multiblock.Multiblock;
import com.klikli_dev.modonomicon.api.multiblock.Multiblock.SimulateResult;
import com.klikli_dev.modonomicon.book.page.BookMultiblockPage;
import com.klikli_dev.modonomicon.client.ClientTicks;
import com.klikli_dev.modonomicon.client.gui.BookGuiManager;
import com.klikli_dev.modonomicon.client.gui.book.BookContentRenderer;
import com.klikli_dev.modonomicon.client.gui.book.button.VisualizeButton;
import com.klikli_dev.modonomicon.client.gui.book.entry.BookEntryScreen;
import com.klikli_dev.modonomicon.client.render.MultiblockPreviewRenderer;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.datafixers.util.Pair;
import com.mojang.math.Axis;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.gui.components.Button;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.client.multiplayer.ClientLevel;
import net.minecraft.client.renderer.MultiBufferSource;
import net.minecraft.client.renderer.blockentity.BlockEntityRenderer;
import net.minecraft.client.renderer.texture.OverlayTexture;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Vec3i;
import net.minecraft.network.chat.Style;
import net.minecraft.util.RandomSource;
import net.minecraft.world.level.block.EntityBlock;
import net.minecraft.world.level.block.Rotation;
import net.minecraft.world.level.block.entity.BlockEntity;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.BlockHitResult;
import org.jetbrains.annotations.Nullable;
import org.joml.Matrix4f;
import org.joml.Vector4f;

import java.util.*;

public class BookMultiblockPageRenderer extends BookPageRenderer<BookMultiblockPage> implements PageWithTextRenderer {

    private static final RandomSource randomSource = RandomSource.createNewThreadLocalInstance();
    private final Map<BlockPos, BlockEntity> blockEntityCache = new Object2ObjectOpenHashMap<>();
    private final Set<BlockEntity> erroredBlockEntities = Collections.newSetFromMap(new WeakHashMap<>());

    protected Pair<BlockPos, Collection<SimulateResult>> multiblockSimulation;
    protected Button visualizeButton;

    public BookMultiblockPageRenderer(BookMultiblockPage page) {
        super(page);
    }

    public void handleButtonVisualize(Button button) {
        MultiblockPreviewRenderer.setMultiblock(this.page.getMultiblock(), this.page.getMultiblockName().getComponent(), true);
        BookGuiManager.get().closeScreenStack(this.parentScreen); //will cause the book to close entirely, and save the open page

        //TODO: visualizer bookmark to go back to this page quickly?
        //String entryKey =  this.parentEntry.getId().toString(); will be used for bookmark for multiblock
//        Bookmark bookmark = new Bookmark(entryKey, pageNum / 2);
//        parent.addBookmarkButtons();
    }

    private void renderMultiblock(GuiGraphics guiGraphics) {
        var mc = Minecraft.getInstance();
        var level = mc.level;

        var pos = BlockPos.ZERO;
        var facingRotation = Rotation.NONE;

        this.page.getMultiblock().setLevel(level);

        if (this.page.getMultiblock().isSymmetrical()) {
            facingRotation = Rotation.NONE;
        }

        Vec3i size = this.page.getMultiblock().getSize();
        int sizeX = size.getX();
        int sizeY = size.getY();
        int sizeZ = size.getZ();
        float maxX = 90;
        float maxY = 90;
        float diag = (float) Math.sqrt(sizeX * sizeX + sizeZ * sizeZ);
        float scaleX = maxX / diag;
        float scaleY = maxY / sizeY;
        float scale = -Math.min(scaleX, scaleY);

        int xPos = BookEntryScreen.PAGE_WIDTH / 2;
        int yPos = 60;

        guiGraphics.pose().pushPose();

        guiGraphics.pose().translate(xPos, yPos, 100);
        guiGraphics.pose().scale(scale, scale, scale);
        guiGraphics.pose().translate(-(float) sizeX / 2, -(float) sizeY / 2, 0);


        // Initial eye pos somewhere off in the distance in the -Z direction
        Vector4f eye = new Vector4f(0, 0, -100, 1);
        Matrix4f rotMat = new Matrix4f();
        rotMat.identity();

        // For each GL rotation done, track the opposite to keep the eye pos accurate
        guiGraphics.pose().mulPose(Axis.XP.rotationDegrees(-30F));
        rotMat.rotate(Axis.XP.rotationDegrees(30));

        float offX = (float) -sizeX / 2;
        float offZ = (float) -sizeZ / 2 + 1;

        float time = this.parentScreen.getTicksInBook() * 0.5F;
        if (!Screen.hasShiftDown()) {
            time += ClientTicks.partialTicks;
        }
        guiGraphics.pose().translate(-offX, 0, -offZ);
        guiGraphics.pose().mulPose(Axis.YP.rotationDegrees(time));
        rotMat.rotate(Axis.YP.rotationDegrees(-time));
        guiGraphics.pose().mulPose(Axis.YP.rotationDegrees(45));
        rotMat.rotate(Axis.YP.rotationDegrees(-45));
        guiGraphics.pose().translate(offX, 0, offZ);

        // Finally apply the rotations
        rotMat.transform(eye);
        eye.div(eye.w);


        var buffers = mc.renderBuffers().bufferSource();

        BlockPos checkPos = null;
        if (mc.hitResult instanceof BlockHitResult blockRes) {
            checkPos = blockRes.getBlockPos().relative(blockRes.getDirection());
        }

        guiGraphics.pose().pushPose();
        RenderSystem.setShaderColor(1F, 1F, 1F, 1F);
        guiGraphics.pose().translate(0, 0, -1);

        for (Multiblock.SimulateResult r : this.multiblockSimulation.getSecond()) {
            float alpha = 0.3F;
            if (r.getWorldPosition().equals(checkPos)) {
                alpha = 0.6F + (float) (Math.sin(ClientTicks.total * 0.3F) + 1F) * 0.1F;
            }

            BlockState renderState = r.getStateMatcher().getDisplayedState(ClientTicks.ticks).rotate(facingRotation);

            this.renderBlock(buffers, level, renderState, r.getWorldPosition(), alpha, guiGraphics.pose());

            if (renderState.getBlock() instanceof EntityBlock eb) {
                //if our cached be is not compatible with the render state, remove it.
                //this happens e.g. if there is a blocktag that contains multible blocks with different BEs
                var be = this.blockEntityCache.compute(r.getWorldPosition().immutable(), (p, cachedBe) -> {
                    if (cachedBe != null && !cachedBe.getType().isValid(renderState)) {
                        return eb.newBlockEntity(p, renderState);
                    }
                    return cachedBe != null ? cachedBe : eb.newBlockEntity(p, renderState);
                });

                if (be != null && !this.erroredBlockEntities.contains(be)) {
                    be.setLevel(mc.level);

                    // fake cached state in case the renderer checks it as we don't want to query the actual world
                    //noinspection deprecation
                    be.setBlockState(renderState);

                    guiGraphics.pose().pushPose();
                    var bePos = r.getWorldPosition();
                    guiGraphics.pose().translate(bePos.getX(), bePos.getY(), bePos.getZ());

                    try {
                        BlockEntityRenderer<BlockEntity> renderer = Minecraft.getInstance().getBlockEntityRenderDispatcher().getRenderer(be);
                        if (renderer != null) {
                            renderer.render(be, ClientTicks.partialTicks, guiGraphics.pose(), buffers, 0xF000F0, OverlayTexture.NO_OVERLAY);
                        }
                    } catch (Exception e) {
                        this.erroredBlockEntities.add(be);
                        Modonomicon.LOG.error("Error rendering block entity", e);
                    }
                    guiGraphics.pose().popPose();
                }
            }
        }
        guiGraphics.pose().popPose();
        buffers.endBatch();
        guiGraphics.pose().popPose();

    }

    private void renderBlock(MultiBufferSource.BufferSource buffers, ClientLevel level, BlockState state, BlockPos pos, float alpha, PoseStack ps) {
        if (pos != null) {
            ps.pushPose();
            ps.translate(pos.getX(), pos.getY(), pos.getZ());

            ClientServices.MULTIBLOCK.renderBlock(state, pos, this.page.getMultiblock(), ps, buffers, randomSource);

            ps.popPose();
        }
    }

    @Override
    public int getTextY() {
        //text is always below multiblock, and we don't shift based on multiblock name (unlike title for text pages)
        return 115;
    }

    @Override
    public void onBeginDisplayPage(BookEntryScreen parentScreen, int left, int top) {
        super.onBeginDisplayPage(parentScreen, left, top);

        this.multiblockSimulation = this.page.getMultiblock().simulate(null, BlockPos.ZERO, Rotation.NONE, true, true);

        if (this.page.showVisualizeButton()) {
            this.addButton(this.visualizeButton = new VisualizeButton(this.parentScreen, 13, 102, this::handleButtonVisualize));
        }
    }

    @Override
    public void render(GuiGraphics guiGraphics, int mouseX, int mouseY, float ticks) {

        //render a frame for the multiblock render area
        int x = BookEntryScreen.PAGE_WIDTH / 2 - 53;
        int y = 7;
        RenderSystem.enableBlend();
        RenderSystem.setShaderColor(1F, 1F, 1F, 1F);
        BookContentRenderer.drawFromContentTexture(guiGraphics, this.page.getBook(), x, y, 405, 149, 106, 106);

        //render multiblock name in place of title
        if (!this.page.getMultiblockName().isEmpty()) {
            this.renderTitle(guiGraphics, this.page.getMultiblockName(), false, BookEntryScreen.PAGE_WIDTH / 2, 0);
        }

        this.renderMultiblock(guiGraphics);

        var textY = this.getTextY();
        this.renderBookTextHolder(guiGraphics, this.page.getText(), 0, textY, BookEntryScreen.PAGE_WIDTH, BookEntryScreen.PAGE_HEIGHT - textY);

        //TODO: render button to show multiblock in world
        //            //TODO: show multiblock preview on button click
//            var block = MultiblockDataManager.get().getMultiblock(ResourceLocation.tryParse("modonomicon:blockentity"));
//            MultiblockPreviewRenderer.setMultiblock(block, Component.translatable("multiblock.modonomicon.test"), true);

        var style = this.getClickedComponentStyleAt(mouseX, mouseY);
        if (style != null)
            this.parentScreen.renderComponentHoverEffect(guiGraphics, style, mouseX, mouseY);
    }

    @Nullable
    @Override
    public Style getClickedComponentStyleAt(double pMouseX, double pMouseY) {
        if (pMouseX > 0 && pMouseY > 0) {
            var multiblockNameStyle = this.getClickedComponentStyleAtForTitle(this.page.getMultiblockName(), BookEntryScreen.PAGE_WIDTH / 2, 0, pMouseX, pMouseY);
            if (multiblockNameStyle != null) {
                return multiblockNameStyle;
            }

            var x = this.parentScreen.getBook().getBookTextOffsetX();
            var y = this.getTextY() + this.parentScreen.getBook().getBookTextOffsetY();
            var width = BookEntryScreen.PAGE_WIDTH + this.parentScreen.getBook().getBookTextOffsetWidth() - x; //always remove the offset x from the width to avoid overflow
            var height = BookEntryScreen.PAGE_HEIGHT + this.parentScreen.getBook().getBookTextOffsetHeight() - y; //always remove the offset y from the height to avoid overflow

            var textStyle = this.getClickedComponentStyleAtForTextHolder(this.page.getText(), x, y, width, height, pMouseX, pMouseY);
            if (textStyle != null) {
                return textStyle;
            }
        }
        return super.getClickedComponentStyleAt(pMouseX, pMouseY);
    }


}
