// SPDX-FileCopyrightText: 2023 klikli-dev
// SPDX-FileCopyrightText: 2023 mezz
//
// SPDX-License-Identifier: MIT

package com.klikli_dev.modonomicon.client.fluid;

import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.fluid.FabricFluidHolder;
import com.klikli_dev.modonomicon.fluid.FluidHolder;
import com.klikli_dev.modonomicon.platform.services.FluidHelper;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.*;
import net.fabricmc.fabric.api.transfer.v1.client.fluid.FluidVariantRendering;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariantAttributes;
import net.minecraft.class_1058;
import net.minecraft.class_124;
import net.minecraft.class_1723;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_286;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_293;
import net.minecraft.class_332;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_5250;
import net.minecraft.class_757;
import org.joml.Matrix4f;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Fluid rendering based on FluidHelper from JEI
 */
public class FluidRenderHelper {
    private static final int TEXTURE_SIZE = 16;
    private static final int MIN_FLUID_HEIGHT = 1; // ensure tiny amounts of fluid are still visible

    /**
     * @param guiGraphics the gui graphics.
     * @param width       the width of the fluid "slot" to render.
     * @param height      the height of the fluid "slot" to render.
     * @param fluidHolder the fluid stack representing the fluid and the amount to render.
     * @param capacity    the capacity of the fluid "slot" - together with the amount it determines the actual height of the fluid rendered within the slot.
     */
    public static void drawFluid(class_332 guiGraphics, final int width, final int height, FabricFluidHolder fluidHolder, int capacity) {
        var fluidVariant = fluidHolder.toVariant();
        class_3611 fluid = fluidHolder.getFluid().comp_349();
        if (fluid.method_15780(class_3612.field_15906)) {
            return;
        }

        getStillFluidSprite(fluidVariant)
                .ifPresent(fluidStillSprite -> {
                    int fluidColor = getColorTint(fluidVariant);

                    long amount = fluidHolder.getAmount();
                    long scaledAmount = (amount * height) / capacity;
                    if (amount > 0 && scaledAmount < MIN_FLUID_HEIGHT) {
                        scaledAmount = MIN_FLUID_HEIGHT;
                    }
                    if (scaledAmount > height) {
                        scaledAmount = height;
                    }

                    drawTiledSprite(guiGraphics, width, height, fluidColor, scaledAmount, fluidStillSprite);
                });
    }

    private static int getColorTint(FluidVariant fluidVariant) {
        int fluidColor = FluidVariantRendering.getColor(fluidVariant);
        return fluidColor | 0xFF000000;
    }

    private static Optional<class_1058> getStillFluidSprite(FluidVariant fluidVariant) {
        class_1058 sprite = FluidVariantRendering.getSprite(fluidVariant);
        return Optional.ofNullable(sprite);
    }


    private static void drawTiledSprite(class_332 guiGraphics, final int tiledWidth, final int tiledHeight, int color, long scaledAmount, class_1058 sprite) {
        RenderSystem.setShaderTexture(0, class_1723.field_21668);
        Matrix4f matrix = guiGraphics.method_51448().method_23760().method_23761();
        setGLColorFromInt(color);

        final int xTileCount = tiledWidth / TEXTURE_SIZE;
        final int xRemainder = tiledWidth - (xTileCount * TEXTURE_SIZE);
        final long yTileCount = scaledAmount / TEXTURE_SIZE;
        final long yRemainder = scaledAmount - (yTileCount * TEXTURE_SIZE);

        final int yStart = tiledHeight;

        for (int xTile = 0; xTile <= xTileCount; xTile++) {
            for (int yTile = 0; yTile <= yTileCount; yTile++) {
                int width = (xTile == xTileCount) ? xRemainder : TEXTURE_SIZE;
                long height = (yTile == yTileCount) ? yRemainder : TEXTURE_SIZE;
                int x = (xTile * TEXTURE_SIZE);
                int y = yStart - ((yTile + 1) * TEXTURE_SIZE);
                if (width > 0 && height > 0) {
                    long maskTop = TEXTURE_SIZE - height;
                    int maskRight = TEXTURE_SIZE - width;

                    drawTextureWithMasking(matrix, x, y, sprite, maskTop, maskRight, 100);
                }
            }
        }
        //now reset color
        RenderSystem.setShaderColor(1f, 1f, 1f, 1f);
    }

    private static void setGLColorFromInt(int color) {
        float red = (color >> 16 & 0xFF) / 255.0F;
        float green = (color >> 8 & 0xFF) / 255.0F;
        float blue = (color & 0xFF) / 255.0F;
        float alpha = ((color >> 24) & 0xFF) / 255F;

        RenderSystem.setShaderColor(red, green, blue, alpha);
    }

    private static void drawTextureWithMasking(Matrix4f matrix, float xCoord, float yCoord, class_1058 textureSprite, long maskTop, long maskRight, float zLevel) {
        float uMin = textureSprite.method_4594();
        float uMax = textureSprite.method_4577();
        float vMin = textureSprite.method_4593();
        float vMax = textureSprite.method_4575();
        uMax = uMax - (maskRight / 16F * (uMax - uMin));
        vMax = vMax - (maskTop / 16F * (vMax - vMin));

        RenderSystem.setShader(class_757::method_34542);

        class_287 bufferBuilder = class_289.method_1348().method_60827(class_293.class_5596.field_27382, class_290.field_1585);
        bufferBuilder.method_22918(matrix, xCoord, yCoord + 16, zLevel).method_22913(uMin, vMax);
        bufferBuilder.method_22918(matrix, xCoord + 16 - maskRight, yCoord + 16, zLevel).method_22913(uMax, vMax);
        bufferBuilder.method_22918(matrix, xCoord + 16 - maskRight, yCoord + maskTop, zLevel).method_22913(uMax, vMin);
        bufferBuilder.method_22918(matrix, xCoord, yCoord + maskTop, zLevel).method_22913(uMin, vMin);
        class_286.method_43433(bufferBuilder.method_60800());
    }

    public static List<class_2561> getTooltip(FabricFluidHolder fluidHolder, int capacity, class_1836 tooltipFlag, FluidHelper.TooltipMode tooltipMode) {
        var variant = fluidHolder.toVariant();
        class_3611 fluidType = fluidHolder.getFluid().comp_349();
        try {
            if (fluidType.method_15780(class_3612.field_15906)) {
                return new ArrayList<>();
            }

            List<class_2561> tooltip = FluidVariantRendering.getTooltip(variant);

            long amount = fluidHolder.getAmount();
            long milliBuckets = (amount * 1000) / FluidHolder.BUCKET_VOLUME;

            if (tooltipMode == FluidHelper.TooltipMode.SHOW_AMOUNT_AND_CAPACITY) {
                class_5250 amountString = class_2561.method_43469(ModonomiconConstants.I18n.Tooltips.FLUID_AMOUNT_AND_CAPACITY, milliBuckets, capacity);
                tooltip.add(amountString.method_27692(class_124.field_1080));
            } else if (tooltipMode == FluidHelper.TooltipMode.SHOW_AMOUNT) {
                class_5250 amountString = class_2561.method_43469(ModonomiconConstants.I18n.Tooltips.FLUID_AMOUNT, milliBuckets);
                tooltip.add(amountString.method_27692(class_124.field_1080));
            }
            return tooltip;
        } catch (RuntimeException e) {
            class_2561 displayName = FluidVariantAttributes.getName(variant);
            Modonomicon.LOG.error("Failed to get tooltip for fluid: " + displayName, e);
        }

        return new ArrayList<>();
    }
}
