/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.client.gui.book.entry;

import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.ModonomiconConstants.I18n.Gui;
import com.klikli_dev.modonomicon.book.Book;
import com.klikli_dev.modonomicon.book.entries.BookContentEntry;
import com.klikli_dev.modonomicon.book.page.BookPage;
import com.klikli_dev.modonomicon.bookstate.BookVisualStateManager;
import com.klikli_dev.modonomicon.bookstate.visual.EntryVisualState;
import com.klikli_dev.modonomicon.client.gui.BookGuiManager;
import com.klikli_dev.modonomicon.client.gui.book.BookAddress;
import com.klikli_dev.modonomicon.client.gui.book.BookContentRenderer;
import com.klikli_dev.modonomicon.client.gui.book.BookPaginatedScreen;
import com.klikli_dev.modonomicon.client.gui.book.BookParentScreen;
import com.klikli_dev.modonomicon.client.gui.book.button.AddBookmarkButton;
import com.klikli_dev.modonomicon.client.gui.book.button.BackButton;
import com.klikli_dev.modonomicon.client.gui.book.button.RemoveBookmarkButton;
import com.klikli_dev.modonomicon.client.gui.book.button.SearchButton;
import com.klikli_dev.modonomicon.client.gui.book.entry.linkhandler.*;
import com.klikli_dev.modonomicon.client.gui.book.entry.linkhandler.LinkHandler.ClickResult;
import com.klikli_dev.modonomicon.client.render.page.BookPageRenderer;
import com.klikli_dev.modonomicon.fluid.FluidHolder;
import com.klikli_dev.modonomicon.integration.jei.ModonomiconJeiIntegration;
import com.klikli_dev.modonomicon.networking.AddBookmarkMessage;
import com.klikli_dev.modonomicon.networking.SyncBookVisualStatesMessage;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.klikli_dev.modonomicon.platform.Services;
import com.klikli_dev.modonomicon.platform.services.FluidHelper;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.glfw.GLFW;

import java.util.Collection;
import java.util.List;
import net.minecraft.class_124;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_364;
import net.minecraft.class_4068;
import net.minecraft.class_6379;
import net.minecraft.class_7919;

public abstract class BookEntryScreen extends BookPaginatedScreen implements ContentRenderingScreen {

    public static final int TOP_PADDING = 15;
    public static final int LEFT_PAGE_X = 12;
    public static final int RIGHT_PAGE_X = 141;
    public static final int SINGLE_PAGE_X = LEFT_PAGE_X;
    public static final int PAGE_WIDTH = 124;
    public static final int PAGE_HEIGHT = 155;

    public static final int MAX_TITLE_WIDTH = PAGE_WIDTH - 4;

    public static final int CLICK_SAFETY_MARGIN = 20;

    protected final BookParentScreen parentScreen;
    protected final BookContentEntry entry;
    protected final class_2960 bookContentTexture;

    protected int ticksInBook;
    protected List<BookPage> unlockedPages;

    /**
     * The index of the leftmost unlocked page being displayed.
     */
    protected int openPagesIndex;
    protected List<LinkHandler> linkHandlers;
    private List<class_2561> tooltip;
    private class_1799 tooltipStack;
    private FluidHolder tooltipFluidStack;
    private boolean isHoveringItemLink;

    public BookEntryScreen(BookParentScreen parentScreen, BookContentEntry entry) {
        super(class_2561.method_43470(""));

        this.parentScreen = parentScreen;

        this.field_22787 = class_310.method_1551();

        this.entry = entry;

        this.bookContentTexture = this.parentScreen.getBook().getBookContentTexture();

        //We're doing that here to ensure unlockedPages is available for state modification during loading
        this.unlockedPages = this.entry.getUnlockedPagesFor(this.field_22787.field_1724);

        this.linkHandlers = List.of(
                new BookLinkHandler(this),
                new PatchouliLinkHandler(this),
                new ItemLinkHandler(this),
                new CommandLinkHandler(this)
        );
    }

    @Override
    public Book getBook() {
        return this.entry.getBook();
    }

    @Override
    public BookContentEntry getEntry() {
        return this.entry;
    }

    @Override
    public class_327 getFont() {
        return this.field_22787.field_1772;
    }

    @Override
    public int getTicksInBook() {
        return this.ticksInBook;
    }

    @Override
    public void setTooltip(List<class_2561> tooltip) {
        this.resetTooltip();
        this.tooltip = tooltip;
    }

    @Override
    public void setTooltipStack(class_1799 stack) {
        this.resetTooltip();
        this.tooltipStack = stack;
    }

    @Override
    public void setTooltipStack(FluidHolder stack) {
        this.resetTooltip();
        this.tooltipFluidStack = stack;
    }

    @Override
    public int getBookLeft() {
        return this.bookLeft;
    }

    @Override
    public int getBookTop() {
        return this.bookTop;
    }

    @Override
    public boolean isHoveringItemLink(){
        return this.isHoveringItemLink;
    }

    @Override
    public void isHoveringItemLink(boolean value){
        this.isHoveringItemLink = value;
    }

    public int getCurrentPageNumber() {
        return this.unlockedPages.get(this.openPagesIndex).getPageNumber();
    }

    public void setOpenPagesIndex(int openPagesIndex) {
        this.openPagesIndex = openPagesIndex;
    }

    /**
     * Will change to the specified page, if not open already
     */
    public void goToPage(int pageIndex, boolean playSound) {
        int openPagesIndex = this.getOpenPagesIndexForPage(pageIndex);
        if (openPagesIndex >= 0 && openPagesIndex < this.unlockedPages.size()) {
            if (this.openPagesIndex != openPagesIndex) {
                this.openPagesIndex = openPagesIndex;

                this.onPageChanged();
                if (playSound) {
                    BookContentRenderer.playTurnPageSound(this.getBook());
                }
            }
        } else {
            Modonomicon.LOG.warn("Tried to change to page index {} corresponding with " +
                    "openPagesIndex {} but max open pages index is {}.", pageIndex, openPagesIndex, this.unlockedPages.size());
        }
    }

    protected class_2583 getClickedComponentStyleAtForPage(BookPageRenderer<?> page, double pMouseX, double pMouseY) {
        if (page != null) {
            return page.getClickedComponentStyleAt(pMouseX - this.bookLeft - page.left, pMouseY - this.bookTop - page.top);
        }

        return null;
    }

    public void removeRenderableWidgets(@NotNull Collection<? extends class_4068> renderables) {
        this.field_33816.removeIf(renderables::contains);
        this.method_25396().removeIf(c -> c instanceof class_4068 && renderables.contains(c));
        this.field_33815.removeIf(n -> n instanceof class_4068 && renderables.contains(n));
    }

    protected void drawTooltip(class_332 guiGraphics, int pMouseX, int pMouseY) {
        if (this.tooltipStack != null) {
            List<class_2561> tooltip = this.getTooltipFromItem(this.tooltipStack);
            guiGraphics.method_51434(class_310.method_1551().field_1772, tooltip, pMouseX, pMouseY);
        } else if (this.tooltipFluidStack != null) {
            List<class_2561> tooltip = this.getTooltipFromFluid(this.tooltipFluidStack);
            guiGraphics.method_51434(class_310.method_1551().field_1772, tooltip, pMouseX, pMouseY);
        } else if (this.tooltip != null && !this.tooltip.isEmpty()) {
            guiGraphics.method_51434(class_310.method_1551().field_1772, this.tooltip, pMouseX, pMouseY);
        }
    }

    protected boolean clickPage(BookPageRenderer<?> page, double mouseX, double mouseY, int mouseButton) {
        if (page != null) {
            return page.mouseClicked(mouseX - this.bookLeft - page.left, mouseY - this.bookTop - page.top, mouseButton);
        }

        return false;
    }

    protected void renderPage(class_332 guiGraphics, BookPageRenderer<?> page, int pMouseX, int pMouseY, float pPartialTick) {
        if (page == null) {
            return;
        }

        guiGraphics.method_51448().method_22903();
        guiGraphics.method_51448().method_46416(page.left, page.top, 0);
        page.render(guiGraphics, pMouseX - this.bookLeft - page.left, pMouseY - this.bookTop - page.top, pPartialTick);
        guiGraphics.method_51448().method_22909();
    }

    protected void onPageChanged() {
        this.beginDisplayPages();
    }

    protected void resetTooltip() {
        this.tooltip = null;
        this.tooltipStack = null;
        this.tooltipFluidStack = null;
    }

    public void loadState(EntryVisualState state) {
        this.openPagesIndex = state.openPagesIndex;
    }

    public void saveState(EntryVisualState state, boolean savePage) {
        state.openPagesIndex = savePage ? this.openPagesIndex : 0;
    }

    @Override
    public boolean method_25404(int key, int scanCode, int modifiers) {
        if (key == GLFW.GLFW_KEY_ESCAPE) {
            BookGuiManager.get().closeScreenStack(this);
            return true;
        }
        return super.method_25404(key, scanCode, modifiers);
    }

    @Override
    public void method_25419() {
        //do not call super, as it would close the screen stack
        //In most cases closeEntryScreen should be called directly, but if our parent BookPaginatedScreen wants us to close we need to handle that
        BookGuiManager.get().closeEntryScreen(this);
    }

    /**
     * Make public to access from pages
     */
    @Override
    public <T extends class_364 & class_4068 & class_6379> T method_37063(T pWidget) {
        return super.method_37063(pWidget);
    }

    public List<class_2561> getTooltipFromItem(class_1799 pItemStack) {
        var tooltip = method_25408(class_310.method_1551(), pItemStack);

        if (this.isHoveringItemLink()) {
            tooltip.add(class_2561.method_43470(""));
            if (ModonomiconJeiIntegration.get().isLoaded()) {
                tooltip.add(class_2561.method_43471(Gui.HOVER_ITEM_LINK_INFO).method_27696(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1060)));
                tooltip.add(class_2561.method_43471(Gui.HOVER_ITEM_LINK_INFO_LINE2).method_27696(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1080)));
            } else {
                tooltip.add(class_2561.method_43471(Gui.HOVER_ITEM_LINK_INFO_NO_JEI).method_27696(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1061)));
            }
        }

        return tooltip;
    }

    public List<class_2561> getTooltipFromFluid(FluidHolder fluidStack) {
        var tooltip = ClientServices.FLUID.getTooltip(fluidStack, FluidHolder.BUCKET_VOLUME, this.field_22787.field_1690.field_1827 ? class_1836.class_1837.field_41071 : class_1836.class_1837.field_41070, FluidHelper.TooltipMode.SHOW_AMOUNT_AND_CAPACITY);

        if (this.isHoveringItemLink()) {
            tooltip.add(class_2561.method_43470(""));
            if (ModonomiconJeiIntegration.get().isLoaded()) {
                tooltip.add(class_2561.method_43471(Gui.HOVER_ITEM_LINK_INFO).method_27696(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1060)));
                tooltip.add(class_2561.method_43471(Gui.HOVER_ITEM_LINK_INFO_LINE2).method_27696(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1080)));
            } else {
                tooltip.add(class_2561.method_43471(Gui.HOVER_ITEM_LINK_INFO_NO_JEI).method_27696(class_2583.field_24360.method_10978(true).method_10977(class_124.field_1061)));
            }
        }

        return tooltip;
    }

    @Override
    public boolean method_25430(@Nullable class_2583 pStyle) {
        if (pStyle != null) {
            for (LinkHandler handler : this.linkHandlers) {
                var result = handler.handleClick(pStyle);

                //before the command pattern was implemented we returned false for failures
                //however, I believe failure should also be treated as "handled" to avoid vanilla code doing fun stuff.
                //We retain the failure result in case that turns out to be wrong
                if (result == LinkHandler.ClickResult.FAILURE)
                    return true;

                if (result == LinkHandler.ClickResult.SUCCESS)
                    return true;

                //unhandled -> continue to next
            }
        }
        return super.method_25430(pStyle);
    }

    @Override
    protected void method_25426() {
        super.method_25426();

        this.unlockedPages = this.entry.getUnlockedPagesFor(this.field_22787.field_1724);
        this.beginDisplayPages();
    }

    @Override
    protected void initNavigationButtons() {
        super.initNavigationButtons();

        this.method_37063(new BackButton(this, this.field_22789 / 2 - BackButton.WIDTH / 2, this.bookTop + FULL_HEIGHT - BackButton.HEIGHT / 2));

        this.updateBookmarksButton();
    }

    protected boolean isBookmarked() {
        return BookVisualStateManager.get().getBookmarksFor(this.field_22787.field_1724, this.entry.getBook()).stream().anyMatch(b -> b.entryId().equals(this.entry.getId()));
    }

    protected void updateBookmarksButton() {
        this.field_33816.removeIf(b -> b instanceof AddBookmarkButton || b instanceof SearchButton);
        this.method_25396().removeIf(b -> b instanceof AddBookmarkButton || b instanceof SearchButton);
        this.field_33815.removeIf(b -> b instanceof AddBookmarkButton || b instanceof SearchButton);

        int buttonHeight = 20;
        int searchButtonX = this.bookLeft + FULL_WIDTH - 5;
        int searchButtonY = this.bookTop + FULL_HEIGHT - 30;
        int searchButtonWidth = 44; //width in png
        int scissorX = this.bookLeft + FULL_WIDTH;//this is the render location of our frame so our search button never overlaps


        if (this.isBookmarked()) {
            var removeBookMarkButton = new RemoveBookmarkButton(this, searchButtonX, searchButtonY,
                    scissorX,
                    searchButtonWidth, buttonHeight,
                    (b) -> this.onRemoveBookmarksButtonClick((RemoveBookmarkButton) b),
                    class_7919.method_47407(class_2561.method_43471(Gui.ADD_BOOKMARK)));
            this.method_37063(removeBookMarkButton);
        } else {
            var addBookmarkButton = new AddBookmarkButton(this, searchButtonX, searchButtonY,
                    scissorX,
                    searchButtonWidth, buttonHeight,
                    (b) -> this.onAddBookmarksButtonClick((AddBookmarkButton) b),
                    class_7919.method_47407(class_2561.method_43471(Gui.ADD_BOOKMARK)));
            this.method_37063(addBookmarkButton);
        }

    }

    protected void onAddBookmarksButtonClick(AddBookmarkButton button) {
        if (!this.isBookmarked()) {
            var bookmarkAddress = BookAddress.ignoreSaved(this.entry, this.getPageForOpenPagesIndex(this.openPagesIndex));
            BookVisualStateManager.get().addBookmarkFor(this.field_22787.field_1724, this.entry.getBook(), bookmarkAddress);

            Services.NETWORK.sendToServer(new AddBookmarkMessage(bookmarkAddress));

            this.updateBookmarksButton();
        }
    }

    protected void onRemoveBookmarksButtonClick(RemoveBookmarkButton button) {
        //no need to check for is bookmarked because we query the bookmark in question directly anyway
        var bookmarkAddress = BookVisualStateManager.get().getBookmarksFor(this.field_22787.field_1724, this.entry.getBook()).stream().filter(b -> b.entryId().equals(this.entry.getId())).findFirst().orElse(null);
        if (bookmarkAddress != null) {
            BookVisualStateManager.get().removeBookmarkFor(this.field_22787.field_1724, this.entry.getBook(), bookmarkAddress);

            this.updateBookmarksButton();
        }
    }

    @Override
    public void method_25393() {
        super.method_25393();

        if (!method_25442()) {
            this.ticksInBook++;
        }
    }

    @Override
    public boolean method_25402(double pMouseX, double pMouseY, int pButton) {
        if (pButton == GLFW.GLFW_MOUSE_BUTTON_LEFT) {
            var style = this.getClickedComponentStyleAt(pMouseX, pMouseY);
            if (style != null && this.method_25430(style)) {
                return true;
            }
        }

        if (super.method_25402(pMouseX, pMouseY, pButton)) {
            return true;
        }

        return this.mouseClickedPage(pMouseX, pMouseY, pButton);
    }

    @Override
    public boolean method_25421() {
        return false;
    }

    public void onSyncBookVisualStatesMessage(SyncBookVisualStatesMessage message) {
        this.updateBookmarksButton();
    }

    protected abstract int getOpenPagesIndexForPage(int pageIndex);

    /**
     * Gets the page index for the first page to display for the given open pages index.
     */
    protected abstract int getPageForOpenPagesIndex(int openPagesIndex);

    @Nullable
    protected abstract class_2583 getClickedComponentStyleAt(double pMouseX, double pMouseY);

    protected abstract boolean mouseClickedPage(double pMouseX, double pMouseY, int pButton);

    protected abstract void beginDisplayPages();
}
