/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.client.gui.book.markdown;

import com.klikli_dev.modonomicon.book.Book;
import com.klikli_dev.modonomicon.client.gui.book.markdown.internal.renderer.ListHolder;
import com.klikli_dev.modonomicon.client.gui.book.markdown.internal.renderer.NodeRendererMap;
import org.commonmark.Extension;
import org.commonmark.node.Node;
import org.commonmark.renderer.NodeRenderer;
import org.commonmark.renderer.text.TextContentRenderer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2588;
import net.minecraft.class_5250;
import net.minecraft.class_5251;
import net.minecraft.class_7225;

public class ComponentRenderer {

    private final List<ComponentNodeRendererFactory> nodeRendererFactories;
    private final List<LinkRenderer> linkRenderers;
    private final List<class_5250> components;
    private final boolean renderSoftLineBreaks;
    private final boolean replaceSoftLineBreaksWithSpace;
    private final class_5251 linkColor;
    private class_5250 currentComponent;
    private class_2583 currentStyle;
    private ListHolder listHolder;

    private ComponentRenderer(ComponentRenderer.Builder builder) {
        this.renderSoftLineBreaks = builder.renderSoftLineBreaks;
        this.replaceSoftLineBreaksWithSpace = builder.replaceSoftLineBreaksWithSpace;
        this.linkColor = builder.linkColor;
        this.currentStyle = builder.style;
        this.linkRenderers = builder.linkRenderers;

        this.components = new ArrayList<>();
        this.currentComponent = class_2561.method_43471("");

        this.nodeRendererFactories = new ArrayList<>(builder.nodeRendererFactories.size() + 1);
        this.nodeRendererFactories.addAll(builder.nodeRendererFactories);
        // Add as last. This means clients can override the rendering of core nodes if they want.
        this.nodeRendererFactories.add(CoreComponentNodeRenderer::new);
    }

    /**
     * Create a new builder for configuring an {@link ComponentRenderer}.
     *
     * @return a builder
     */
    public static ComponentRenderer.Builder builder() {
        return new ComponentRenderer.Builder();
    }

    public List<class_5250> render(Node node, Book book, class_7225.class_7874 provider) {
        ComponentRenderer.RendererContext context = new ComponentRenderer.RendererContext(book, provider);
        context.render(node);
        context.cleanupPostRender();
        return context.getComponents();
    }

    /**
     * Extension for {@link ComponentRenderer}.
     */
    public interface ComponentRendererExtension extends Extension {
        void extend(ComponentRenderer.Builder rendererBuilder);
    }

    /**
     * Builder for configuring an {@link TextContentRenderer}. See methods for default configuration.
     */
    public static class Builder {

        private final List<LinkRenderer> linkRenderers = new ArrayList<>();
        private final List<ComponentNodeRendererFactory> nodeRendererFactories = new ArrayList<>();
        private boolean renderSoftLineBreaks = false;
        private boolean replaceSoftLineBreaksWithSpace = true;
        private class_5251 linkColor = class_5251.method_27717(0x5555FF);
        private class_2583 style = class_2583.field_24360;

        /**
         * @return the configured {@link TextContentRenderer}
         */
        public ComponentRenderer build() {
            return new ComponentRenderer(this);
        }

        /**
         * True to render soft line breaks (deviating from MD spec). Should usually be false.
         */
        public ComponentRenderer.Builder renderSoftLineBreaks(boolean renderSoftLineBreaks) {
            this.renderSoftLineBreaks = renderSoftLineBreaks;
            return this;
        }

        /**
         * True to replace soft line breaks with spaces. Should usually be true, prevents IDE line breaks from causing
         * words to be rendered without spaces inbetween.
         */
        public ComponentRenderer.Builder replaceSoftLineBreaksWithSpace(boolean replaceSoftLineBreaksWithSpace) {
            this.replaceSoftLineBreaksWithSpace = replaceSoftLineBreaksWithSpace;
            return this;
        }

        /**
         * The color to use for http and book page links. Default: Blue: 0x5555FF
         */
        public ComponentRenderer.Builder linkColor(class_5251 linkColor) {
            this.linkColor = linkColor;
            return this;
        }

        /**
         * The style to start rendering with. Will be modified by md instructions.
         */
        public ComponentRenderer.Builder style(class_2583 style) {
            this.style = style;
            return this;
        }

        /**
         * Add a factory for instantiating a node renderer (done when rendering). This allows to override the rendering
         * of node types or define rendering for custom node types.
         * <p>
         * If multiple node renderers for the same node type are created, the one from the factory that was added first
         * "wins". (This is how the rendering for core node types can be overridden; the default rendering comes last.)
         *
         * @param nodeRendererFactory the factory for creating a node renderer
         * @return {@code this}
         */
        public ComponentRenderer.Builder nodeRendererFactory(ComponentNodeRendererFactory nodeRendererFactory) {
            this.nodeRendererFactories.add(nodeRendererFactory);
            return this;
        }

        /**
         * @param extensions extensions to use on this component renderer
         * @return {@code this}
         */
        public ComponentRenderer.Builder extensions(Iterable<? extends Extension> extensions) {
            for (var extension : extensions) {
                if (extension instanceof ComponentRenderer.ComponentRendererExtension componentRendererExtension) {
                    componentRendererExtension.extend(this);
                }
            }
            return this;
        }

        /**
         * @param linkRenderers link renderers to use on this component renderer
         * @return {@code this}
         */
        public ComponentRenderer.Builder linkRenderers(Collection<? extends LinkRenderer> linkRenderers) {
            this.linkRenderers.addAll(linkRenderers);
            return this;
        }
    }

    private class RendererContext implements ComponentNodeRendererContext {

        private final NodeRendererMap nodeRendererMap = new NodeRendererMap();
        private final Book book;
        private final class_7225.class_7874 provider;

        private RendererContext(Book book, class_7225.class_7874 provider) {
            this.book = book;
            this.provider = provider;
            // The first node renderer for a node type "wins".
            for (int i = ComponentRenderer.this.nodeRendererFactories.size() - 1; i >= 0; i--) {
                var nodeRendererFactory = ComponentRenderer.this.nodeRendererFactories.get(i);
                var nodeRenderer = nodeRendererFactory.create(this);
                this.nodeRendererMap.add(nodeRenderer);
            }
        }

        @Override
        public class_5250 getCurrentComponent() {
            return ComponentRenderer.this.currentComponent;
        }

        @Override
        public void setCurrentComponent(class_5250 component) {
            ComponentRenderer.this.currentComponent = component;
        }

        @Override
        public List<class_5250> getComponents() {
            return ComponentRenderer.this.components;
        }

        @Override
        public ListHolder getListHolder() {
            return ComponentRenderer.this.listHolder;
        }

        @Override
        public void setListHolder(ListHolder listHolder) {
            ComponentRenderer.this.listHolder = listHolder;
        }

        @Override
        public class_2583 getCurrentStyle() {
            return ComponentRenderer.this.currentStyle;
        }

        @Override
        public void setCurrentStyle(class_2583 style) {
            ComponentRenderer.this.currentStyle = style;
        }

        @Override
        public void render(Node node) {
            this.nodeRendererMap.render(node);
        }

        /**
         * Needs to be called after rendering to handle the last component.
         */
        @Override
        public void cleanupPostRender() {
            if (!this.isEmptyComponent()) {
                this.finalizeCurrentComponent();
            }
        }

        public boolean isEmptyComponent() {
            //translation contents have no content, they have a key (which doubles as content).
            return ((class_2588) this.getCurrentComponent().method_10851()).method_11022().isEmpty() && this.getCurrentComponent().method_10855().isEmpty();
        }

        public void finalizeCurrentComponent() {
            this.getComponents().add(this.getCurrentComponent());
            this.setCurrentComponent(this.getListHolder() == null ?
                    class_2561.method_43471("") : class_5250.method_43477(new ListItemContents(this.getListHolder(), "")));
        }

        @Override
        public boolean getRenderSoftLineBreaks() {
            return ComponentRenderer.this.renderSoftLineBreaks;
        }

        @Override
        public boolean getReplaceSoftLineBreaksWithSpace() {
            return ComponentRenderer.this.replaceSoftLineBreaksWithSpace;
        }

        @Override
        public class_5251 getLinkColor() {
            return ComponentRenderer.this.linkColor;
        }

        @Override
        public List<LinkRenderer> getLinkRenderers() {
            return ComponentRenderer.this.linkRenderers;
        }

        @Override
        public Book getBook() {
            return this.book;
        }

        @Override
        public class_7225.class_7874 getProvider() {
            return this.provider;
        }
    }
}
