/*
 * SPDX-FileCopyrightText: 2022 Authors of Patchouli
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.client.render;

import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.klikli_dev.modonomicon.api.multiblock.Multiblock;
import com.klikli_dev.modonomicon.api.multiblock.MultiblockPreviewData;
import com.klikli_dev.modonomicon.client.ClientTicks;
import com.klikli_dev.modonomicon.multiblock.AbstractMultiblock;
import com.klikli_dev.modonomicon.multiblock.matcher.DisplayOnlyMatcher;
import com.klikli_dev.modonomicon.multiblock.matcher.Matchers;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.klikli_dev.modonomicon.util.GuiGraphicsExt;
import com.mojang.datafixers.util.Pair;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.*;
import java.util.*;
import java.util.function.Function;
import net.minecraft.class_1074;
import net.minecraft.class_1109;
import net.minecraft.class_11658;
import net.minecraft.class_11954;
import net.minecraft.class_12075;
import net.minecraft.class_1268;
import net.minecraft.class_1269;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1921;
import net.minecraft.class_1937;
import net.minecraft.class_2246;
import net.minecraft.class_2338;
import net.minecraft.class_2343;
import net.minecraft.class_2470;
import net.minecraft.class_2561;
import net.minecraft.class_2586;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3417;
import net.minecraft.class_3532;
import net.minecraft.class_3965;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_898;
import net.minecraft.class_9799;

public class MultiblockPreviewRenderer {

    private static final Map<class_2338, class_2586> blockEntityCache = new Object2ObjectOpenHashMap<>();
    private static final Set<class_2586> erroredBlockEntities = Collections.newSetFromMap(new WeakHashMap<>());
    private static final List<class_11954> blockEntityRenderStates = new ArrayList<>();
    public static boolean hasMultiblock;
    private static Multiblock multiblock;
    private static class_2561 name;
    private static class_2338 pos;
    private static boolean isAnchored;
    private static class_2470 facingRotation;
    private static Function<class_2338, class_2338> offsetApplier;
    private static int blocks, blocksDone, airFilled;
    private static int timeComplete;
    private static class_2680 lookingState;
    private static class_2338 lookingPos;
    private static class_4597.class_4598 buffers = null;

    public static void setMultiblock(Multiblock multiblock, class_2561 name, boolean flip) {
        setMultiblock(multiblock, name, flip, pos -> pos);
    }

    public static void setMultiblock(Multiblock multiblock, class_2561 name, boolean flip, Function<class_2338, class_2338> offsetApplier) {
        if (flip && hasMultiblock && MultiblockPreviewRenderer.multiblock == multiblock) {
            hasMultiblock = false;
        } else {
            MultiblockPreviewRenderer.multiblock = multiblock;
            MultiblockPreviewRenderer.blockEntityCache.clear();
            MultiblockPreviewRenderer.erroredBlockEntities.clear();
            MultiblockPreviewRenderer.name = name;
            MultiblockPreviewRenderer.offsetApplier = offsetApplier;
            pos = null;
            hasMultiblock = multiblock != null;
            isAnchored = false;
        }
    }

    public static void onRenderHUD(class_332 guiGraphics, float partialTicks) {
        if (hasMultiblock) {
            int waitTime = 40;
            int fadeOutSpeed = 4;
            int fullAnimTime = waitTime + 10;
            float animTime = timeComplete + (timeComplete == 0 ? 0 : partialTicks);

            if (animTime > fullAnimTime) {
                hasMultiblock = false;
                return;
            }

            guiGraphics.method_51448().pushMatrix();
            guiGraphics.method_51448().translate(0, -Math.max(0, animTime - waitTime) * fadeOutSpeed);

            class_310 mc = class_310.method_1551();
            int x = mc.method_22683().method_4486() / 2;
            int y = 12;

            GuiGraphicsExt.drawString(guiGraphics, mc.field_1772, name, x - mc.field_1772.method_27525(name) / 2.0F, y, -1, false);

            int width = 180;
            int height = 9;
            int left = x - width / 2;
            int top = y + 10;

            if (timeComplete > 0) {
                String s = class_1074.method_4662(ModonomiconConstants.I18n.Multiblock.COMPLETE);
                guiGraphics.method_51448().pushMatrix();
                guiGraphics.method_51448().translate(0, Math.min(height + 5, animTime));
                guiGraphics.method_51433(mc.field_1772, s, (int) (x - mc.field_1772.method_1727(s) / 2.0F), top + height - 10, 0x00FF00, false);
                guiGraphics.method_51448().popMatrix();
            }

            //render a black square at the "bottom", 1px larger than the actual progress bar, so it acts as a border
            guiGraphics.method_25294(left - 1, top - 1, left + width + 1, top + height + 1, 0xFF000000);

            //then, on top of that, render a gray gradient as "empty progress"
            guiGraphics.method_25296(left, top, left + width, top + height, 0xFF666666, 0xFF666666);

            float fract = (float) blocksDone / Math.max(1, blocks);
            int progressWidth = (int) ((float) width * fract);
            int color = class_3532.method_15369(fract / 3.0F, 1.0F, 1.0F) | 0xFF000000;
            int color2 = new Color(color).darker().getRGB();

            //finally, on top of that, render a colored gradient as "filled progress"
            guiGraphics.method_25296(left, top, left + progressWidth, top + height, color, color2);

            if (!isAnchored) {
                String s = class_1074.method_4662(ModonomiconConstants.I18n.Multiblock.NOT_ANCHORED);
                guiGraphics.method_51433(mc.field_1772, s, (int) (x - mc.field_1772.method_1727(s) / 2.0F), top + height + 8, 0xFFFFFF, false);
            } else {
                if (lookingState != null) {
                    // try-catch around here because the state isn't necessarily present in the world in this instance,
                    // which isn't really expected behavior for getPickBlock
                    try {
                        var stack = lookingState.method_65171(mc.field_1687, lookingPos, true);
                        if (!stack.method_7960()) {
                            guiGraphics.method_51439(mc.field_1772, stack.method_7964(), left + 20, top + height + 8, 0xFFFFFF, false);

                            guiGraphics.method_51427(stack, left, top + height + 2);
                        }
                    } catch (Exception ignored) {
                    }
                }

                if (timeComplete == 0) {
                    color = 0xFFFFFF;
                    int posx = left + width;
                    int posy = top + height + 2;
                    float mult = 1;
                    String progress = blocksDone + "/" + blocks;

                    if (blocksDone == blocks && airFilled > 0) {
                        progress = class_1074.method_4662(ModonomiconConstants.I18n.Multiblock.REMOVE_BLOCKS);
                        color = 0xDA4E3F;
                        mult *= 2;
                        posx -= width / 2;
                        posy += 2;
                    }

                    guiGraphics.method_51433(mc.field_1772, progress, (int) (posx - mc.field_1772.method_1727(progress) / mult), posy, color, true);
                }
            }

            guiGraphics.method_51448().popMatrix();
        }
    }

    public static void onRenderLevelLastEvent(class_11658 levelRenderState, class_4587 poseStack) {
        if (hasMultiblock && multiblock != null) {
            renderMultiblock(levelRenderState, poseStack);
        }
    }

    public static void anchorTo(class_2338 target, class_2470 rot) {
        pos = target;
        facingRotation = rot;
        isAnchored = true;
    }

    public static class_1269 onPlayerInteract(class_1657 player, class_1937 world, class_1268 hand, class_3965 hit) {
        if (hasMultiblock && !isAnchored && player == class_310.method_1551().field_1724) {
            anchorTo(hit.method_17777(), getRotation(player));
            return class_1269.field_5812;
        }
        return class_1269.field_5811;
    }

    public static void onClientTick(class_310 mc) {
        if (class_310.method_1551().field_1687 == null) {
            hasMultiblock = false;
        } else if (isAnchored && blocks == blocksDone && airFilled == 0) {
            timeComplete++;
            if (timeComplete == 14) {
                class_310.method_1551().method_1483().method_4873(class_1109.method_4758(class_3417.field_14627, 1.0F));
            }
        } else {
            timeComplete = 0;
        }
    }

    /**
     * Phase 1: Extract render state from the level.
     * This should be called during the render state extraction phase.
     *
     * @param levelRenderState The level render state to extract data into
     */
    public static void extractRenderState(class_11658 levelRenderState) {
        if (!hasMultiblock || multiblock == null) {
            return;
        }

        blockEntityRenderStates.clear();

        class_310 mc = class_310.method_1551();
        class_1937 level = mc.field_1687;
        if (level == null) {
            return;
        }

        if (!isAnchored) {
            facingRotation = getRotation(mc.field_1724);
            if (mc.field_1765 instanceof class_3965) {
                pos = ((class_3965) mc.field_1765).method_17777();
            }
        } else if (pos.method_19770(mc.field_1724.method_73189()) > 64 * 64) {
            return;
        }

        if (pos == null) {
            return;
        }
        if (multiblock.isSymmetrical()) {
            facingRotation = class_2470.field_11467;
        }

        multiblock.setLevel(level);

        // Extract block entity render states from the simulated multiblock
        class_2338 startPos = getStartPos();
        Pair<class_2338, Collection<Multiblock.SimulateResult>> sim = multiblock.simulate(level, startPos, getFacingRotation(), true, false);
        for (Multiblock.SimulateResult r : sim.getSecond()) {
            try {
                class_2680 displayedState = r.getStateMatcher().getDisplayedState(ClientTicks.ticks).method_26186(facingRotation);

                if (displayedState.method_26204() instanceof class_2343 eb) {
                    // Cache/create a fake block entity at the simulated position (translate by startPos)
                    var cacheKey = r.getWorldPosition().method_10059(startPos).method_10062();
                    var be = blockEntityCache.compute(cacheKey, (p, cachedBe) -> {
                        if (cachedBe != null && !cachedBe.method_11017().method_20526(displayedState)) {
                            return eb.method_10123(p, displayedState);
                        }
                        return cachedBe != null ? cachedBe : eb.method_10123(p, displayedState);
                    });

                    if (be != null && !erroredBlockEntities.contains(be)) {
                        be.method_31662(mc.field_1687);
                        // Provide the displayed state to avoid querying the real world
                        be.method_31664(displayedState);

                        var dispatcher = class_310.method_1551().method_31975();
                        var renderer = dispatcher.method_3550(be);
                        if (renderer != null) {
                            var renderState = renderer.method_74335();
                            var eye = mc.method_1561().field_4686.method_19326();
                            eye = eye.method_1023(startPos.method_10263(), startPos.method_10264(), startPos.method_10260());

                            //Note: we cannot use Minecraft.getInstance().getBlockEntityRenderDispatcher().tryExtractRenderState because that takes the camera eye position of the in-world camera, but our multiblock exists in a virtual level close to 0 0 0
                            renderer.method_74331(be, renderState, ClientTicks.partialTicks, eye, null);
                            blockEntityRenderStates.add(renderState);
                        }
                    }
                }
            } catch (Exception e) {
                // Don't let a failure extracting one block entity abort the entire extraction loop
                Modonomicon.LOG.error("Error extracting block entity render state", e);
            }
        }
    }


    /**
     * Phase 2: Render the multiblock using the extracted render state.
     * This should be called during the actual rendering phase with the levelRenderState.
     *
     * @param levelRenderState The level render state containing extracted data
     * @param ms               The pose stack for rendering
     */
    public static void renderMultiblock(class_11658 levelRenderState, class_4587 ms) {
        if (!hasMultiblock || multiblock == null) {
            return;
        }

        class_310 mc = class_310.method_1551();
        class_1937 level = mc.field_1687;
        if (level == null) {
            return;
        }

        if (!isAnchored) {
            if (mc.field_1765 instanceof class_3965) {
                pos = ((class_3965) mc.field_1765).method_17777();
            }
        } else if (pos.method_19770(mc.field_1724.method_73189()) > 64 * 64) {
            return;
        }

        if (pos == null) {
            return;
        }

        class_898 erd = mc.method_1561();
        double renderPosX = erd.field_4686.method_19326().method_10216();
        double renderPosY = erd.field_4686.method_19326().method_10214();
        double renderPosZ = erd.field_4686.method_19326().method_10215();
        ms.method_22903();
        ms.method_22904(-renderPosX, -renderPosY, -renderPosZ);

        if (buffers == null) {
            buffers = initBuffers(mc.method_22940().method_23000());
        }

        class_2338 checkPos = null;
        if (mc.field_1765 instanceof class_3965 blockRes) {
            checkPos = blockRes.method_17777().method_10093(blockRes.method_17780());
        }

        class_2338 startPos = getStartPos();

        Pair<class_2338, Collection<Multiblock.SimulateResult>> sim = multiblock.simulate(level, startPos, getFacingRotation(), true, false);
        for (Multiblock.SimulateResult r : sim.getSecond()) {
            float alpha = 0.3F;
            if (r.getWorldPosition().equals(checkPos)) {
                alpha = 0.6F + (float) (Math.sin(ClientTicks.total * 0.3F) + 1F) * 0.1F;
            }

            if (!r.getStateMatcher().equals(Matchers.ANY) && r.getStateMatcher().getType() != DisplayOnlyMatcher.TYPE) {
                boolean air = !r.getStateMatcher().countsTowardsTotalBlocks();

                if (!r.test(level, facingRotation)) {
                    class_2680 displayedState = r.getStateMatcher().getDisplayedState(ClientTicks.ticks).method_26186(facingRotation);
                    renderBlock(level, displayedState, r.getWorldPosition(), multiblock, air, alpha, ms);
                }
            }
        }

        // Render block entities using the extracted render states
        for (var blockEntityRenderState : blockEntityRenderStates) {
            ms.method_22903();
            ms.method_46416(blockEntityRenderState.field_62673.method_10263(),
                    blockEntityRenderState.field_62673.method_10264(),
                    blockEntityRenderState.field_62673.method_10260());

            //TODO: We see no BEs in the world preview (GUI works though ... )

            var dispatcher = class_310.method_1551().method_31975();
            var featureDispatcher = class_310.method_1551().field_1773.method_72911();
            var cameraRenderState = new class_12075();
            dispatcher.method_3555(blockEntityRenderState, ms, featureDispatcher.method_73003(), cameraRenderState);
            featureDispatcher.method_73002();

            ms.method_22909();
        }
        blockEntityRenderStates.clear();

        buffers.method_22993();
        ms.method_22909();
    }

    public static void renderBlock(class_1937 world, class_2680 state, class_2338 pos, Multiblock multiblock, boolean isAir, float alpha, class_4587 ms) {
        if (pos != null) {
            ms.method_22903();
            ms.method_46416(pos.method_10263(), pos.method_10264(), pos.method_10260());

            if (state.method_26204() == class_2246.field_10124) {
                float scale = 0.3F;
                float off = (1F - scale) / 2;
                ms.method_46416(off, off, off);
                ms.method_22905(scale, scale, scale);

                state = class_2246.field_10058.method_9564();
            }

            ClientServices.MULTIBLOCK.renderBlock(state, pos, multiblock, ms, buffers, world.method_8409());

//            Minecraft.getInstance().getBlockRenderer().renderSingleBlock(state, ms, buffers, 0xF000F0, OverlayTexture.NO_OVERLAY);

            ms.method_22909();
        }
    }

    @Nullable
    public static MultiblockPreviewData getMultiblockPreviewData() {
        if (!hasMultiblock) {
            return null;
        }
        return new MultiblockPreviewData(multiblock, pos, facingRotation, isAnchored);
    }

    public static Multiblock getMultiblock() {
        return multiblock;
    }

    public static boolean isAnchored() {
        return isAnchored;
    }

    public static class_2470 getFacingRotation() {
        return multiblock.isSymmetrical() ? class_2470.field_11467 : facingRotation;
    }

    public static class_2338 getStartPos() {
        return offsetApplier.apply(pos);
    }

    /**
     * Returns the Rotation of a multiblock structure based on the given entity's facing direction.
     */
    private static class_2470 getRotation(class_1297 entity) {
        return AbstractMultiblock.rotationFromFacing(entity.method_5735());
    }

    private static class_4597.class_4598 initBuffers(class_4597.class_4598 original) {
        var fallback = original.field_52156;
        var layerBuffers = original.field_20953;
        return new GhostBuffers(fallback, layerBuffers);
    }

    private static class GhostBuffers extends class_4597.class_4598 {
        protected GhostBuffers(class_9799 fallback, SequencedMap<class_1921, class_9799> layerBuffers) {
            super(fallback, layerBuffers);
        }

        @Override
        public @NotNull class_4588 method_73477(@NotNull class_1921 type) {
            return GhostVertexConsumer.remap(super.method_73477(type));
        }
    }
}
