/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book;

import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.klikli_dev.modonomicon.api.ModonomiconConstants;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.minecraft.class_10799;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2960;
import net.minecraft.class_332;
import net.minecraft.class_3518;
import net.minecraft.class_7225;
import net.minecraft.class_7923;
import net.minecraft.class_9129;
import net.minecraft.class_9326;

public class BookIcon {

    /**
     * A custom codec that still uses the "item" field instead of "id" for backwards comp,
     */
    public static final Codec<class_1799> CUSTOM_ITEM_STACK_CODEC = RecordCodecBuilder.create((builder) -> builder.group(
            class_1792.field_54952.fieldOf("item").forGetter(class_1799::method_41409),
            Codec.INT.optionalFieldOf("count", 1).forGetter(class_1799::method_7947),
            class_9326.field_49589.optionalFieldOf("components", class_9326.field_49588).forGetter(class_1799::method_57380)
    ).apply(builder, class_1799::new));

    /**
     * We allow both vanilla item stack syntax and our custom syntax.
     */
    public static final Codec<class_1799> ITEM_STACK_CODEC = Codec.withAlternative(CUSTOM_ITEM_STACK_CODEC, class_1799.field_24671);

    private final class_1799 itemStack;
    private final class_2960 texture;

    private final int width;
    private final int height;

    public BookIcon(class_1799 stack) {
        this.itemStack = stack;
        this.texture = null;
        this.width = ModonomiconConstants.Data.Icon.DEFAULT_WIDTH;
        this.height = ModonomiconConstants.Data.Icon.DEFAULT_HEIGHT;
    }

    public BookIcon(class_2960 texture, int width, int height) {
        this.texture = texture;
        this.itemStack = class_1799.field_8037;
        this.width = width;
        this.height = height;
    }

    public static BookIcon fromJson(JsonElement jsonElement, class_7225.class_7874 provider) {
        //if string -> use from string
        //if json object -> parse from json
        if (jsonElement.isJsonPrimitive()) {
            return fromString(class_2960.method_60654(jsonElement.getAsString()));
        }

        var jsonObject = jsonElement.getAsJsonObject();
        if (jsonObject.has("texture")) {
            var width = class_3518.method_15282(jsonObject, "width", ModonomiconConstants.Data.Icon.DEFAULT_WIDTH);
            var height = class_3518.method_15282(jsonObject, "height", ModonomiconConstants.Data.Icon.DEFAULT_HEIGHT);
            var texture = class_2960.method_60654(class_3518.method_15265(jsonObject, "texture"));
            return new BookIcon(texture, width, height);
        } else {
            var stack = ITEM_STACK_CODEC.decode(provider.method_57093(JsonOps.INSTANCE), jsonObject).getOrThrow((e) -> {
                throw new JsonParseException("BookIcon must have either item or texture defined." + jsonElement, new Throwable(e));
            }).getFirst();

            return new BookIcon(stack);
        }
    }

    private static BookIcon fromString(class_2960 value) {
        if (value.method_12832().endsWith(".png")) {
            return new BookIcon(value, ModonomiconConstants.Data.Icon.DEFAULT_WIDTH, ModonomiconConstants.Data.Icon.DEFAULT_HEIGHT);
        } else {
            class_1792 item = class_7923.field_41178.method_63535(value);
            return new BookIcon(new class_1799(item));
        }
    }

    public static BookIcon fromNetwork(class_9129 buffer) {
        if (buffer.readBoolean()) {
            class_2960 texture = buffer.method_10810();
            int width = buffer.method_10816();
            int height = buffer.method_10816();
            return new BookIcon(texture, width, height);
        }

        var stack = class_1799.field_48349.decode(buffer);
        return new BookIcon(stack);
    }

    public void render(class_332 guiGraphics, int x, int y) {
        if (this.texture != null) {
            //1.21.3+ parameter order taken from ImageWidget#renderWidget
            guiGraphics.method_25302(
                    class_10799.field_56883,
                    this.texture,
                    x, y,
                    0, 0,
                   16, 16,
                    this.width, this.height,
                    this.width, this.height
            );

            //    guiGraphics.blit(RenderPipelines.GUI_TEXTURED, this.page.getImages()[this.index], x * 2 + 6, y * 2 + 6, 0, 0, 200, 200, 200, 200, 200, 200);
        } else {
            guiGraphics.method_51427(this.itemStack, x, y);
        }
    }

    public void toNetwork(class_9129 buffer) {
        buffer.method_52964(this.texture != null);
        if (this.texture != null) {
            buffer.method_10812(this.texture);
            buffer.method_10804(this.width);
            buffer.method_10804(this.height);
        } else {
            class_1799.field_48349.encode(buffer, this.itemStack);
        }
    }
}
