/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.book.page;

import com.google.gson.JsonObject;
import com.klikli_dev.modonomicon.api.ModonomiconConstants.Data.Page;
import com.klikli_dev.modonomicon.book.BookTextHolder;
import com.klikli_dev.modonomicon.book.RenderedBookTextHolder;
import com.klikli_dev.modonomicon.book.conditions.BookCondition;
import com.klikli_dev.modonomicon.book.conditions.BookNoneCondition;
import com.klikli_dev.modonomicon.book.entries.BookContentEntry;
import com.klikli_dev.modonomicon.client.gui.book.markdown.BookTextRenderer;
import com.klikli_dev.modonomicon.util.BookGsonHelper;
import com.mojang.datafixers.util.Either;
import com.mojang.serialization.Codec;
import com.mojang.serialization.JsonOps;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import net.minecraft.class_10363;
import net.minecraft.class_1074;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_1937;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_5250;
import net.minecraft.class_5699;
import net.minecraft.class_7225;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.class_9326;
import org.jetbrains.annotations.NotNull;

public class BookSpotlightPage extends BookPage {
    /**
     * A custom codec that still uses the "item" field instead of "id" for backwards comp,
     */
    public static final Codec<class_1799> CUSTOM_ITEM_STACK_CODEC = Codec.lazyInitialized(
            () -> RecordCodecBuilder.create((builder) -> builder.group(
                    class_1792.field_54952.fieldOf("item").forGetter(class_1799::method_41409),
                    class_5699.method_48766(1, 99).fieldOf("count").orElse(1).forGetter(class_1799::method_7947),
                    class_9326.field_49589.optionalFieldOf("components", class_9326.field_49588).forGetter(class_1799::method_57380)
            ).apply(builder, class_1799::new))
    );
    
    /**
     * We allow both vanilla item stack syntax and our custom syntax.
     */
    public static final Codec<class_1799> ITEM_STACK_CODEC = Codec.lazyInitialized(() -> Codec.withAlternative(CUSTOM_ITEM_STACK_CODEC, class_1799.field_24671));

    /**
     * We allow both ingredients and item stacks.
     */
    public static final Codec<Either<class_1799, class_1856>> ITEM_CODEC = Codec.lazyInitialized(() -> Codec.either(ITEM_STACK_CODEC, class_1856.field_46095));

    public static final class_9139<class_9129, Either<class_1799, class_1856>> ITEM_STREAM_CODEC = new class_9139<>() {


        @Override
        public void encode(@NotNull class_9129 buf, Either<class_1799, class_1856> item) {
            item.ifRight(i -> {
                buf.method_52964(true);
                class_1856.field_48355.encode(buf, i);
            });
            item.ifLeft(i -> {
                buf.method_52964(false);
                class_1799.field_48349.encode(buf, i);
            });
        }

        @Override
        public @NotNull Either<class_1799, class_1856> decode(@NotNull class_9129 buf) {
            boolean isIngredient = buf.readBoolean();
            if (isIngredient) {
                return Either.right(class_1856.field_48355.decode(buf));
            } else {
                return Either.left(class_1799.field_48349.decode(buf));
            }
        }
    };

    protected BookTextHolder title;
    protected BookTextHolder text;
    protected Either<class_1799, class_1856> item;

    public BookSpotlightPage(BookTextHolder title, BookTextHolder text, Either<class_1799, class_1856> item, String anchor, BookCondition condition) {
        super(anchor, condition);
        this.title = title;
        this.text = text;
        this.item = item;
    }

    public static BookSpotlightPage fromJson(class_2960 entryId, JsonObject json, class_7225.class_7874 provider) {
        var title = BookGsonHelper.getAsBookTextHolder(json, "title", BookTextHolder.EMPTY, provider);
        var item = ITEM_CODEC.parse(provider.method_57093(JsonOps.INSTANCE), json.get("item")).result().get();
        var text = BookGsonHelper.getAsBookTextHolder(json, "text", BookTextHolder.EMPTY, provider);
        var anchor = class_3518.method_15253(json, "anchor", "");
        var condition = json.has("condition")
                ? BookCondition.fromJson(entryId, json.getAsJsonObject("condition"), provider)
                : new BookNoneCondition();
        return new BookSpotlightPage(title, text, item, anchor, condition);
    }

    public static BookSpotlightPage fromNetwork(class_9129 buffer) {
        var title = BookTextHolder.fromNetwork(buffer);
        var item = ITEM_STREAM_CODEC.decode(buffer);
        var text = BookTextHolder.fromNetwork(buffer);
        var anchor = buffer.method_19772();
        var condition = BookCondition.fromNetwork(buffer);
        return new BookSpotlightPage(title, text, item, anchor, condition);
    }

    public Either<class_1799, class_1856> getItem() {
        return this.item;
    }

    public BookTextHolder getTitle() {
        return this.title;
    }

    public BookTextHolder getText() {
        return this.text;
    }

    public boolean hasTitle() {
        return !this.title.isEmpty();
    }

    @Override
    public class_2960 getType() {
        return Page.SPOTLIGHT;
    }

    @Override
    public void build(class_1937 level, BookContentEntry parentEntry, int pageNum) {
        super.build(level, parentEntry, pageNum);

        if (this.title.isEmpty()) {
            //use ingredient name if we don't have a custom title
            var item = this.item.map(i -> i, i -> i.method_64673().method_64742(class_10363.method_65008(level)));

            this.title = new BookTextHolder(((class_5250) item.method_7964())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getBook().getDefaultTitleColor())
                    ));
        }
    }

    @Override
    public void prerenderMarkdown(BookTextRenderer textRenderer) {
        super.prerenderMarkdown(textRenderer);

        if (!this.title.hasComponent()) {
            this.title = new BookTextHolder(class_2561.method_43471(this.title.getKey())
                    .method_27696(class_2583.field_24360
                            .method_10982(true)
                            .method_36139(this.getParentEntry().getBook().getDefaultTitleColor())));
        }
        if (!this.text.hasComponent()) {
            this.text = new RenderedBookTextHolder(this.text, textRenderer.render(this.text.getString()));
        }
    }

    @Override
    public void toNetwork(class_9129 buffer) {
        this.title.toNetwork(buffer);
        ITEM_STREAM_CODEC.encode(buffer, this.item);
        this.text.toNetwork(buffer);
        super.toNetwork(buffer);
    }

    @Override
    public boolean matchesQuery(String query, class_1937 level) {
        return this.title.getString().toLowerCase().contains(query)
                || this.itemStackMatchesQuery(query)
                || this.ingredientMatchesQuery(query, level)
                || this.text.getString().toLowerCase().contains(query);
    }

    protected boolean itemStackMatchesQuery(String query) {
        return this.item.mapLeft(l -> this.matchesQuery(l, query)).left().orElse(false);
    }

    protected boolean ingredientMatchesQuery(String query, class_1937 level) {
        return this.item.mapRight(r -> r.method_64673().method_64738(class_10363.method_65008(level)).stream().anyMatch(i -> this.matchesQuery(i, query))).right().orElse(false);
    }

    protected boolean matchesQuery(class_1799 stack, String query) {
        return class_1074.method_4662(stack.method_7909().method_7876()).toLowerCase().contains(query);
    }
}
