/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.client.gui.book;

import com.klikli_dev.modonomicon.book.Book;
import com.klikli_dev.modonomicon.book.BookCategory;
import com.klikli_dev.modonomicon.book.entries.BookEntry;
import com.klikli_dev.modonomicon.util.StreamCodecs;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import org.jetbrains.annotations.NotNull;

import java.util.Optional;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_9135;
import net.minecraft.class_9139;

/**
 * Represents an address in a book, consisting of the book, category, entry and page.
 * Used to navigate to a specific page in a book and to store such a state
 */
@SuppressWarnings("OptionalUsedAsFieldOrParameterType")
public record BookAddress(@NotNull class_2960 bookId,
                          class_2960 categoryId, boolean ignoreSavedCategory,
                          class_2960 entryId, boolean ignoreSavedEntry,
                          int page, boolean ignoreSavedPage
) {
    public static final Codec<BookAddress> CODEC = RecordCodecBuilder.create(instance -> instance.group(
            class_2960.field_25139.fieldOf("bookId").forGetter(BookAddress::bookId),
            class_2960.field_25139.optionalFieldOf("categoryId").forGetter((address) -> Optional.ofNullable(address.categoryId)),
            Codec.BOOL.fieldOf("ignoreSavedCategory").forGetter(BookAddress::ignoreSavedCategory),
            class_2960.field_25139.optionalFieldOf("entryId").forGetter((address) -> Optional.ofNullable(address.entryId)),
            Codec.BOOL.fieldOf("ignoreSavedEntry").forGetter(BookAddress::ignoreSavedEntry),
            Codec.INT.fieldOf("page").forGetter(BookAddress::page),
            Codec.BOOL.fieldOf("ignoreSavedPage").forGetter(BookAddress::ignoreSavedPage)
    ).apply(instance, BookAddress::new));

    public static final class_9139<class_2540, BookAddress> STREAM_CODEC = StreamCodecs.composite(
            class_2960.field_48267,
            BookAddress::bookId,
            class_9135.method_56382(class_2960.field_48267),
            (address) -> Optional.ofNullable(address.categoryId),
            class_9135.field_48547,
            BookAddress::ignoreSavedCategory,
            class_9135.method_56382(class_2960.field_48267),
            (address) -> Optional.ofNullable(address.entryId),
            class_9135.field_48547,
            BookAddress::ignoreSavedEntry,
            class_9135.field_49675,
            BookAddress::page,
            class_9135.field_48547,
            BookAddress::ignoreSavedPage,
            BookAddress::new
    );

    private BookAddress(@NotNull class_2960 bookId,
                        Optional<class_2960> categoryId, boolean ignoreSavedCategory,
                        Optional<class_2960> entryId, boolean ignoreSavedEntry,
                        int page, boolean ignoreSavedPage
    ) {
        this(bookId, categoryId.orElse(null), ignoreSavedCategory, entryId.orElse(null), ignoreSavedEntry, page, ignoreSavedPage);
    }

    public static BookAddress ignoreSaved(@NotNull BookEntry entry, int page) {
        return ignoreSaved(entry.getBook().getId(), entry.getCategory().getId(), entry.getId(), page);
    }


    public static BookAddress ignoreSaved(@NotNull BookEntry entry) {
        return ignoreSaved(entry, -1);
    }

    public static BookAddress defaultFor(@NotNull BookCategory category) {
        return of(category.getBook().getId(), category.getId(), null, -1);
    }

    public static BookAddress defaultFor(@NotNull BookEntry entry) {
        return of(entry.getBook().getId(), entry.getCategory().getId(), entry.getId(), -1);
    }

    public static BookAddress defaultFor(@NotNull Book book) {
        return defaultFor(book.getId());
    }

    public static BookAddress defaultFor(@NotNull class_2960 bookId) {
        return of(bookId, null, null, -1);
    }

    public static BookAddress of(@NotNull class_2960 bookId,
                                 class_2960 categoryId,
                                 class_2960 entryId,
                                 int page) {
        return new BookAddress(bookId, categoryId, false, entryId, false, page, false);
    }

    public static BookAddress ignoreSaved(@NotNull class_2960 bookId,
                                          class_2960 categoryId,
                                          class_2960 entryId,
                                          int page) {
        return new BookAddress(bookId, categoryId, true, entryId, true, page, true);
    }

    public BookAddress withPage(int page) {
        return new BookAddress(this.bookId, this.categoryId, this.ignoreSavedCategory, this.entryId, this.ignoreSavedEntry, page, this.ignoreSavedPage);
    }
}