/*
 * SPDX-FileCopyrightText: 2022 klikli-dev
 * SPDX-FileCopyrightText: 2021 Authors of Arcana
 *
 * SPDX-License-Identifier: MIT
 */
package com.klikli_dev.modonomicon.client.gui.book.node;

import com.klikli_dev.modonomicon.api.events.EntryClickedEvent;
import com.klikli_dev.modonomicon.book.BookCategory;
import com.klikli_dev.modonomicon.book.BookCategoryBackgroundParallaxLayer;
import com.klikli_dev.modonomicon.book.conditions.context.BookConditionEntryContext;
import com.klikli_dev.modonomicon.book.entries.BookEntry;
import com.klikli_dev.modonomicon.bookstate.BookUnlockStateManager;
import com.klikli_dev.modonomicon.bookstate.visual.CategoryVisualState;
import com.klikli_dev.modonomicon.client.gui.BookGuiManager;
import com.klikli_dev.modonomicon.client.gui.book.BookAddress;
import com.klikli_dev.modonomicon.client.gui.book.BookCategoryScreen;
import com.klikli_dev.modonomicon.client.gui.book.BookContentRenderer;
import com.klikli_dev.modonomicon.client.gui.book.entry.EntryConnectionRenderer;
import com.klikli_dev.modonomicon.client.gui.book.entry.EntryDisplayState;
import com.klikli_dev.modonomicon.events.ModonomiconEvents;
import com.klikli_dev.modonomicon.platform.ClientServices;
import com.mojang.blaze3d.systems.RenderSystem;
import org.lwjgl.glfw.GLFW;

import java.util.ArrayList;
import java.util.Optional;
import net.minecraft.class_10799;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_5684;
import net.minecraft.class_9848;


public class BookCategoryNodeScreen implements BookCategoryScreen {

    public static final int ENTRY_GRID_SCALE = 30;
    public static final int ENTRY_GAP = 2;

    public static final int ENTRY_HEIGHT = 26;
    public static final int ENTRY_WIDTH = 26;

    private final BookParentNodeScreen bookParentScreen;
    private final BookCategory category;
    private final EntryConnectionRenderer connectionRenderer;
    private float scrollX = 0;
    private float scrollY = 0;
    private boolean isScrolling;
    private float targetZoom;
    private float currentZoom;

    public BookCategoryNodeScreen(BookParentNodeScreen bookOverviewScreen, BookCategory category) {
        this.bookParentScreen = bookOverviewScreen;
        this.category = category;

        this.connectionRenderer = new EntryConnectionRenderer(category.getEntryTextures());

        this.targetZoom = 0.7f;
        this.currentZoom = this.targetZoom;
    }

    @Override
    public BookCategory getCategory() {
        return this.category;
    }

    public float getXOffset() {
        return ((this.bookParentScreen.getInnerWidth() / 2f) * (1 / this.currentZoom)) - this.scrollX / 2;
    }

    public float getYOffset() {
        return ((this.bookParentScreen.getInnerHeight() / 2f) * (1 / this.currentZoom)) - this.scrollY / 2;
    }

    public void render(class_332 guiGraphics, int pMouseX, int pMouseY, float pPartialTick) {
        if (ClientServices.CLIENT_CONFIG.enableSmoothZoom()) {
            float diff = this.targetZoom - this.currentZoom;
            this.currentZoom = this.currentZoom + Math.min(pPartialTick * (2 / 3f), 1) * diff;
        } else
            this.currentZoom = this.targetZoom;

        //GL Scissors to the inner frame area so entries do not stick out
        int innerX = this.bookParentScreen.getInnerX();
        int innerY = this.bookParentScreen.getInnerY();
        int innerWidth = this.bookParentScreen.getInnerWidth();
        int innerHeight = this.bookParentScreen.getInnerHeight();
        //the -1 are magic numbers to avoid an overflow of 1px.
        guiGraphics.method_44379(innerX, innerY, innerX + innerWidth - 1, innerY + innerHeight - 1);
        this.renderEntries(guiGraphics, pMouseX, pMouseY);
        guiGraphics.method_44380();
    }

    public void zoom(double delta) {
        float step = 1.2f;
        if ((delta < 0 && this.targetZoom > 0.5) || (delta > 0 && this.targetZoom < 1))
            this.targetZoom *= delta > 0 ? step : 1 / step;
        if (this.targetZoom > 1f)
            this.targetZoom = 1f;
    }

    public boolean mouseDragged(double pMouseX, double pMouseY, int pButton, double pDragX, double pDragY) {
        //Based on advancementsscreen
        if (pButton != 0) {
            this.isScrolling = false;
            return false;
        } else {
            if (!this.isScrolling) {
                this.isScrolling = true;
            } else {
                this.scroll(pDragX * 1.5, pDragY * 1.5);
            }
            return true;
        }
    }

    public boolean mouseClicked(double pMouseX, double pMouseY, int pButton) {

        float xOffset = this.getXOffset();
        float yOffset = this.getYOffset();
        for (var entry : this.category.getEntries().values()) {
            var displayStyle = this.getEntryDisplayState(entry);

            if (this.isEntryHovered(entry, xOffset, yOffset, (int) pMouseX, (int) pMouseY)) {

                var event = new EntryClickedEvent(this.category.getBook().getId(), entry.getId(), pMouseX, pMouseY, pButton, displayStyle);
                //if event is canceled -> click was handled and we do not open the entry.
                if (ModonomiconEvents.client().entryClicked(event)) {
                    return true;
                }

                //only if the entry is unlocked we open it
                if (displayStyle == EntryDisplayState.UNLOCKED) {
                    BookGuiManager.get().openEntry(entry, BookAddress.defaultFor(entry));
                    return true;
                }
            }
        }

        return false;
    }

    public void renderBackground(class_332 guiGraphics) {
        //based on the frame's total width and its thickness, calculate where the inner area starts
        int innerX = this.bookParentScreen.getInnerX();
        int innerY = this.bookParentScreen.getInnerY();

        //then calculate the corresponding inner area width/height so we don't draw out of the frame
        int innerWidth = this.bookParentScreen.getInnerWidth();
        int innerHeight = this.bookParentScreen.getInnerHeight();

        //we do not use our static max_scroll here because it makes some issues, so we use the tex instead.
        int backgroundWidth = this.category.getBackgroundWidth();
        int backgroundHeight = this.category.getBackgroundHeight();
        final int MAX_SCROLL = Math.max(backgroundWidth, backgroundHeight);
        float backgroundTextureZoomMultiplier = this.category.getBackgroundTextureZoomMultiplier();


        // Adjust scale calculations to take into account actual texture size
        float xScale = MAX_SCROLL * 2.0f / ((float) MAX_SCROLL + this.bookParentScreen.getFrameThicknessW() - this.bookParentScreen.getFrameWidth());
        float yScale = MAX_SCROLL * 2.0f / ((float) MAX_SCROLL + this.bookParentScreen.getFrameThicknessH() - this.bookParentScreen.getFrameHeight());
        float scale = Math.max(xScale, yScale);
        float xOffset = xScale == scale ? 0 : (MAX_SCROLL - (innerWidth + MAX_SCROLL * 2.0f / scale)) / 2;
        float yOffset = yScale == scale ? 0 : (MAX_SCROLL - (innerHeight + MAX_SCROLL * 2.0f / scale)) / 2;

        //note we cannot translate -z here because even -1 immediately pushes us behind the scene -> not visible
        if (!this.category.getBackgroundParallaxLayers().isEmpty()) {
            this.category.getBackgroundParallaxLayers().forEach(layer -> {
                this.renderBackgroundParallaxLayer(guiGraphics, layer, innerX, innerY, innerWidth, innerHeight, this.scrollX, this.scrollY, scale, xOffset, yOffset, this.currentZoom, backgroundWidth, backgroundHeight, backgroundTextureZoomMultiplier);
            });
        } else {
            //for some reason on this one blit overload tex width and height are switched. It does correctly call the followup though, so we have to go along
            //force offset to int here to reduce difference to entry rendering which is pos based and thus int precision only
            guiGraphics.method_25290(class_10799.field_56883, this.category.getBackground(), innerX, innerY,
                    (this.scrollX + MAX_SCROLL) / scale + xOffset,
                    (this.scrollY + MAX_SCROLL) / scale + yOffset,
                    innerWidth, innerHeight, (int) (backgroundHeight * backgroundTextureZoomMultiplier), (int) (backgroundWidth * backgroundTextureZoomMultiplier));

        }
    }

    public void renderBackgroundParallaxLayer(class_332 guiGraphics, BookCategoryBackgroundParallaxLayer layer, int x, int y, int width, int height, float scrollX, float scrollY, float parallax, float xOffset, float yOffset, float zoom, int backgroundWidth, int backgroundHeight, float backgroundTextureZoomMultiplier) {
        float parallax1 = parallax / layer.getSpeed();

        if (layer.getVanishZoom() == -1 || layer.getVanishZoom() > zoom) {
            //for some reason on this one blit overload tex width and height are switched. It does correctly call the followup though, so we have to go along
            guiGraphics.method_25290(class_10799.field_56883, layer.getBackground(), x, y,
                    (scrollX + this.getCategory().getMaxScrollX()) / parallax1 + xOffset,
                    (scrollY + this.getCategory().getMaxScrollY()) / parallax1 + yOffset,
                    width, height, (int) (backgroundHeight * backgroundTextureZoomMultiplier), (int) (backgroundWidth * backgroundTextureZoomMultiplier));
        }

    }

    private void renderEntries(class_332 guiGraphics, int mouseX, int mouseY) {

        //calculate the render offset
        float xOffset = this.getXOffset();
        float yOffset = this.getYOffset();

        guiGraphics.method_51448().pushMatrix();
        guiGraphics.method_51448().scale(this.currentZoom, this.currentZoom);

        for (var entry : this.category.getEntries().values()) {
            var displayState = this.getEntryDisplayState(entry);
            var isHovered = this.isEntryHovered(entry, xOffset, yOffset, mouseX, mouseY);

            if (displayState == EntryDisplayState.HIDDEN)
                continue;

            int texX = entry.getEntryBackgroundVIndex() * ENTRY_HEIGHT;
            int texY = entry.getEntryBackgroundUIndex() * ENTRY_WIDTH;

            guiGraphics.method_51448().pushMatrix();
            //we translate instead of applying the offset to the entry x/y to avoid jittering when moving
            guiGraphics.method_51448().translate(xOffset, yOffset);


            //we apply a z offset to push the entries before the connection arrows
            //TODO here we had a translate +10z

            //As of 1.20 this is not necessary, in fact it causes the entry to render behind the bg
            //guiGraphics.pose().translate(0, 0, -10); //push the whole entry behind the frame


            int color = class_9848.method_61318(1.0f, 1.0F, 1.0F, 1.0F);
            if (displayState == EntryDisplayState.LOCKED) {
                //Draw locked entries greyed out
                //TODO shader color needs to be handed as last parameter to blit
                color = class_9848.method_61318(1f, 0.2F, 0.2F, 0.2F);
            } else if (isHovered) {
                //Draw hovered entries slightly greyed out
                color = class_9848.method_61318(1f, 0.8F, 0.8F, 0.8F);
            }
            //render entry background
            guiGraphics.method_25291(class_10799.field_56883, this.category.getEntryTextures(), entry.getX() * ENTRY_GRID_SCALE + ENTRY_GAP, entry.getY() * ENTRY_GRID_SCALE + ENTRY_GAP, texX, texY, ENTRY_WIDTH, ENTRY_HEIGHT, 256, 256, color);

            guiGraphics.method_51448().pushMatrix();

            //render icon
            entry.getIcon().render(guiGraphics, entry.getX() * ENTRY_GRID_SCALE + ENTRY_GAP + 5, entry.getY() * ENTRY_GRID_SCALE + ENTRY_GAP + 5);

            guiGraphics.method_51448().popMatrix();

            //render unread icon
            if (displayState == EntryDisplayState.UNLOCKED && !BookUnlockStateManager.get().isReadFor(class_310.method_1551().field_1724, entry)) {
                final int U = 350;
                final int V = 19;
                final int width = 11;
                final int height = 11;

                guiGraphics.method_51448().pushMatrix();
                //TODO here we had translate +11 z
                //if focused we go to the right of our normal button (instead of down, like mc buttons do)
                BookContentRenderer.drawFromContentTexture(class_10799.field_56883, guiGraphics, this.bookParentScreen.getBook(),
                        entry.getX() * ENTRY_GRID_SCALE + ENTRY_GAP + 16 + 2,
                        entry.getY() * ENTRY_GRID_SCALE + ENTRY_GAP - 2, U + (isHovered ? width : 0), V, width, height);
                guiGraphics.method_51448().popMatrix();
            }

            guiGraphics.method_51448().popMatrix();

            this.renderConnections(guiGraphics, entry, xOffset, yOffset);
        }
        guiGraphics.method_51448().popMatrix();
    }

    public void renderEntryTooltips(class_332 guiGraphics, int mouseX, int mouseY, float partialTicks) {
        //calculate the render offset
        float xOffset = this.getXOffset();
        float yOffset = this.getYOffset();

        for (var entry : this.category.getEntries().values()) {
            var displayState = this.getEntryDisplayState(entry);
            if (displayState == EntryDisplayState.HIDDEN)
                continue;

            this.renderTooltip(guiGraphics, entry, displayState, xOffset, yOffset, mouseX, mouseY);
        }
    }

    private boolean isEntryHovered(BookEntry entry, float xOffset, float yOffset, int mouseX, int mouseY) {
        int x = (int) ((entry.getX() * ENTRY_GRID_SCALE + xOffset + 2) * this.currentZoom);
        int y = (int) ((entry.getY() * ENTRY_GRID_SCALE + yOffset + 2) * this.currentZoom);
        int innerX = this.bookParentScreen.getInnerX();
        int innerY = this.bookParentScreen.getInnerY();
        int innerWidth = this.bookParentScreen.getInnerWidth();
        int innerHeight = this.bookParentScreen.getInnerHeight();
        return mouseX >= x && mouseX <= x + (ENTRY_WIDTH * this.currentZoom)
                && mouseY >= y && mouseY <= y + (ENTRY_HEIGHT * this.currentZoom)
                && mouseX >= innerX && mouseX <= innerX + innerWidth
                && mouseY >= innerY && mouseY <= innerY + innerHeight;
    }

    private void renderTooltip(class_332 guiGraphics, BookEntry entry, EntryDisplayState displayState, float xOffset, float yOffset, int mouseX, int mouseY) {
        //hovered?
        if (this.isEntryHovered(entry, xOffset, yOffset, mouseX, mouseY)) {

            var tooltip = new ArrayList<class_5684>();
            var tooltipComponents = new ArrayList<class_2561>();

            if (displayState == EntryDisplayState.LOCKED) {
                tooltip.addAll(
                        entry.getCondition().getTooltip(class_310.method_1551().field_1724, BookConditionEntryContext.of(this.bookParentScreen.getBook(), entry)).stream().map(class_2561::method_30937).map(class_5684::method_32662).toList());
                tooltipComponents.addAll(
                        entry.getCondition().getTooltip(class_310.method_1551().field_1724, BookConditionEntryContext.of(this.bookParentScreen.getBook(), entry)));
            } else if (displayState == EntryDisplayState.UNLOCKED) {
                //add name in bold
                tooltip.add(class_5684.method_32662(class_2561.method_43471(entry.getName()).method_27692(class_124.field_1067).method_30937()));
                tooltipComponents.add(class_2561.method_43471(entry.getName()).method_27692(class_124.field_1067));
                //add description
                if (!entry.getDescription().isEmpty()) {
                    tooltip.add(class_5684.method_32662(class_2561.method_43471(entry.getDescription()).method_30937()));
                    tooltipComponents.add(class_2561.method_43471(entry.getDescription()));
                }
            }

            //draw description
            guiGraphics.method_51434(class_310.method_1551().field_1772, tooltipComponents, mouseX, mouseY);
//            guiGraphics.renderTooltip(Minecraft.getInstance().font, tooltip, mouseX, mouseY, DefaultTooltipPositioner.INSTANCE, null);
        }
    }

    private void renderConnections(class_332 guiGraphics, BookEntry entry, float xOffset, float yOffset) {
        //our arrows are aliased and need blending

        for (var parent : entry.getParents()) {
            var parentDisplayState = this.getEntryDisplayState(parent.getEntry());
            if (parentDisplayState == EntryDisplayState.HIDDEN)
                continue;

            guiGraphics.method_51448().pushMatrix();
            guiGraphics.method_51448().translate(xOffset, yOffset);
            this.connectionRenderer.render(guiGraphics, entry, parent);
            guiGraphics.method_51448().popMatrix();
        }
    }

    private void scroll(double pDragX, double pDragY) {
        this.scrollX = (float) class_3532.method_15350(this.scrollX - pDragX, -this.getCategory().getMaxScrollX(), this.getCategory().getMaxScrollX());
        this.scrollY = (float) class_3532.method_15350(this.scrollY - pDragY, -this.getCategory().getMaxScrollY(), this.getCategory().getMaxScrollY());
    }

    /**
     * Sets the visual elements of the state, but not the open entry (handled by Gui Manager)
     */
    @Override
    public void loadState(CategoryVisualState state) {
        this.scrollX = state.scrollX;
        this.scrollY = state.scrollY;
        this.targetZoom = state.targetZoom;
        this.currentZoom = state.targetZoom;
    }

    @Override
    public void saveState(CategoryVisualState state) {
        state.scrollX = this.scrollX;
        state.scrollY = this.scrollY;
        state.targetZoom = this.targetZoom;
    }

    @Override
    public void onDisplay() {

    }

    @Override
    public void onClose() {
        //do not call super - gui manager should handle gui removal
        //Note: As this is not a vanilla screen child we don't even have a super :)
    }

    public boolean keyPressed(int key, int scanCode, int modifiers) {
        if (key == GLFW.GLFW_KEY_ESCAPE) {
            BookGuiManager.get().closeScreenStack(this);
            return true;
        }
        return false;
    }
}
