/*
 * SPDX-FileCopyrightText: 2022 Authors of Patchouli
 * SPDX-FileCopyrightText: 2022 klikli-dev
 *
 * SPDX-License-Identifier: MIT
 */

package com.klikli_dev.modonomicon.multiblock;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import com.klikli_dev.modonomicon.Modonomicon;
import com.klikli_dev.modonomicon.api.multiblock.StateMatcher;
import com.klikli_dev.modonomicon.data.LoaderRegistry;
import com.klikli_dev.modonomicon.multiblock.matcher.Matchers;
import com.mojang.datafixers.util.Pair;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2382;
import net.minecraft.class_2470;
import net.minecraft.class_2540;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_7225;
import net.minecraft.class_9129;

public class SparseMultiblock extends AbstractMultiblock {

    public static final class_2960 TYPE = Modonomicon.loc("sparse");

    private final Map<class_2338, StateMatcher> stateMatchers;
    private final class_2382 size;

    public SparseMultiblock(Map<class_2338, StateMatcher> stateMatchers) {
        Preconditions.checkArgument(!stateMatchers.isEmpty(), "No data given to sparse multiblock!");

        //this differs from dense multiblock, where we keep a copy of the originally loaded data for serialization, but it should be fine as we have less entries.
        this.stateMatchers = ImmutableMap.copyOf(stateMatchers);
        this.size = this.calculateSize();
    }

    public static SparseMultiblock fromJson(JsonObject json, class_7225.class_7874 provider) {
        var jsonMapping = class_3518.method_15296(json, "mapping");
        var mapping = mappingFromJson(jsonMapping, provider);

        //        "pattern": {
//            "N": [
//            [-1, 0, -2], [0, 0, -2], [1, 0, -2]
//          ],
//            "S": [
//            [-1, 0, 2], [0, 0, 2], [1, 0, 2]
//          ],
//            "W": [
//            [-2, 0, -1], [-2, 0, 0], [-2, 0, 1]
//          ],
//            "E": [
//            [2, 0, -1], [2, 0, 0], [2, 0, 1]
//          ]
//        }

        var jsonPattern = class_3518.method_15296(json, "pattern");

        Map<class_2338, StateMatcher> stateMatchers = new Object2ObjectOpenHashMap<>();
        for (Entry<String, JsonElement> entry : jsonPattern.entrySet()) {
            if (entry.getKey().length() != 1)
                throw new JsonSyntaxException("Pattern key needs to be only 1 character");

            var matcher = mapping.get(entry.getKey().charAt(0));

            var jsonPositions = class_3518.method_15252(entry.getValue(), entry.getKey());
            for (JsonElement jsonPosition : jsonPositions) {
                var jsonPos = class_3518.method_15252(jsonPosition, entry.getKey());
                if (jsonPos.size() != 3) {
                    throw new JsonSyntaxException("Each matcher position needs to be an array of 3 integers");
                }
                stateMatchers.put(
                        new class_2338(jsonPos.get(0).getAsInt(), jsonPos.get(1).getAsInt(), jsonPos.get(2).getAsInt()),
                        matcher);
            }
        }

        var multiblock = new SparseMultiblock(stateMatchers);

        return additionalPropertiesFromJson(multiblock, json);
    }

    public static SparseMultiblock fromNetwork(class_9129 buffer) {
        var symmetrical = buffer.readBoolean();
        var offX = buffer.method_10816();
        var offY = buffer.method_10816();
        var offZ = buffer.method_10816();
        var viewOffX = buffer.method_10816();
        var viewOffY = buffer.method_10816();
        var viewOffZ = buffer.method_10816();

        var size = buffer.method_10816();
        var stateMatchers = new Object2ObjectOpenHashMap<class_2338, StateMatcher>();
        for (int i = 0; i < size; i++) {
            var pos = buffer.method_10811();
            var type = buffer.method_10810();
            var matcher = LoaderRegistry.getStateMatcherNetworkLoader(type).fromNetwork(buffer);
            stateMatchers.put(pos, matcher);
        }

        var multiblock = new SparseMultiblock(stateMatchers);
        multiblock.setSymmetrical(symmetrical);
        multiblock.setOffset(offX, offY, offZ);
        multiblock.setViewOffset(viewOffX, viewOffY, viewOffZ);
        return multiblock;
    }

    private class_2382 calculateSize() {
        int minX = this.stateMatchers.keySet().stream().mapToInt(class_2338::method_10263).min().getAsInt();
        int maxX = this.stateMatchers.keySet().stream().mapToInt(class_2338::method_10263).max().getAsInt();
        int minY = this.stateMatchers.keySet().stream().mapToInt(class_2338::method_10264).min().getAsInt();
        int maxY = this.stateMatchers.keySet().stream().mapToInt(class_2338::method_10264).max().getAsInt();
        int minZ = this.stateMatchers.keySet().stream().mapToInt(class_2338::method_10260).min().getAsInt();
        int maxZ = this.stateMatchers.keySet().stream().mapToInt(class_2338::method_10260).max().getAsInt();
        return new class_2382(maxX - minX + 1, maxY - minY + 1, maxZ - minZ + 1);
    }

    @Override
    public class_2382 getSize() {
        return this.size;
    }

    @Override
    public class_2960 getType() {
        return TYPE;
    }

    @Override
    public Pair<class_2338, Collection<SimulateResult>> simulate(class_1937 world, class_2338 anchor, class_2470 rotation, boolean forView, boolean disableOffset) {
        class_2338 disp = forView
                ? new class_2338(-this.viewOffX, -this.viewOffY + 1, -this.viewOffZ).method_10070(rotation)
                : new class_2338(-this.offX, -this.offY, -this.offZ).method_10070(rotation);
        if (disableOffset)
            disp = class_2338.field_10980;

        // the local origin of this multiblock, in world coordinates
        class_2338 origin = anchor.method_10081(disp);
        List<SimulateResult> ret = new ArrayList<>();
        for (var e : this.stateMatchers.entrySet()) {
            class_2338 currDisp = e.getKey().method_10070(rotation);
            class_2338 actionPos = origin.method_10081(currDisp);
            ret.add(new SimulateResultImpl(actionPos, e.getValue(), null));
        }
        return Pair.of(origin, ret);
    }

    @Override
    public boolean test(class_1937 world, class_2338 start, int x, int y, int z, class_2470 rotation) {
        this.setLevel(world);
        class_2338 checkPos = start.method_10081(new class_2338(x, y, z).method_10070(rotation));
        class_2680 state = world.method_8320(checkPos).method_26186(AbstractMultiblock.fixHorizontal(rotation));
        StateMatcher matcher = this.stateMatchers.getOrDefault(new class_2338(x, y, z), Matchers.ANY);
        return matcher.getStatePredicate().test(world, checkPos, state);
    }

    @Override
    public void toNetwork(class_2540 buffer) {
        buffer.method_52964(this.symmetrical);
        buffer.method_10804(this.offX);
        buffer.method_10804(this.offY);
        buffer.method_10804(this.offZ);
        buffer.method_10804(this.viewOffX);
        buffer.method_10804(this.viewOffY);
        buffer.method_10804(this.viewOffZ);

        buffer.method_10804(this.stateMatchers.size());
        for (Entry<class_2338, StateMatcher> entry : this.stateMatchers.entrySet()) {
            buffer.method_10807(entry.getKey());
            buffer.method_10812(entry.getValue().getType());
            entry.getValue().toNetwork(buffer);
        }
    }

    @Override
    public class_2680 method_8320(class_2338 pos) {
        long ticks = this.level != null ? this.level.method_8510() : 0L;
        return this.stateMatchers.getOrDefault(pos, Matchers.AIR).getDisplayedState(ticks);
    }
}
